package App::Manoc::Controller::Rack;
#ABSTRACT: Rack controller
use Moose;

our $VERSION = '2.99.3'; ##TRIAL VERSION

use namespace::autoclean;
BEGIN { extends 'Catalyst::Controller'; }
with
    'App::Manoc::ControllerRole::CommonCRUD' => { -excludes => 'delete_object' },
    'App::Manoc::ControllerRole::JSONView'   => { -excludes => 'get_json_object', };

use App::Manoc::Form::Rack;

__PACKAGE__->config(
    # define PathPart
    action => {
        setup => {
            PathPart => 'rack',
        }
    },
    class               => 'ManocDB::Rack',
    form_class          => 'App::Manoc::Form::Rack',
    view_object_perm    => undef,
    json_columns        => [ 'id', 'name' ],
    object_list_options => {
        prefetch => 'building',
        join     => 'building',
        order_by => 'me.name',
    }
);


before 'create' => sub {
    my ( $self, $c ) = @_;

    my $building_id = $c->req->query_parameters->{'building'};
    $building_id and $c->log->debug("new rack in $building_id");
    $c->stash( form_defaults => { building => $building_id } );
};


sub delete_object {
    my ( $self, $c ) = @_;

    my $rack = $c->stash->{'object'};

    if ( $rack->hwassets->count ) {
        $c->flash( error_msg => "Rack contains hardware assets. Cannot be deleted." );
        return;
    }
    if ( $rack->devices->count ) {
        $c->flash( error_msg => "Rack has associated devices. Cannot be deleted." );
        return;
    }

    return $rack->delete;
}


sub get_json_object {
    my ( $self, $c, $rack ) = @_;

    my $r = $self->prepare_json_object( $c, $rack );
    $r->{building} = {
        id   => $rack->building->id,
        name => $rack->building->name,
    };
    $r->{devices} = [ map +{ id => $_->id, name => $_->name }, $rack->devices ];
    return $r;
}

__PACKAGE__->meta->make_immutable;

1;

# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

App::Manoc::Controller::Rack - Rack controller

=head1 VERSION

version 2.99.3

=head1 ACTIONS

=head2 create

=head1 METHODS

=head2 delete_object

=head2 get_json_object

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
