#! /usr/bin/env false

use v6.d;

use Config;
use Grammar::TodoTxt;
use IO::Path::XDG;
use Terminal::ANSIColor;

use App::GTD::GrammarActions;
use App::GTD::Types;

unit module App::GTD;

subset PrioStr of Str is export where /^ <[a..z A..Z]> $/;

my Config $config .= new.read: {
	colors => {
		enabled => True,
		contexts => 'green',
		projects => 'red',
		leftover-labels => 'cyan',
		labels => {
			gtd => 'yellow',
		},
	},
	data => {
		file => xdg-data-home.add('todo.txt').absolute,
	},
};

multi sub cmp (
	GtdState:D $lhs,
	GtdState:D $rhs,
) is export {
	my %order =
		GtdState::Inbox   => 0,
		GtdState::Next    => 1,
		GtdState::Someday => 2,
	;

	return Order::Same if $lhs eq $rhs;
	return Order::More if %order{$lhs} > %order{$rhs};

	Order::Less;
}

sub gtd-config-file (
	--> IO::Path
) is export {
	xdg-config-home.add('gtd/config.toml')
}

sub gtd-config-load () is export
{
	my $file = gtd-config-file;

	return unless $file.e;

	$config.=read($file.absolute);

	return;
}

multi sub gtd-config (
	--> Config
) is export {
	$config
}

multi sub gtd-config (
	Str:D $key,
	--> Any
) is export {
	$config.get($key)
}

multi sub gtd-parse-todo (
	Str:D $line,
	--> Iterable
) is export {
	samewith([$line]);
}

multi sub gtd-parse-todo (
	@lines,
	--> Iterable
) is export {
	@lines
		.map({
			Grammar::TodoTxt
				.parse($_, actions => App::GTD::GrammarActions)
				.made
		})
		.grep(*.defined)
}

sub gtd-records-read (
	--> Iterable
) is export {
	my $file = gtd-config('data.file').IO;

	return unless $file.e;

	gtd-parse-todo($file.lines)
}

sub gtd-records-write (
	@records,
) is export {
	gtd-config('data.file')
		.IO
		.spurt(@records.grep(*.defined).join("\n") ~ "\n")
}

sub gtd-records-string (
	@records,
	--> Str
) is export {
	my $index-length = @records.sort(*.id).tail.id.Str.chars;

	@records
		.map({
			"[%{$index-length}d] %s".sprintf(
				$_.id,
				colorize-output(~$_)
			)
		})
		.join("\n")
}

sub colorize-output (
	Str:D $input,
	Bool:D :$stderr = False,
) is export {
	return $input unless $config<colors><enabled>;

	$input
		.words
		.map({
			given $_ {
				when $_.starts-with('+') { colored($_, gtd-config('colors.projects')) }
				when $_.starts-with('@') { colored($_, gtd-config('colors.contexts')) }
				when $_.contains(':') {
					my $key = $_.split(':').first;

					colored(
						$_,
						gtd-config("colors.labels.$key")
							// gtd-config('colors.leftover-labels'),
					)
				}
				default { $_ }
			}
		})
		.join(' ')
		;
}

=begin pod

=NAME    App::GTD
=AUTHOR  Patrick Spek <p.spek@tyil.work>
=VERSION 0.2.0

=head1 Synopsis

=item1 gtd [--help]
=item1 gtd add <words...>
=item1 gtd inbox
=item1 gtd inbox <id>
=item1 gtd next
=item1 gtd next <id> [--prio[=<a..z>]] [words...]
=item1 gtd someday
=item1 gtd someday <id>
=item1 gtd done <id>

=head1 Description

A command line application to support the L<Getting Things
Done|https://gettingthingsdone.com/> methodology, written using the Raku
programming language.

=head1 Examples

=head2 Adding items to your inbox

Items are not required to be in quotes, but your shell may require you to quote
some characters. Items that you C<add> will be put into the I<inbox>.

    $ gtd add "something to do"
    $ gtd add another thing to do

=head2 Reading your inbox items

Your inbox contains all the items that were added, and not yet sorted out. You
can view your inbox with the similarly named command, C<inbox>.

    $ gtd inbox
    [2] 2020-02-01 another thing to do gtd:inbox
    [1] 2020-02-01 something to do gtd:inbox

Your inbox is sorted based on creation date, and then alphabetically. The
numbers in front of each item is the I<id> of the item, and corresponds to the
line number they appear on in your C<todo.txt>.

=head2 Sorting out your inbox

Items in your inbox can be moved to your I<next items> or your I<someday
items>. Additionally, you can mark them as I<done>.

=head3 Moving an item to your next items

To move items to your next items, you can use the C<next> subcommand.

    $ gtd next 1
    $ gtd next 2 Attend FOSDEM

Now, if you check your inbox, it will be empty.

    $ gtd inbox
    Your inbox is empty!

These items have been moved to the next items, which you can list using C<next>
without any arguments.

    $ gtd next
    [2] 2020-02-01 Attend FOSDEM gtd:next
    [1] 2020-02-01 something to do gtd:next

You might notice that the item with id U<2> got updated to reflect the words
after the id of the next command. This allows you to expand on the quick notes
you might make for your inbox.

=head3 Moving an item to your someday items

This is similar to moving items to your next box, but instead you use
C<someday> as the subcommand. Note that items don't have to be in your inbox in
order to move them around.

    $ gtd someday 2
    $ gtd next
    [1] 2020-02-01 something to do gtd:next
    $ gtd someday
    [2] 2020-02-01 Attend FOSDEM gtd:someday

=head2 Marking an item as done

Once you've completed an item, you can mark it as done. The C<done> subcommand
manages this.

    $ gtd done 1

This will mark the item done, and make it not appear in the lists that
C<App::GTD> shows.

=head2 Cleaning up

The store for C<App::GTD> is a plaintext file, formatted according to the
L<todo.txt|http://todotxt.org/> specification. In this spec, the items you mark
as done are being kept around, and as such, so does C<App::GTD>. To clean up
this file, the C<clean> subcommand can be used.

    $ gtd clean

This not only removed items marked as done, but also sorts all items again to
be correct again. B<Note that this means that the IDs of the items are
reordered.> Additionally, all lines that do not conform to the todo.txt
specification are also B<removed>.

=head1 Configuration

You can use the C<config> subcommand to dump your current configuration as it
is used by the application. This also shows which default values are used. For
more information about the available configuration options, refer to
C<App::GTD::Config>.

=head1 License

This module is distributed under the terms of the GNU AGPL, version 3.0.

=end pod

# vim: ft=perl6 noet
