package SBOM::CycloneDX::Vulnerability::Analysis;

use 5.010001;
use strict;
use warnings;
use utf8;

use SBOM::CycloneDX::Enum;
use SBOM::CycloneDX::Timestamp;

use Types::Standard qw(Str Enum InstanceOf);
use Types::TypeTiny qw(ArrayLike);

use Moo;
use namespace::autoclean;

extends 'SBOM::CycloneDX::Base';

has state => (is => 'rw', isa => Enum [SBOM::CycloneDX::Enum->values('IMPACT_ANALYSIS_STATE')]);

has justification => (is => 'rw', isa => Enum [SBOM::CycloneDX::Enum->values('IMPACT_ANALYSIS_JUSTIFICATION')]);

has response => (
    is      => 'rw',
    isa     => ArrayLike [Enum [qw(can_not_fix will_not_fix update rollback workaround_available)]],
    default => sub { SBOM::CycloneDX::List->new }
);

has detail => (is => 'rw', isa => Str);

has first_issued => (
    is     => 'rw',
    isa    => InstanceOf ['SBOM::CycloneDX::Timestamp'],
    coerce => sub { ref($_[0]) ? $_[0] : SBOM::CycloneDX::Timestamp->new($_[0]) }
);

has last_updated => (
    is     => 'rw',
    isa    => InstanceOf ['SBOM::CycloneDX::Timestamp'],
    coerce => sub { ref($_[0]) ? $_[0] : SBOM::CycloneDX::Timestamp->new($_[0]) }
);

sub TO_JSON {

    my $self = shift;

    my $json = {};

    $json->{state}         = $self->state         if $self->state;
    $json->{justification} = $self->justification if $self->justification;
    $json->{response}      = $self->response      if $self->response;
    $json->{detail}        = $self->detail        if $self->detail;
    $json->{firstIssued}   = $self->first_issued  if $self->first_issued;
    $json->{lastUpdated}   = $self->last_updated  if $self->last_updated;

    return $json;

}

1;

=encoding utf-8

=head1 NAME

SBOM::CycloneDX::Vulnerability::Analysis - Impact Analysis

=head1 SYNOPSIS

    SBOM::CycloneDX::Vulnerability::Analysis->new();


=head1 DESCRIPTION

L<SBOM::CycloneDX::Vulnerability::Analysis> An assessment of the impact and
exploitability of the vulnerability.

=head2 METHODS

L<SBOM::CycloneDX::Vulnerability::Analysis> inherits all methods from L<SBOM::CycloneDX::Base>
and implements the following new ones.

=over

=item SBOM::CycloneDX::Vulnerability::Analysis->new( %PARAMS )

Properties:

=over

=item * C<detail>, Detailed description of the impact including methods used
during assessment. If a vulnerability is not exploitable, this field should
include specific details on why the component or service is not impacted by
this vulnerability.

=item * C<first_issued>, The date and time (timestamp) when the analysis was
first issued.

=item * C<justification>, The rationale of why the impact analysis state was asserted.

=item * C<last_updated>, The date and time (timestamp) when the analysis was
last updated.

=item * C<response>, A response to the vulnerability by the manufacturer,
supplier, or project responsible for the affected component or service.
More than one response is allowed. Responses are strongly encouraged for
vulnerabilities where the analysis state is exploitable.

=item * C<state>, Declares the current state of an occurrence of a vulnerability,
after automated or manual analysis.

=back

=item $analysis->detail

=item $analysis->first_issued

=item $analysis->justification

=item $analysis->last_updated

=item $analysis->response

=item $analysis->state

=back


=head1 SUPPORT

=head2 Bugs / Feature Requests

Please report any bugs or feature requests through the issue tracker
at L<https://github.com/giterlizzi/perl-SBOM-CycloneDX/issues>.
You will be notified automatically of any progress on your issue.

=head2 Source Code

This is open source software.  The code repository is available for
public review and contribution under the terms of the license.

L<https://github.com/giterlizzi/perl-SBOM-CycloneDX>

    git clone https://github.com/giterlizzi/perl-SBOM-CycloneDX.git


=head1 AUTHOR

=over 4

=item * Giuseppe Di Terlizzi <gdt@cpan.org>

=back


=head1 LICENSE AND COPYRIGHT

This software is copyright (c) 2025-2026 by Giuseppe Di Terlizzi.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
