# MIDI.pm is a simplified front end to the 
#
# load everything we'll need, and get ready to rock.

package POE::Framework::MIDI;

use strict;
use POE;
use POE::Framework::MIDI::POEConductor;
use POE::Framework::MIDI::POEMusician;
use POE::Framework::MIDI::Bar;
use POE::Framework::MIDI::Note;
use POE::Framework::MIDI::Rest;
use Class::MethodMaker   new_with_init => [ qw /new/] , get_set =>  qw/honk/;
our $VERSION = 0.06;

sub init
{
	my ($self, %args) = @_;
	$self->{args} = \%args;
	die "No filename set in data hashref" unless $args{data}->{filename};
	die "No bar count set in data hashref" unless $args{data}->{bars};
}

sub musicians
{
	my ($self,$new_musicians) = @_;
	die "Non array-ref passed to musicians()" if (ref($new_musicians)  and ref($new_musicians) ne 'ARRAY');
	$new_musicians ?
		$self->{args}->{musicians} = $new_musicians
		: return $self->{args}->{musicians};
}

sub run
{
	my $self = shift;
	POE::Framework::MIDI::POEConductor->spawn({
		debug     => $self->{args}->{data}->{debug},
		verbose   => $self->{args}->{data}->{verbose},
		bars      => $self->{args}->{data}->{bars},
		filename  => $self->{args}->{data}->{filename},
	musicians => $self->{args}->{musicians}  });

# $poe_kernel is exported by POE
$poe_kernel->run;

}

1;


__END__

=head1 NAME

POE::Framework::MIDI - A simplified front end to the POE::Framework::MIDI environment

=head1 ABSTRACT

=head1 DESCRIPTION

This module allows you to specify the data required by the POE::Framework::MIDI environment
without having to dive into the guts of the framework. 

=head1 SYNOPSIS

	# assuming you've already 'use'd the module MyMusician up here somewhere

	my @musicians = (
		{
			name => 'Frank',
			package => 'MyMusician',
			patch => 50,
			channel => 1,
		},
		{
			name => 'Ian',
			package => 'MyMusician',
			patch => 75,
			channel => 2,
		}
	);
	
	my %data = (
		bars => 15,
		debug => 1,
		verbose => 1,
		filename => 'miditest.mid',
	);
	
	my $midi = POE::Framework::MIDI->new( musicians => \@musicians, data => \%data );
	$midi->run;
	
	# by this point, you should have generated a midi file (miditest.mid) and since
	# debug is turned on, a debug.perl file to show you the autogenerated perl code
	# fed to MIDI::Simple


=head1 SEE ALSO

L<POE>
L<POE::Framework::MIDI::*>
L<MIDI::Simple>


=head1 AUTHOR

Primary: Steve McNabb E<lt>steve@justsomeguy.comE<gt>

CPAN ID: SMCNABB

Secondary: Gene Boggs E<lt>cpan@ology.netE<gt>

CPAN ID: GENE

=head1 COPYRIGHT AND LICENCE

Copyright (c) 2004 Steve McNabb. All rights reserved.
This program is free software; you can redistribute it and/or modify 
it under the same terms as Perl itself.

The full text of the license can be found in the LICENSE file 
included with this module.

=cut
