package Dancer::Core::Role::SessionFactory;
{
    $Dancer::Core::Role::SessionFactory::VERSION = '2.0000_01';
}

#ABSTRACT: Role for session factories


use strict;
use warnings;
use Carp 'croak';
use Dancer::Core::Session;
use Dancer::Core::Types;
use Digest::SHA1 'sha1_hex';
use List::Util 'shuffle';

use Moo::Role;
with 'Dancer::Core::Role::Engine';

sub supported_hooks {
    qw/
      engine.session.before_retrieve
      engine.session.after_retrieve

      engine.session.before_create
      engine.session.after_create

      engine.session.before_destroy
      engine.session.after_destroy

      engine.session.before_flush
      engine.session.after_flush
      /;
}

sub _build_type {
    'SessionFactory';
}    # XXX vs 'Session'?  Unused, so I can't tell -- xdg


has cookie_name => (
    is      => 'ro',
    isa     => Str,
    default => sub {'dancer.session'},
);


has cookie_domain => (
    is        => 'ro',
    isa       => Str,
    predicate => 1,
);


has cookie_path => (
    is      => 'ro',
    isa     => Str,
    default => sub {"/"},
);


has cookie_duration => (
    is        => 'ro',
    isa       => Num,
    predicate => 1,
);


has is_secure => (
    is      => 'rw',
    isa     => Bool,
    default => sub {0},
);


has is_http_only => (
    is      => 'rw',
    isa     => Bool,
    default => sub {1},
);


sub create {
    my ($self) = @_;

    my %args = (id => $self->generate_id,);

    $args{expires} = $self->cookie_duration
      if $self->has_cookie_duration;

    my $session = Dancer::Core::Session->new(%args);

    $self->execute_hook('engine.session.before_create', $session);

    eval { $self->_flush($session->id, $session->data) };
    croak "Unable to create a new session: $@"
      if $@;

    $self->execute_hook('engine.session.after_create', $session);
    return $session;
}


{
    my $COUNTER = 0;

    sub generate_id {
        my ($self) = @_;

        my $seed = rand(1_000_000_000)    # a random number
          . __FILE__                      # the absolute path as a secret key
          . $COUNTER++    # impossible to have two consecutive dups
          . time()        # impossible to have dups between seconds
          . $$            # the process ID as another private constant
          . "$self"       # the instance's memory address for more entropy
          . join('',
            shuffle('a' .. 'z', 'A' .. 'Z', 0 .. 9)
          )    # a shuffled list of 62 chars, another random component
          ;

        return sha1_hex($seed);
    }
}


requires '_retrieve';

sub retrieve {
    my ($self, %params) = @_;
    my $id = $params{id};

    $self->execute_hook('engine.session.before_retrieve', $id);

    my $data = eval { $self->_retrieve($id) };
    croak "Unable to retrieve session with id '$id'"
      if $@;

    my %args = (id => $id,);

    $args{data} = $data
      if $data and ref $data eq 'HASH';

    $args{expires} = $self->cookie_duration
      if $self->has_cookie_duration;

    my $session = Dancer::Core::Session->new(%args);

    $self->execute_hook('engine.session.after_retrieve', $session);
    return $session;
}


requires '_destroy';

sub destroy {
    my ($self, %params) = @_;
    my $id = $params{id};
    $self->execute_hook('engine.session.before_destroy', $id);

    eval { $self->_destroy($id) };
    croak "Unable to destroy session with id '$id': $@"
      if $@;

    $self->execute_hook('engine.session.after_destroy', $id);
    return $id;
}


requires '_flush';

sub flush {
    my ($self, %params) = @_;
    my $session = $params{session};
    $self->execute_hook('engine.session.before_flush', $session);

    eval { $self->_flush($session->id, $session->data) };
    croak "Unable to flush session: $@"
      if $@;

    $self->execute_hook('engine.session.after_flush', $session);
    return $session->id;
}


sub cookie {
    my ($self, %params) = @_;
    my $session = $params{session};
    croak "cookie() requires a valid 'session' parameter"
      unless ref($session) && $session->isa("Dancer::Core::Session");

    my %cookie = (
        value     => $session->id,
        name      => $self->cookie_name,
        path      => $self->cookie_path,
        secure    => $self->is_secure,
        http_only => $self->is_http_only,
    );

    $cookie{domain} = $self->cookie_domain
      if $self->has_cookie_domain;

    if (my $expires = $session->expires) {
        $cookie{expires} = $expires;
    }

    return Dancer::Core::Cookie->new(%cookie);
}


requires '_sessions';

sub sessions {
    my ($self) = @_;
    my $sessions = $self->_sessions;

    croak "_sessions() should return an array ref"
      if ref($sessions) ne ref([]);

    return $sessions;
}

1;

__END__

=pod

=head1 NAME

Dancer::Core::Role::SessionFactory - Role for session factories

=head1 VERSION

version 2.0000_01

=head1 DESCRIPTION

Any class that consumes this role will be able to store, create, retrieve and
destroy session objects.

=head1 ATTRIBUTES

=head2 cookie_name

The name of the cookie to create for storing the session key

Defaults to C<dancer.session>

=head2 cookie_domain

The domain of the cookie to create for storing the session key.
Defaults to the empty string and is unused as a result.

=head2 cookie_path

The path of the cookie to create for storing the session key.
Defaults to "/".

=head2 cookie_duration

Default duration before session cookie expiration.  If set, the
L<Dancer::Core::Session> C<expires> attribute will be set to the current time
plus this duration.

=head2 is_secure

Boolean flag to tell if the session cookie is secure or not.

Default is false.

=head2 is_http_only

Boolean flag to tell if the session cookie is http only.

Default is true.

=head1 INTERFACE

Following is the interface provided by this role. When specified the required
methods to implement are described.

=head2 create

Create a brand new session object and store it. Returns the newly created
session object.

Triggers an exception if the session is unable to be created.

    my $session = MySessionFactory->create();

This method does not need to be implemented in the class.

=head2 generate_id

Returns a randomly-generated, guaranteed-unique string.  (It is not
guaranteed cryptographically secure, but it's still reasonably
strong for general use.)  This method is used internally by create()
to set the session ID.

This method does not need to be implemented in the class unless an
alternative method for session ID generation is desired.

=head2 retrieve

Return the session object corresponding to the session ID given. If none is
found, triggers an exception.

    my $session = MySessionFactory->retrieve(id => $id);

The method C<_retrieve> must be implemented.  It must take C<$id> as a single
argument and must return a hash reference of session data.

=head2 destroy

Purges the session object that matches the ID given. Returns the ID of the
destroyed session if succeeded, triggers an exception otherwise.

    MySessionFactory->destroy(id => $id);

The C<_destroy> method must be implemented. It must take C<$id> as a single
argumenet and destroy the underlying data.

=head2 flush

Make sure the session object is stored in the factory's backend. This method is
called to notify the backend about the change in the session object.

An exception is triggered if the session is unable to be updated in the backend.

    MySessionFactory->flush(session => $session);

The C<_flush> method must be implemented.  It must take two arguments: the C<$id>
and a hash reference of session data.

=head2 cookie

Coerce a session object into a L<Dancer::Core::Cookie> object.

    MySessionFactory->cookie(session => $session);

=head2 sessions

Return a list of all session IDs stored in the backend.
Useful to create cleaning scripts, in conjunction with session's creation time.

The C<_sessions> method must be implemented.  It must return an array reference
of session IDs (or an empty array reference).

=head1 AUTHOR

Dancer Core Developers

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2012 by Alexis Sukrieh.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
