/*                                                                
**  Copyright (C) 1996,2007,2010,2019  Smithsonian Astrophysical Observatory 
*/                                                                

/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 3 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License along */
/*  with this program; if not, write to the Free Software Foundation, Inc., */
/*  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.             */
/*                                                                          */

/*
 *
 * find.c -- find files via the path environment variable
 * (and related routines)
 *
 */


#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <time.h>

#include "find.h"

/*
 *
 * 	private routines 
 *
 */

void cxcObsoleteOldParFile (const char *name, char *ext, char *path);


static char cxcAccessName[MAXBUFSIZE + 2];

static int
cxcamparse (char *mode)
{
  int xmode = 0;

  xmode |= (strpbrk (mode, "r") != NULL ? R_OK : 0);
  xmode |= (strpbrk (mode, "w") != NULL ? W_OK : 0);
  xmode |= (strpbrk (mode, "x") != NULL ? X_OK : 0);
  xmode |= (strpbrk (mode, "f") != NULL ? F_OK : 0);

  return xmode;
}

static char *
cxcfindpath (const char *name, char *mode, char *path)
{
  char pathbuff[4096] = "";
  char namebuff[MAXBUFSIZE] = "";
  char backmode[10] = "";
  char *here = NULL;
  char *found = NULL;
  int mark = 0;
  int skip = strpbrk (mode, ">") != NULL;
  int pick = strpbrk (mode, "<") != NULL;

  if (skip && pick)
    return NULL;

  if ((path == NULL) || (name[0] == '.' && name[1] == '/') || name[0] == '/')
    return cxcAccess (name, mode);

  strcpy (pathbuff, path);
  path = pathbuff;

  if ((here = strpbrk (pathbuff, " :;")))
    {
      mark = *here;
      *here++ = '\0';
    }
  while (path)
    {
      if (path[0] == '$')
	{
	  if ((path = getenv (&path[1])))
	    if ((found = cxcfindpath (name, mode, path)))
	      return found;
	}
      else
	{
	  if (!skip)
	    {
	      if (!strcmp (".", path))
		path[0] = '\0';

	      strcpy (namebuff, path);
	      if (path[0] && path[strlen (path) - 1] != '/')
		strcat (namebuff, "/");
	      strcat (namebuff, name);

	      if ((found = cxcAccess (namebuff, mode)))
		return found;
	    }
	}

      if (mark == ';')
	{
	  if (skip)
	    {
	      skip = 0;
	      /* Knock down the skip mode to select all
	       * directories in path after the first ";"
	       */
	      strcpy (backmode, mode);
	      backmode[strlen (backmode) - 1] = '\0';
	      mode = backmode;
	    }
	  if (pick)
	    return NULL;
	}

      path = here;
      if (here && (here = strpbrk (here, " :;")))
	{
	  mark = *here;
	  *here++ = '\0';
	}
    }

  return NULL;
}


/*
 *
 * 	public routines 
 *
 */

/*
 *
 * cxcResolvePath -- resolve the path to remove . and .. entries
 *
 */
char *
cxcResolvePath (char *ibuf, char *obuf)
{
  char path[MAXBUFSIZE + 2] = "";
  char *part[MAXBUFSIZE];
  char *cxc_tbuf_p = NULL;
  int i, j;
  int len;
  int npart = 0;

  /* if we have no path separators, we really don't have much to do! */
  if (strchr (ibuf, '/') == NULL)
    {
      strcpy (obuf, ibuf);
      return (obuf);
    }

  /* if its just "/" or "/.", its easy */
  if (!strcmp (ibuf, "/") || !strcmp (ibuf, "/."))
    {
      strcpy (obuf, "/");
      return (obuf);
    }

  /* if we have a relative path to deal with, get current directory */
  if ((*ibuf == '.') || ((strchr (ibuf, '/') != NULL) && (*ibuf != '/')))
    {
      getcwd (path, MAXBUFSIZE);
    }
  else
    {
      *path = '\0';
    }

  /* construct the total string we have to deal with */
  len = strlen (path) + strlen (ibuf) + 1;
  cxc_tbuf_p = (char *) malloc (len + 1);
  if (*path)
    {
      strcpy (cxc_tbuf_p, path);
      strcat (cxc_tbuf_p, "/");
      strcat (cxc_tbuf_p, ibuf);
    }
  else
    {
      strcpy (cxc_tbuf_p, ibuf);
    }

  /* construct the parts array from this string, removing / characters
     and null-terminating each part */
  for (i = 0; i < len; i++)
    {
      if (cxc_tbuf_p[i] == '/')
	{
	  cxc_tbuf_p[i] = '\0';
	  part[npart] = &cxc_tbuf_p[i + 1];
	  npart++;
	}
    }

  /* loop through the parts array and resolve the  . and .. entries */
  for (i = 0; i < npart; i++)
    {
      /* for ".", just remove it */
      if (!strcmp (part[i], "."))
	{
	  part[i] = NULL;
	}
      /* for "..", also remove the previous part -- if possible */
      else if (!strcmp (part[i], ".."))
	{
	  part[i] = NULL;
	  for (j = i - 1; j >= 0; j--)
	    {
	      if (part[j])
		{
		  part[j] = NULL;
		  break;
		}
	    }
	}
    }

  /* construct a new string from the remaining parts */
  *obuf = '\0';
  for (i = 0; i < npart; i++)
    {
      if (part[i] != NULL)
	{
	  strcat (obuf, "/");
	  strcat (obuf, part[i]);
	}
    }

  /* free up buffer space */
  free (cxc_tbuf_p);
  cxc_tbuf_p = NULL;

  /* return the string */
  return (obuf);
}

void
cxcExpandEnv (const char *name, char *fullname, int maxlen)
{
  char brace[2];
  char cxc_tbuf[MAXBUFSIZE];
  const char *mip = NULL;
  const char *ip = NULL;
  char *s = NULL;
  int len;
  int i, j;

  i = 0;
  fullname[i] = '\0';
  for (ip = name; *ip; ip++)
    {
      if (*ip != '$')
	{
	  fullname[i++] = *ip;
	  fullname[i] = '\0';
	}
      else
	{
	  mip = ip;
	  /* skip past '$' */
	  ip++;
	  /* skip past brace, if necessary */
	  if (*ip == '{')
	    {
	      brace[0] = '{';
	      ip++;
	    }
	  else if (*ip == '(')
	    {
	      brace[0] = '(';
	      ip++;
	    }
	  else
	    brace[0] = '\0';
	  /* get variable up to next white space */
	  for (*cxc_tbuf = '\0', j = 0;
	       (!isspace (*ip)) && (*ip != '"') && (*ip != '\'') && (*ip);
	       ip++)
	    {
	      /* look for trailing brace, if necessary */
	      if (*brace && *ip == (*brace == '(' ? ')' : '}'))
		{
		  ip++;
		  break;
		}
	      /* a "/" will end the environment variable as well */
	      if (*ip == '/')
		{
		  break;
		}
	      cxc_tbuf[j++] = *ip;
	      cxc_tbuf[j] = '\0';
	    }
	  /* back up so we can process the white space in the outer loop */
	  ip--;
	  if ((s = getenv (cxc_tbuf)) != NULL)
	    {
	      i += strlen (s);
	      if (i <= maxlen)
		strcat (fullname, s);
	    }
	  /* if we don't recognize this macro, put it back onto the string */
	  else
	    {
	      len = ip - mip + 1;
	      i += len;
	      if (i <= maxlen)
		strncat (fullname, mip, len);
	    }
	}
    }
}

char *
cxcAccess (const char *name, char *mode)
{
  struct stat info;
  char fullname[MAXBUFSIZE];

  cxcExpandEnv (name, fullname, MAXBUFSIZE);
  if (stat (fullname, &info) != 0)
    return NULL;

  if (mode)
    {
      int m = cxcamparse (mode);

#if 0
      if (getuid () == info.st_uid)
	{
	  if (m & R_OK && !(info.st_mode & S_IRUSR))
	    return NULL;
	  if (m & W_OK && !(info.st_mode & S_IWUSR))
	    return NULL;
	  if (m & X_OK && !(info.st_mode & S_IXUSR))
	    return NULL;
	}
      else if (getgid () == info.st_gid)
	{
	  if (m & R_OK && !(info.st_mode & S_IRGRP))
	    return NULL;
	  if (m & W_OK && !(info.st_mode & S_IWGRP))
	    return NULL;
	  if (m & X_OK && !(info.st_mode & S_IXGRP))
	    return NULL;
	}
      else
	{
	  if (m & R_OK && !(info.st_mode & S_IROTH))
	    return NULL;
	  if (m & W_OK && !(info.st_mode & S_IWOTH))
	    return NULL;
	  if (m & X_OK && !(info.st_mode & S_IXOTH))
	    return NULL;
	}
#endif

      if (access (fullname, m) != 0)
	return NULL;
    }



  cxcResolvePath (fullname, cxcAccessName);
  return (cxcAccessName);
}

char *
cxcFind (const char *name, char *mode, char *extn, char *path)
{
  char extnbuff[MAXBUFSIZE];
  char namebuff[MAXBUFSIZE];
  char *here = NULL;
  char *found = NULL;

  if (extn == NULL)
    return cxcfindpath (name, mode, path);

  strcpy (extnbuff, extn);
  extn = extnbuff;

  if ((here = strpbrk (extnbuff, " :;")))
    *here++ = '\0';

  while (extn)
    {
      if (extn[0] == '$')
	{
	  if ((extn = getenv (&extn[1])))
	    {
	      if ((found = cxcFind (name, mode, extn, path)))
		return found;
	    }
	}
      else
	{
	  char *e = strstr (name, extn);

	  strcpy (namebuff, name);
	  if ((e == NULL) || (e && *(e + strlen (extn))))
	    strcat (namebuff, extn);

	  cxcObsoleteOldParFile (name, extn, path);

	  if ((found = cxcfindpath (namebuff, mode, path)))
	    return found;

	}

      extn = here;
      if (here && (here = strpbrk (here, " :;")))
	*here++ = '\0';
    }

  return NULL;
}

void
cxcObsoleteOldParFile (const char *tname, char *ext, char *path)
{


  char name[MAXBUFSIZE];
  char goo[MAXBUFSIZE];



  char *foo;
  char *new;

  struct stat baseline;
  struct stat userfile;

  if (tname && ext)
    {
      int extlen = strlen (ext);
      int pos = strlen (tname);

      /* if tname already ends in ext, then don't append ext */
      if ((pos > extlen) && (!strcmp (&tname[pos - extlen], ext)))
	{
	  strncpy (name, tname, MAXBUFSIZE);
	  if (pos >= MAXBUFSIZE)
	    {
	      name[MAXBUFSIZE - 1] = '\0';
	    }
	}
      else
	{
	  sprintf (name, "%s%s", tname, ext);
	}
    }


  memset (goo, 0, MAXBUFSIZE);

  foo = cxcfindpath (name, "rw<", path);

  while (foo && strcmp (goo, foo))
    {
      strcpy (goo, foo);

      stat (foo, &userfile);

      new = cxcfindpath (name, "r>", path);
      if (new)
	{
	  stat (new, &baseline);

	  if (baseline.st_mtime > userfile.st_mtime)
	    {
	      time_t tod;
	      struct tm lt;
	      char forName[MAXBUFSIZE];
	      char newName[MAXBUFSIZE];
	      char preName[MAXBUFSIZE];
	      int status;
	      time (&tod);
	      lt = *localtime (&tod);

	      strftime (forName, MAXBUFSIZE, "%%s_%Y%m%d.%H:%M:%S%%s", &lt);

	      memset (preName, 0, MAXBUFSIZE);
	      strncpy (preName, goo, strlen (goo) - strlen (ext));

	      sprintf (newName, forName, preName, ext);


	      if ((status = rename (goo, newName)) == 0)
		{
		  fprintf (stderr,
			   "Warning:  found possibly out of date user parameter file, renamed to %s\n",
			   newName);
		}
	      else
		{
		  fprintf (stderr,
			   "Warning:  found possibly out of date user parameter file, tried to rename but failed: %s\n",
			   goo);
		}
	      /* break; */
	    }

	}

      foo = cxcfindpath (name, "rw<", path);
    }


}
