## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(echo=TRUE,results='hide')

## ----message=FALSE, warning=FALSE---------------------------------------------
library(sequenceR)
library(tuneR)

## ----results='markup'---------------------------------------------------------
# a 0.1-second A note
d <- 0.1 # duration of the sample in seconds
rate <- 44100 # sampling rate in values per second (this is a typical value and is used as a default throughout this package)
n <- round(rate*d) # number of values in the waveform
w <- sin(2*pi*440*seq(0,d,length.out=n)) # waveform (440Hz A note)
A <- soundSample(wave=w,rate=rate) 

## ----results='hold'-----------------------------------------------------------
# listen(A) # uncomment to listen to the sample
plot(A) # plot the sample

## ----results='hold'-----------------------------------------------------------
set.seed(123456)
noisyA <- soundSample(wave=w+0.1*rnorm(n))
# listen(noisyA) # uncomment to listen to the sample
plot(noisyA)

## ----results='hold'-----------------------------------------------------------
sun <- soundSample(sunspots,rate=44100/10) # sampling rate is lowered to get a low pitch
# listen(sun) # uncomment to listen to the sample
plot(sun)

## ----results='hold'-----------------------------------------------------------
w <- tuneR::readMP3('07027201.mp3')
tuneR::plot(w)

## ----results='hold'-----------------------------------------------------------
# Cast Wave to sound Sample 
sam <-as.soundSample(w)
# listen(sam) # uncomment to listen to the sample
plot(sam)

## ----results='markup'---------------------------------------------------------
A_seq <- sequence(A,time=c(0,0.5,1,1.5),
                  volume=c(1,0.4,0.4,0.4),
                  pan=c(0,-0.5,0.5,1))
tuneR::plot(A_seq)

## ----results='hold'-----------------------------------------------------------
sam_seq <- sequence(sam,time=c(0,1.2),pan=c(-1,1),letRing=FALSE)
tuneR::plot(sam_seq)
sam_seq <- sequence(sam,time=c(0,1.2),pan=c(-1,1),letRing=TRUE)
tuneR::plot(sam_seq)

## ----results='markup'---------------------------------------------------------
myMix <- mix(list(A_seq,sam_seq),volume=c(1,0.3))
# tuneR::play(myMix) # uncomment to play
# tuneR::writeWave(myMix,'myMix.wav') # uncomment to save to disk
tuneR::plot(myMix)

## -----------------------------------------------------------------------------
plot(WaggaWagga$Year,WaggaWagga$Precipitation,type='l',xlab='Year',ylab='precip. [mm]')
plot(WaggaWagga$Year,WaggaWagga$Temperature,type='l',xlab='Year',ylab='temp. [C]')

## -----------------------------------------------------------------------------
n <- NROW(WaggaWagga) # series size
dur <- 9 # duration in seconds
tim <- dur*seq(0,1,length.out=n) # regular time vector between 0 and dur
master <- rescale(WaggaWagga$Temperature,0.2,1) # master volume = temperature time series rescaled between 0.2 and 1

## -----------------------------------------------------------------------------
every4=(((1:n)-1))%%4==0 # T F F F T F F F etc.
accents <- rescale(as.numeric(every4),0.2,1) # 1 0.2 0.2 0.2 1 0.2 0.2 0.2 etc.  
hh <- sequence(hiHat,time=tim,volume=master*accents) # create hi-hat sequence
# tuneR::play(hh) # uncomment to play

## -----------------------------------------------------------------------------
mask=WaggaWagga$Precipitation<450 # time steps with low pp
k <- sequence(kick,time=tim[mask],volume=master[mask]) # play a kick at those time steps
# tuneR::play(k) # uncomment to play

## -----------------------------------------------------------------------------
mask=WaggaWagga$Precipitation>800 # time steps with high pp
s <- sequence(snare,time=tim[mask],volume=master[mask]) # play a snare at those time steps
# tuneR::play(s) # uncomment to play

## -----------------------------------------------------------------------------
final <- mix(list(hh,k,s),volume=c(0.5,0.75,1))
writeWave(final,'WaggaWagga.wav') # write to disc
# tuneR::play(final) # uncomment to play

## ----warning=FALSE,eval=FALSE-------------------------------------------------
#  library(tidyr);library(ggplot2);library(gganimate)
#  # Modify the shape of the WaggaWagga dataset to facilitate plotting
#  DF <- pivot_longer(WaggaWagga,-Year) # function from tidyr
#  # Plot precipitation and temperature time series using ggplot
#  g <- ggplot(DF,aes(x=Year,y=value))
#  g <- g + geom_line(aes(color=name),linewidth=1)+geom_point(size=4)
#  g <- g + scale_color_manual(values = c('blue','red'),guide=FALSE)
#  g <- g + facet_wrap(vars(name),ncol=1,scales='free_y')+theme_bw()
#  g <- g + geom_hline(data=data.frame(y=450,name='Precipitation'),aes(yintercept=y))
#  g <- g + geom_hline(data=data.frame(y=800,name='Precipitation'),aes(yintercept=y))
#  # Make it look nicer
#  g <- g+theme_bw()+theme(axis.title=element_text(size=18),
#            axis.text=element_text(size=14),
#            strip.text=element_text(size=18))
#  # Create an animated plot
#  g <- g + transition_reveal(Year)
#  # 'Render' the animated plot into a .mp4 movie
#  fps=n/dur # number of frames divided by duration
#  animate(g,nframes=NROW(WaggaWagga),fps=fps,width=1280,height=720,
#          renderer = av_renderer('WaggaWaggaGroove.mp4',audio='WaggaWagga.wav'))

## -----------------------------------------------------------------------------
drums <- instrument(samples=list(kick,snare,hiHat),notes=c('boom','tat','cheet'))
w=play.instrument(drums,notes=rep(c('boom','cheet','tat','cheet'),4),
                  time=0.25*(0:15),volume=(1:16)/16,
                  pan=rep(c(0,-0.5,0.5,-0.5),4))
# tuneR::play(w) # uncomment to play
tuneR::plot(w)

## -----------------------------------------------------------------------------
# Create a synthesizer instrument and play it
synth <- getSynth(c('E2','B2','E3','G3','A3')) 
w=play.instrument(synth,time=(0:(length(synth)-1))*0.5,fadeout=rep(Inf,length(synth)))
# tuneR::play(w) # uncomment to play
tuneR::plot(w)

## -----------------------------------------------------------------------------
dry=oscillator(freq=110,duration=1) # a low A note
# Define an ADSR envelope (https://en.wikipedia.org/wiki/Envelope_(music)#ADSR)
env <- envelope(t=c(0,0.05,0.2,0.8,1),v=c(0,1,0.6,0.2,0))
plot(env)

## -----------------------------------------------------------------------------
# Apply effects: envelope, saturation and delay
w1=applyEnvelope(dry,env)
w2=applyDisto(w1,type='clip',level=0.5)
wet=applyDelay(w2,delayTime=0.5,echoes=1/(1:5))
# listen(wet) # uncomment to play
plot(wet)

## ----message=FALSE, warning=FALSE---------------------------------------------
library(dplyr)
wet=dry %>% applyEnvelope(env) %>% applyDisto(type='clip',level=0.5) %>%
  applyDelay(delayTime=0.5,echoes=1/(1:5))
# listen(wet) # uncomment to play
plot(wet)

## ----eval=FALSE, message=FALSE, warning=FALSE---------------------------------
#  w=sonifyStripes(videoFile='stripes.mp4')

