# Copyright (C) 1997-2000  Adrian Trapletti
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the Free
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#
# ffnet R user interface
#


ffnet <- function (obj, ...) { UseMethod("ffnet") }
hessian <- function (obj, ...) { UseMethod("hessian") }

as.double.mat <- function (x)
{
  x <- as.matrix(x)
  storage.mode(x) <- "double"
  return (x)
}

hidtyNr <- function (str)
{
  return (pmatch(str,c("SIG","TAN"))-1)
}

outtyNr <- function (str)
{
  return (pmatch(str,c("LIN","SOFT"))-1)
}

errfNr <- function (str)
{
  return (pmatch(str,c("SSE","ENTROPY","MAD","GSSE"))-1)
}

regNr <- function (str)
{
  return (pmatch(str,c("NOR","WDR","LASSO"))-1)
}

optimNr <- function (str)
{
  return (pmatch(str,c("GRDDSC","STPDSC","FRPRMN","DFPMIN","SANN","NRSANN"))-1)
}

traceNr <- function (str)
{
  return (pmatch(str,c("NO","PRINT","PLOT"))-1)
}

predict.ffnet <- function (nn, x, type)
{
  if (!inherits(nn, "ffnet")) stop ("method is only for ffnet objects")
  if (missing(x))
    x <- nn$x
  x <- as.double.mat(x)
  if (dim(x)[2] != nn$nin) stop ("number of columns of x does not match number of inputs")
  if (any(is.na(x))) stop ("missing values in x")
  if (dim(x)[1] <= 0) stop ("no observations in x")
  hidty <- hidtyNr(nn$hidtype)
  outty <- outtyNr(nn$outtype)
  y <- matrix (0, nr=dim(x)[1], nc=nn$nout)
  storage.mode(y) <- "double"
  res <- .C ("R_ffnet_predict", x, as.integer(dim(x)), y=y, as.integer(dim(y)),
             as.integer(nn$nhid), as.integer(nn$nout), as.integer(hidty),
             as.integer(outty), as.integer(nn$shortcut), as.vector(nn$wts,mode="double"),
             as.integer(length(nn$wts)), PACKAGE="ffnet")
  if (missing(type))
  {
    if (nn$outtype == "SOFT")
      ynet <- as.factor.cl.code (res$y, nn$lev)
    else
      ynet <- res$y
  }
  else if (type == "CLASS")
    ynet <- as.factor.cl.code (res$y, nn$lev)
  else if (type == "RAW")
    ynet <- res$y
  else
    stop ("not valid type")
  return (drop(ynet))
}

predict.ffnet.ts <- function (nn, y, x, genuine = FALSE)
{
  if (!inherits(nn, "ffnet.ts")) stop ("method is only for ffnet.ts objects")
  mx <- missing(x)
  if (missing(y))
  {
    if (!mx) stop ("y is missing and x not")
    ynet <- predict.ffnet (nn, nn$x)
    ynet <- ts(ynet,end=end(nn$y),frequency=frequency(nn$y))
  }
  else
  {
    if (!is.ts(y)) stop ("y is not a ts object")
    if (!mx)
    {
      if (!is.ts(x)) stop ("x is not a ts object")
      if ((start(x) != start(y)) || (end(x) != end(y)) || (frequency(x) != frequency(y)))
        stop ("ts attributes of y and x do not match")
      if (is.null(nn$lagx)) stop ("nn has been trained without x and x is given")
    }
    else
      if (!is.null(nn$lagx)) stop ("nn has been trained with x and x is missing")
    m <- NCOL(y)
    n <- NROW(y)
    if (nn$lag > n) stop ("too many lags")
    z <- embed (y, nn$lag)
    if (!mx)
    {
      if (nn$lagx > NROW(x)) stop ("too many lags")
      zx <- embed (x, nn$lagx)
      mn <- min(nrow(z),nrow(zx))
      xem <- cbind(z[(nrow(z)-mn+1):nrow(z),],zx[(nrow(zx)-mn+1):nrow(zx),])
      st <- start(y)[1]+(max(nn$lag,nn$lagx)+nn$steps-1)/frequency(y)
    }
    else
    {
      xem <- as.matrix(z)
      st <- start(y)[1]+(nn$lag+nn$steps-1)/frequency(y)
    }
    ynet <- predict.ffnet (nn, xem)
    if (genuine)
      ynet <- ts(ynet,start=st,frequency=frequency(y))
    else
      ynet <- ts(ynet,start=st,end=end(y),frequency=frequency(y))
  }
  return (ynet)
}

ffnet.default <- function (x, y, wts, nhid = 0, hidtype = "SIG", outtype = "LIN",
                           shortcut = TRUE, fwts, hessian = FALSE, errfunc,
                           type, regularizer = "NOR", regc = 1.0, optimizer = "DFPMIN",
                           batch = TRUE, trace = "PRINT", itmax = 1000, tol = 1.0e-6,
                           epoch = 10, itepoch = 10, eta = 0.1, alpha = 0.7, temp = 100.0,
                           ittemp = 10, hold = FALSE, col = "black", xlim = range(0,itmax),
                           ylim = range(nout*nobs*0.001,fo), ptype = "l", ylab = "Error Function",
                           xlab = "Iterations", main =  paste(optimizer,"Optimization"))
{
  
  plot.optim <- function (itn, fn)
  {
    if (itn == 0)
    {
      fo <<- fn
      ito <<- itn
    }
    else if (ito == 0)
    {
      xp <- c(ito,itn)
      yp <- c(fo,fn)
      if (hold)
        lines (xp, yp, type=ptype, col=col)
      else
        plot (xp, yp, xlim=xlim, ylim=ylim, type=ptype,
              ylab=ylab, xlab=xlab, log="y", col=col, main=main)
      fo <<- fn
      ito <<- itn
    }
    else
    {
      xp <- c(ito,itn)
      yp <- c(fo,fn)
      points (xp, yp, type=ptype, col=col)
      fo <<- fn
      ito <<- itn
    }
  }
  
  ito <- NA
  fo <- NA
  
  call <- match.call ()
  nn <- NULL
  x <- as.double.mat(x)
  y <- as.double.mat(y)
  if (missing(errfunc))
  {
    if (NCOL(y) > 1) errfunc <- "GSSE"
    else errfunc <- "SSE"
  }
  if ((errfunc == "ENTROPY") & missing(outtype))
    outtype <- "SOFT"
  if ((outtype == "SOFT") & missing(errfunc))
    errfunc <- "ENTROPY"
  if (((errfunc == "ENTROPY") & (outtype != "SOFT")) |
      ((errfunc != "ENTROPY") & (outtype == "SOFT")))
    stop ("entropy needs softmax")
  nin <- dim(x)[2]
  nout <- dim(y)[2]
  nobs <- dim(y)[1]
  if ((nout == 1) & (outtype == "SOFT")) stop ("only one softmax output")
  if (any(is.na(x))) stop ("missing values in x")
  if (any(is.na(y))) stop ("missing values in y")
  if (dim(x)[1] != dim(y)[1]) stop ("number of rows of x and y must match")
  if (dim(x)[1] <= 0) stop ("no observations in x and y")
  if (nhid < 0) stop ("number of hidden units is negative")
  hidty <- hidtyNr(hidtype)
  outty <- outtyNr(outtype)
  errf <- errfNr(errfunc)
  reg <- regNr(regularizer)
  optim <- optimNr(optimizer)
  trc <- traceNr(trace)
  if (is.na(hidty)) stop ("not valid hidden activation function")
  if (is.na(outty)) stop ("not valid output activation function")
  if (is.na(errf)) stop ("not valid error function")
  if (is.na(reg)) stop ("not valid regularizer")
  if (is.na(optim)) stop ("not valid optimizer")
  if (is.na(trc)) stop ("not valid trace")
  if (((!batch) & (optimizer == "SANN")) | ((!batch) & (optimizer == "NRSANN")))
    stop ("simulated annealing only in batch mode")
  if (errfunc == "ENTROPY")
    if ((any(y < 0)) | (any(y > 1)) | (any(apply(y,1,sum) != 1)))
      stop ("y is not a valid classification code")
  if (shortcut) nw <- (nin+1)*nhid+(nhid+1)*nout+nin*nout
  else nw <- (nin+1)*nhid+(nhid+1)*nout
  if (!missing(wts))
  {
    if (nw != length(wts)) stop ("weight vector has not the correct length")
  }
  else wts <- runif(nw, -1, 1)
  wts <- as.vector(wts,mode="double")
  if (missing(fwts)) fwts <- integer()
  else
  {  
    if (any((fwts < 1) | (nw < fwts))) stop ("not valid fixed weights index vector")
    if (length(setdiff(1:nw,fwts))<1) stop ("no free parameters to estimate")
  }
  fwts <- as.integer(fwts)
  if (hessian)
  {
    hess <- numeric(nw*nw)
    dim(hess) <- c(nw,nw)
  }
  else
    hess <- numeric()
  hess <- as.matrix(hess)
  res <- .C ("R_ffnet_train", x, as.integer(dim(x)), y=y, as.integer(dim(y)), as.integer(nhid),
             as.integer(hidty), as.integer(outty), as.integer(shortcut), as.integer(errf),
             as.integer(reg), as.double(regc), as.integer(optim), as.integer(batch),
             as.integer(trc), as.integer(itmax), as.double(tol), as.integer(epoch),
             as.integer(itepoch), as.double(eta), as.double(alpha), as.double(temp),
             as.integer(ittemp), wts=wts, as.integer(nw), fwts,
             as.integer(length(fwts)), errv=0.0, hess=hess, as.integer(dim(hess)), plot.optim,
             PACKAGE="ffnet")
  if (missing(type))
  {
    if (outtype == "SOFT")
    {
      lev <- colnames (y)
      ynet <- as.factor.cl.code (res$y, lev)
      y <- as.factor.cl.code (y, lev)
      r <- (y == ynet)
    }
    else
    {
      lev <- NULL
      ynet <- res$y
      r <- y-res$y
    }
  }
  else if (type == "CLASS")
  {
    lev <- colnames (y)
    ynet <- as.factor.cl.code (res$y, lev)
    y <- as.factor.cl.code (y, lev)
    r <- (y == ynet)
  }
  else if (type == "RAW")
  {
    lev <- NULL
    ynet <- res$y
    r <- y-res$y
  }
  else
    stop ("not valid type")
  nn <- list(nin=nin,nhid=nhid,nout=nout,hidtype=hidtype,outtype=outtype,
             shortcut=shortcut,wts=res$wts,fwts=fwts,value=res$errv,
             fitted.values=drop(ynet),residuals=drop(r),x=drop(x),y=drop(y),hessian=res$hess,
             errfunc=errfunc,regularizer=regularizer,regc=regc,lev=lev)
  nn$call <- call
  class(nn) <- "ffnet"
  return (nn)
}

ffnet.formula <- function (formula, data = NULL, errfunc, outtype, ...)
{
  call <- match.call ()
  if (!inherits(formula, "formula")) stop ("method is only for formula objects")
  m <- match.call (expand = FALSE)
  if (is.matrix(eval(m$data,sys.frame(sys.parent())))) m$data <- as.data.frame (data)
  m$... <- m$errfunc <- m$outtype <- NULL
  m[[1]] <- as.name ("model.frame")
  m <- eval (m, sys.frame(sys.parent())) 
  Terms <- attr (m, "terms")
  attr (Terms, "intercept") <- 0
  x <- model.matrix (Terms, m)
  y <- model.extract (m, response)
  yor <- y
  if (is.factor(y))
  {
    lev <- levels(y)
    counts <- table(y)
    if (any (counts == 0))
    {
      warning (paste ("group(s)",paste(lev[counts == 0],collapse=" "), "are empty"))
      y <- factor (y, levels=lev[counts > 0])
    }
    y <- as.cl.code (y)
    if (missing(errfunc))
      errfunc <- "ENTROPY"
    if (missing(outtype))
      outtype <- "SOFT"
    nn <- ffnet.default (x, y, errfunc=errfunc, outtype=outtype, ...)
  }
  else
  {
    if (missing(errfunc))
    {
      if (NCOL(y) > 1) errfunc <- "GSSE"
      else errfunc <- "SSE"
    } 
    if (missing(outtype))
      outtype <- "LIN"
    nn <- ffnet.default (x, y, errfunc=errfunc, outtype=outtype, ...)
  }
  nn$x <- drop(x)
  nn$y <- drop(yor)
  nn$terms <- Terms
  nn$coefnames <- colnames (x)
  nn$call <- call
  class(nn) <- c("ffnet")
  return (nn)
}

ffnet.ts <- function (y, x, lag = 1, lagx = 1, steps = 1, errfunc, outtype, ...)
{
  call <- match.call ()
  if (!inherits(y, "ts")) stop ("method is only for ts objects")
  mx <- missing(x)
  if (!mx)
  {
    if (!is.ts(x)) stop ("x is not a ts object")
    if ((start(x) != start(y)) || (end(x) != end(y)) || (frequency(x) != frequency(y)))
      stop ("ts attributes of y and x do not match")
    if (lagx < 1) stop ("wrong lagx") 
  }
  if (lag < 1) stop ("wrong lag")
  if (steps < 1) stop ("wrong steps")
  if (missing(outtype))
    outtype <- "LIN"
  m <- NCOL(y)
  n <- NROW(y)
  if ((lag+steps) > n) stop ("too many lags or steps")
  zy <- embed (y, lag+steps)
  if (!mx)
  {
    if ((lagx+steps) > n) stop ("too many lags or steps") 
    k <- NCOL(x)
    zx <- embed (x, lagx+steps) 
    nzx <- NROW(zx) 
    nzy <- NROW(zy) 
    nz <- min(nzx,nzy) 
    yem <- as.matrix(zy[(nzy-nz+1):nzy,1:m]) 
    xem <- cbind(as.matrix(zy[(nzy-nz+1):nzy,(m*steps+1):(dim(zy)[2])]),
                 as.matrix(zx[(nzx-nz+1):nzx,(k*steps+1):(dim(zx)[2])])) 
  }
  else
  {
    yem <- as.matrix(zy[,1:m])
    xem <- as.matrix(zy[,(m*steps+1):(dim(zy)[2])])
  }
  if (missing(errfunc))
  {
    if (NCOL(yem) > 1) errfunc <- "GSSE"
    else errfunc <- "SSE"
  }
  if ((errfunc == "ENTROPY") | (outtype == "SOFT"))
    stop ("entropy and softmax are not supported for time series modelling")
  nn <- ffnet.default (xem, yem, errfunc=errfunc, outtype=outtype, ...)
  nn$call <- call
  nn$x <- drop(xem)
  nn$lag <- lag
  if (!mx) nn$lagx <- lagx
  nn$steps <- steps
  nn$fitted.values <- ts(nn$fitted.values,end=end(y),frequency=frequency(y))
  nn$residuals <- ts(nn$residuals,end=end(y),frequency=frequency(y))
  nn$y <- ts(drop(yem),end=end(y),frequency=frequency(y))
  class(nn) <- c("ffnet.ts","ffnet")
  return (nn)
}

hessian.ffnet <- function (nn, x, y, errfunc = "SSE", regularizer = "NOR", regc = 1.0) 
{
  if (!inherits(nn, "ffnet")) stop ("method is only for ffnet objects") 
  nw <- length(nn$wts)
  hess <- numeric(nw*nw)
  dim(hess) <- c(nw,nw)
  hess <- as.matrix(hess)
  if (missing(x))
    x <- nn$x
  if (missing(y))
    y <- nn$y
  x <- as.double.mat(x)
  y <- as.double.mat(y)
  nhid <- nn$nhid  
  hidty <- hidtyNr(nn$hidtype)
  outty <- outtyNr(nn$outtype)
  shortcut <- nn$shortcut
  errf <- errfNr(errfunc)
  reg <- regNr(regularizer)
  if (any(is.na(x))) stop ("missing values in x")
  if (any(is.na(y))) stop ("missing values in y")
  if (dim(x)[1] != dim(y)[1]) stop ("number of rows of x and y must match")
  if (dim(x)[2] != nn$nin) stop ("x has not the right number of variables")
  if (dim(y)[2] != nn$nout) stop ("y has not the right number of variables")
  if (dim(x)[1] <= 0) stop ("no observations in x and y")
  if (is.na(errf)) stop ("not valid error function")
  if (((errfunc == "ENTROPY") & (nn$outtype != "SOFT")) |
      ((errfunc != "ENTROPY") & (nn$outtype == "SOFT")))
    stop ("entropy needs softmax")
  if (errfunc == "ENTROPY")
    if ((any(y < 0)) | (any(y > 1)) | (any(apply(y,1,sum) != 1)))
      stop ("y is not valid classification code")
  if (is.na(reg)) stop ("not valid regularizer")
  wts <- as.vector(nn$wts,mode="double")
  fwts <- as.integer(nn$fwts)
  res <- .C ("R_ffnet_hess", x, as.integer(dim(x)), y, as.integer(dim(y)), as.integer(nhid), 
             as.integer(hidty), as.integer(outty), as.integer(shortcut), as.integer(errf),
             as.integer(reg), as.double(regc), wts, as.integer(nw), fwts,
             as.integer(length(fwts)), hess=hess, PACKAGE="ffnet")
  return (res$hess)
}

print.ffnet <- function (nn)
{
  cat("\nCall:", deparse(nn$call), "", sep = "\n")
  cat ("Architecture:\n")
  cat ("a ",nn$nin,"-",nn$nhid,"-",nn$nout," network", sep="")
  if (nn$shortcut) cat (" with shortcuts\n")
  else cat ("\n")
  if (length(nn$coefnames)) cat ("inputs:", nn$coefnames, "\noutput(s):",
                                 deparse(formula(nn)[[2]]), "\n")
  cat ("number of weights: ", length(nn$wts), "\n")
  cat ("hidden activation function: ")
  if (nn$hidtype == "TAN") cat ("tanh\n")
  else if (nn$hidtype == "SIG") cat ("logistic sigmoid\n")
  cat ("output activation function: ")
  if (nn$outtype == "LIN") cat ("linear\n")
  else if (nn$outtype == "SOFT") cat ("softmax\n")
  cat ("\n")
  invisible (nn)
}

fitted.ffnet <- function (nn)
{
  return (nn$fitted.values)
}

residuals.ffnet <- function (nn)
{
  return (nn$residuals)
}

plot.ffnet <- function (nn, colored = TRUE)
{
  op <- par()
  par (ask = TRUE)
  r <- residuals (nn)
  yh <- fitted (nn)
  if (is.factor(yh))
  {
    if (colored)
    {
      col <- codes(yh)
      pairs (nn$x, col=col)
      title (sub = "Fitted Classification")
      col <- as.integer(!r)+1
      pairs (nn$x, col=col)
      title (sub = "Classification Errors")
    }
    else
    {
      pch <- as.character(yh)
      pairs (nn$x, pch=pch)
      title (sub = "Fitted Classification")
      pch <- substring (as.character(!r), 1, 1)
      pairs (nn$x, pch=pch)
      title (sub = "Classification Errors")
    }
  }
  else
  {
    y <- yh+r
    plot (yh, y, xlab = "Fitted values", ylab = "Observations")
    plot (yh, abs(r), xlab = "Fitted values", ylab = "Absolute Residuals")
    abline (h = 0, lty = 3, col = "gray")
  }
  par (ask = op$ask)
  invisible (nn)
}

coef.ffnet <- function (nn)
{
  w <- nn$wts
  if (nn$nhid > 0)
  {
    wn1 <- rep(c(" b",paste("i",seq(length=nn$nin),sep="")),nn$nhid)
    wn1 <- c(wn1,rep(c(" b",paste("h",seq(length=nn$nhid),sep="")),nn$nout))
    wn2 <- rep(paste("h",seq(length=nn$nhid),sep=""),rep(nn$nin+1,nn$nhid))
    wn2 <- c(wn2,rep(paste("o",seq(length=nn$nout),sep=""),rep(nn$nhid+1,nn$nout)))
  }
  else
  {
    wn1 <- rep(" b",nn$nout)
    wn2 <- rep(paste("o",seq(length=nn$nout),sep=""),rep(nn$nhid+1,nn$nout))
  }
  if (nn$shortcut)
  {
    wn1 <- c(wn1,rep(paste("i",seq(length=nn$nin),sep=""),nn$nout))
    wn2 <- c(wn2,rep(paste("o",seq(length=nn$nout),sep=""),rep(nn$nin,nn$nout)))
  }
  names(w) <- apply(cbind(wn1,wn2),1,function(x) paste(x,collapse="->"))
  return (w)
}

summary.ffnet <- function (nn, invert.type = c("svd", "qr", "chol"), ...)
{
  
  r.squared.1 <- function (y, yh)
  {
    sst <- sum((y-mean(y))^2)
    ssr <- sum((yh-mean(yh))^2)
    return (ssr/sst)
  }
  
  r.squared.2 <- function (y, r)
  {
    sst <- sum((y-mean(y))^2)
    sse <- sum(r^2)
    return ((sst-sse)/sst)
  }
  
  det <- function (x)
  {
    prod(diag(qr(x)$qr))*(-1)^(NCOL(x)-1)
  }
  
  cov.mle <- function (x)
  {
    t(x)%*%x/NROW(x)
  }

  se.err <- function (H, invert.type)
  {
    if (invert.type == "qr")
    {
      se <- rep(NA,NROW(H))
      IH <- try(solve(H, ...))
      if (is.null(class(IH)))
      {
        di <- diag(IH)
        if (any(di<0))
          cat ("Warning: Hessian negative-semidefinite\n")
        se[di>=0] <- sqrt(di[di>=0])
      }
    }
    else if (invert.type == "svd")
    {
      se <- rep(NA,NROW(H))
      svdc <- svd(H)
      cond <- abs(max(svdc$d))/abs(min(svdc$d))  # abs() to convert -0 to 0, hence, cond > 0
      if (cond > Machine()$double.xmax)  # see also, NRC, pp. 61
      {
        cat ("Warning: Hessian singular\n")
      }
      else
      {
        if (1/cond < 1.0e4*Machine()$double.eps)
          cat ("Warning: Hessian ill-conditioned\n")
        sing <- matrix(0,NROW(H),NROW(H))
        diag(sing) <- 1/svdc$d
        di <- diag(svdc$v%*%sing%*%t(svdc$u))
        if (any(di<0))
          cat ("Warning: Hessian negative-semidefinite\n")
        se[di>=0] <- sqrt(di[di>=0])
      }
    }
    else if (invert.type == "chol")
    {
      se <- rep(NA,NROW(H))
      CH <- try(chol(H))
      if (is.null(class(CH)))
      {
        di <- diag(chol2inv(CH))
        se <- sqrt(di)
      }      
    }
    else
      stop ("invalid invert.type")
    return(se)
  }
  
  invert.type <- match.arg(invert.type)
  ptree <- substitute(nn)
  if (length(ptree) > 1) nm.nn <- deparse(ptree[[2]])
  else nm.nn <- deparse(ptree)
  ans <- NULL
  r <- residuals(nn)
  n <- NROW(r)
  yh <- fitted(nn)
  nrow <- length(nn$wts)
  idx1 <- setdiff(seq(1:nrow),nn$fwts)
  idx2 <- intersect(seq(1:nrow),nn$fwts)
  nrow <- length(idx1)
  prm <- coef(nn)[idx1]
  k <- length(prm)
  if (is.factor(yh))
  {
    ans$fitted <- yh
    if (is.factor(nn$y))
      ans$observed <- nn$y
    else
      ans$observed <- as.factor.cl.code (nn$y, nn$lev)
    ans$residuals <- nn$residuals
  }
  else
  {
    attr(r,"tsp") <- NULL
    class(r) <- NULL
    attr(yh,"tsp") <- NULL
    class(yh) <- NULL
    y <- yh+r
    if (NCOL(y) > 1)
    {
      ans$r.squared.1 <- r.squared.1 (y[,1], yh[,1])
      ans$r.squared.2 <- r.squared.2 (y[,1], r[,1])
      for (i in 2:NCOL(y))
      {
        ans$r.squared.1 <- c(ans$r.squared.1,r.squared.1 (y[,i], yh[,i]))
        ans$r.squared.2 <- c(ans$r.squared.2,r.squared.2 (y[,i], r[,i]))
      }
      names(ans$r.squared.1) <- names(ans$r.squared.2) <- colnames(r)
    }
    else
    {
      ans$r.squared.1 <- r.squared.1 (y, yh)
      ans$r.squared.2 <- r.squared.2 (y, r)
    }
    ans$residuals <- nn$residuals
    attr(ans$residuals,"tsp") <- NULL
    class(ans$residuals) <- NULL
  }
  if ((nn$errfunc == "SSE") || (nn$errfunc == "GSSE"))
  {
    ans$aic <- n*log(det(cov.mle(nn$residuals)))+2*k
    ans$sc <- n*log(det(cov.mle(nn$residuals)))+log(n)*k
  }
  else if (nn$errfunc == "MAD")
  {
    ans$aic <- n*log(det(cov.mle(nn$residuals)))+2*k
    ans$sc <- n*log(det(cov.mle(nn$residuals)))+log(n)*k
  }
  else if (nn$errfunc == "ENTROPY")
  {
    ans$aic <- n*nn$value+2*k
    ans$sc <- n*nn$value+log(n)*k
  }
  else stop ("invalid errfunc")
  if (dim(nn$hess)[1] <= 0)
  {
    if (is.factor(nn$y))
      obs <- as.cl.code (nn$y)
    else
      obs <- nn$y
    nn$hess <- hessian (nn, nn$x, obs, nn$errfunc, nn$regularizer, nn$regc)
    eval (parse (text = paste (nm.nn,"$hess <<- nn$hess", sep=""))) # side effect!
  }
  hess <- nn$hess[idx1,idx1]
  if (nn$errfunc == "SSE")
    scale <- 1/(2.0*sum(r^2)/length(r))
  else if (nn$errfunc == "ENTROPY")  # Preliminary
    scale <- 1.0
  else if (nn$errfunc == "MAD")  # Preliminary
    scale <- 1/(2.0*sum(abs(r))/length(r))
  else if (nn$errfunc == "GSSE")
    scale <- 1.0
  else
    stop ("nn has invalid errfunc")
  hess <- scale*hess/n
  se <- se.err(hess,invert.type)/sqrt(n)  
  nme <- names(prm)
  est <- as.vector(prm)
  tval <- est/se
  ans$coef <- cbind (est, se, tval, 2*(1-pnorm(abs(tval))))
  dimnames(ans$coef) <- list(nme, c(" Estimate"," Std. Error"," t value","Pr(>|t|)"))
  ans$fwts <- coef(nn)[idx2]
  ans$call <- nn$call
  ans$nin <- nn$nin
  ans$nhid <- nn$nhid
  ans$nout <- nn$nout
  ans$shortcut <- nn$shortcut
  ans$coefnames <- nn$coefnames
  if (length(ans$coefnames))
    ans$outnames <- deparse(formula(nn)[[2]])
  ans$nw <- length(nn$wts)
  ans$hidtype <- nn$hidtype
  ans$outtype <- nn$outtype
  class(ans) <- "summary.ffnet"
  return (ans)
}

print.summary.ffnet <- function (obj, digits = max(3,.Options$digits-3),
                                 signif.stars = .Options$show.signif.stars, ...)
{
  cat("\nCall:", deparse(obj$call), "", sep = "\n")
  cat ("Architecture:\n")
  cat ("a ",obj$nin,"-",obj$nhid,"-",obj$nout," network", sep="")
  if (obj$shortcut) cat (" with shortcuts\n")
  else cat ("\n")
  if (length(obj$coefnames)) cat ("inputs:", obj$coefnames, "\noutput(s):",
                                  obj$outnames, "\n")
  cat ("number of weights: ", obj$nw, "\n")
  cat ("hidden activation function: ")
  if (obj$hidtype == "TAN") cat ("tanh\n")
  else if (obj$hidtype == "SIG") cat ("logistic sigmoid\n")
  cat ("output activation function: ")
  if (obj$outtype == "LIN") cat ("linear\n")
  else if (obj$outtype == "SOFT") cat ("softmax\n")
  cat ("\nResiduals:\n")
  if (is.logical(obj$residuals))
  {
    tb <- table(obj$residuals)
    print(tb,...)
  }
  else
  {
    nam <- c("Min","1Q","Median","3Q","Max")
    if (NCOL(obj$residuals) > 1)
    {
      rq <- apply(obj$residuals,2,quantile)
      rownames(rq) <- nam
    }
    else
      rq <- structure (quantile(obj$residuals), names = nam)
    print (rq, digits=digits, ...)
  }
  cat ("\nGoodness of Fit:\n")
  if (is.logical(obj$residuals))
  {
    conf <- confusion(obj$observed,obj$fitted)
    cat ("Observed vs. Fitted\n")
    print (conf$tbl, ...)
    cat ("Classification Error: ", formatC(conf$error,digits = digits), "\n")
  }
  else
  {
    if (NCOL(obj$residuals) > 1)
    {
      cat ("R-squared (SSR/SST):\n")
      print (obj$r.squared.1, digits=digits, ...)
      cat ("R-squared (1-SSE/SST):\n")
      print (obj$r.squared.2, digits=digits, ...)
    }
    else
    {
      cat ("R-squared (SSR/SST): ", formatC(obj$r.squared.1,digits = digits), ",")
      cat ("   R-squared (1-SSE/SST): ", formatC(obj$r.squared.2,digits = digits), "\n")
    }
  }
  cat ("\nModel Selection Criteria:\n")
  cat ("AIC: ", format(round(obj$aic, digits=digits)), ",")
  cat ("   SC: ", format(round(obj$sc, digits=digits)), "\n")
  cat ("\nCoefficients:\n")
  print.coefmat (obj$coef, digits = digits, signif.stars = signif.stars, ...)
  if (length(obj$fwts) > 0)
  {
    cat ("\nConstants:\n")
    print (obj$fwts, ...)
  }
  cat ("\n")
  invisible (obj)
}

netplot <- function (nn, maxcolors=256, colors="size")
{
  
  seqN <- function(N)
  {
    if (0==length(N)) NULL else if (N<=0) NULL else seq(N)
  }
  
  colors <- pmatch(colors, c("pval", "size", "none"))
  if(length(grep("nnet", class(nn))>0))
  {
    nn$nin <- nn$n[1]
    nn$nhid <- nn$n[2]
    nn$nout<- nn$n[3]
    if(length(grep("i.*->o", names(coef(nn))))>0)
      nn$shortcut <- TRUE
    else
      nn$shortcut <- FALSE
  }
  m <- max(nn$nin, nn$nhid, nn$nout,3)
  plot.new()
  plot.window(c(0, m), c(1,m+0.3))
  w <- coef(nn)
  wnam <- sub(" ", "", names(w))
  wcol <- rep(NA, length(w))
  if(colors==1)
  {
    nns <- summary(nn)
    pval <- nns$coef[,"Pr(>|t|)"]
    if(any(is.na(pval)))
      stop("Some p values not available")
    wcol <- rgb(1, pval, pval)
    names(wcol) <- wnam
  }
  else if(colors==2)
  {
    s <- seq(1,0,length=maxcolors/2+1)
    mypalette <- c(rgb(rev(s), rev(s), 1), rgb(1, s, s))    
    names(wcol) <- wnam
    ok <- grep("b->", wnam)    
    maxweight <- max(abs(w[ok]))
    wcol[ok] <- mypalette[1+floor((w[ok]/maxweight+1)*(maxcolors/2))]
    maxweight <- max(abs(w[-ok]))
    wcol[-ok] <- mypalette[1+floor((w[-ok]/maxweight+1)*(maxcolors/2))]
  } 
  basei <- (m-nn$nin)/2
  baseh <- (m-nn$nhid)/2
  baseo <- (m-nn$nout)/2  
  for(k in seqN(nn$nin))
  {
    rect(0,basei+k,0.3,basei+k+0.3)
    text(0.15, basei+k+0.15, paste("i", k, sep=""), adj=0.5)
    for(l in seqN(nn$nhid))
    {
      arrows(0.3, basei+k+0.1, m/2-0.15, baseh+l+0.1,
             angle=5, col=wcol[paste("i", k, "->h", l, sep="")])
    }
  }             
  for(k in seqN(nn$nhid))
  {
    rect(m/2-0.15,baseh+k,m/2+0.15,baseh+k+0.3,
         col=wcol[paste("b->h", k, sep="")])
    text(m/2, baseh+k+0.15, paste("h", k, sep=""), adj=0.5)
    for(l in seqN(nn$nout))
    {
      arrows(m/2+0.15, baseh+k+0.1, m-0.3, baseo+l+0.1,
             angle=5, col=wcol[paste("h", k, "->o", l, sep="")])
    }
  }
  for(k in seqN(nn$nout))
  {
    rect(m-0.3,baseo+k,m,baseo+k+0.3,
         col=wcol[paste("b->o", k, sep="")])
    text(m-0.15, baseo+k+0.15, paste("o", k, sep=""), adj=0.5)
  }
  for(k in seqN(nn$nin))
  {
    if(nn$shortcut)
    {
      for(l in seqN(nn$nout))
      {
        arrows(0.3, basei+k+0.2, m-0.3, baseo+l+0.2, lty=2,
               angle=5, col=wcol[paste("i", k, "->o", l, sep="")])
      }
    }
  }            
}

    
                   
    

