/* Copyright (C) 1997-1999  Adrian Trapletti
  
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
  
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
  
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   extensions to numerical recipes procedures */


#include <math.h>
#include <stdio.h> 
#include "nrutil.h"
#include "nrlinal.h"
#include "nrext.h"


#include "nr.c"  /* brute force approach to hide the Numerical Recipes in the executable */


long init_ran2 = 1048575;  /* seed value for the random number generator ran2() */


void RPERM (int n, int k, int I[])
     /* Generate k elements I[1],..,I[k] randomly from {1,..,n} without replacement.
	Input is n, k and I[1..n], output I[1..k], k <= n, c.f. "G.S. Fishman (1996): 
	Monte Carlo, Springer, NY, 230-231". Note the error in Fishman's algorithm. */
{
  int i, J, a;
  double temp;

  for (i=1; i<=n; i++) I[i] = i; 
  for (i=1; i<=k; i++)
  {
    temp = (double)(n-i+1)*ran2();
    J = n-(int)temp;  /* instead of J = 1+(int)temp; */
    a = I[i]; I[i] = I[J]; I[J] = a; 
  }
}

int det_chol (double **a, int n, double* det)
     /* If the matrix 'a[1..n][1..n]' is not positive definite, then it returns 
	'zero'. Else it returns 'one' and stores the determinant of 'a' in 'det'.
	On input, only the upper triangle of 'a' need be given. It replaces 'a' 
	by the cholesky factor 'L', except the diagonal elements. */
{
  int i, res;
  double *p;
  
  p = dvector (1, n);
  res = choldc(a, n, p); 
  if (res) 
  {
    (*det) = p[1];
    for (i=2; i<=n; i++) (*det) *= p[i];
    (*det) *= (*det);
  }
  free_dvector (p, 1, n);
  return res;
}

int inv_chol (double **a, int n)
     /* If the matrix 'a[1..n][1..n]' is not positive definite, then it returns 
	'zero'. Else it returns 'one' and replaces 'a' by its inverse. On input, 
	only the upper triangle of 'a' need be given. */
{
  int i, j, k, res;
  double sum;
  double *p, **chol;

  p = dvector (1, n); chol = dmatrix (1, n, 1, n); 
  copy_dmatrix (a, chol, 1, n, 1, n);
  res = choldc (chol, n, p);
  if (res)
  {
    for (i=1; i<=n; i++)  /* find inverse of L */
    {
      chol[i][i] = 1.0/p[i];
      for (j=i+1; j<=n; j++) 
      {
	sum = 0.0;
	for (k=i; k<j; k++) sum -= chol[j][k]*chol[k][i];
	chol[j][i] = sum/p[j];
      }
    }
    for (i=1; i<=n; i++)  /* compute inv(a) = inv(L)'*inv(L) */
    {
      for (j=1; j<=n; j++) 
      {
	sum = 0.0;
	for (k=IMAX(i,j); k<=n; k++) sum += chol[k][i]*chol[k][j];
	a[i][j] = sum;
      }
    }
  }
  free_dvector (p, 1, n); free_dmatrix (chol, 1, n, 1, n);
  return res;
}

#define STEPS 10  /* if trace is set, then output is produced every STEPS steps */
#define EPS 1.0e-10  /* just a small number, c.f. NRC, 423 */

void grddsc (double p[], int n, double eta, double alpha, double ftol, int *iter, double *fret,
	     double (*func)(double []), void (*dfunc)(double [], double []), int trace, int ITMAX)
     /* From a given starting point p[1..n], simple gradient descent with momentum term
	is performed on a function func. The gradient is calculated by the routine dfunc. 
	The convergence tolerance on the function value is given by ftol. The learning 
	rate and the momentum parameter are input as eta and alpha. To make grddsc 
	working silently set trace to zero. Returned quantities are the location of the
	minimum p, the number of iterations that were performed iter (less than or equal 
	ITMAX), and the minimum value of the function fret. */
{
  int i, j;
  double gsum, fp;
  double *g, *dp;
  
  g = dvector (1, n); dp = dvector (1, n); 
  fp = (*func)(p);
  if (trace) 
  {
    if (trace == PRINT) {
      printf ("grddsc objective function values\n");
      printf ("initial     value %f\n",fp);
    }
    else if (trace == RPLOT) {
#ifdef RFLAG
      R_plot (0, fp);
#else
      nrerror ("R callback not installed");
#endif
    }
    else
      nrerror ("grddsc: invalid trace value");
  }
  (*dfunc)(p, g);
  vEQvTId (dp, g, -eta, n);  /* initialization of dp */
  for (i=1; i<=ITMAX; i++) 
  {
    *iter = i;
    vPLEQv (p, dp, n);  /* go to new point p */
    *fret = (*func)(p);  /* new function value *fret */ 
    if (2.0*fabs(*fret-fp) <= ftol*(fabs(*fret)+fabs(fp)+EPS))  /* compare with old function value fp */
    {
      if (trace)
      {
	if (trace == PRINT) {
	  printf ("final       value %f\n",fp);
	  printf ("grddsc converged after %d iterations\n",i);
	}
	else if (trace == RPLOT) {
#ifdef RFLAG
	  R_plot (i, fp);
#else
	  nrerror ("R callback not installed");
#endif
	}
	else
	  nrerror ("grddsc: invalid trace value");
      }
      free_dvector (dp, 1, n); free_dvector (g, 1, n);
      return;
    }
    fp = *fret;  /* old function fp value becomes new function value *fret */
    if ((trace) && ((i % STEPS)==0)) {
      if (trace == PRINT) printf("iter %6d value %f\n",i,fp);
      else if (trace == RPLOT) {
#ifdef RFLAG
	R_plot (i, fp);
#else
	nrerror ("R callback not installed");
#endif
      }
      else
	nrerror ("grddsc: invalid trace value");
    }
    (*dfunc)(p, g);
    gsum = INNER (g, g, n);  /* sum of squares of gradient elements */
    if (gsum == 0.0)  /* gradient is exactly zero */
    {
      if (trace)
      {
	if (trace == PRINT) {
	  printf ("final       value %f\n",fp);
	  printf ("grddsc converged after %d iterations\n",i);
	}
	else if (trace == RPLOT) {
#ifdef RFLAG
	  R_plot (i, fp);
#else
	  nrerror ("R callback not installed");
#endif
	}
	else
	  nrerror ("grddsc: invalid trace value");
      }
      free_dvector (dp, 1, n); free_dvector (g, 1, n);
      return;
    }
    for (j=1; j<=n; j++) dp[j] = -eta*g[j]+alpha*dp[j]; /* update dp */
  }
  if (trace)
  {
    if (trace == PRINT) {
      printf ("final       value %f\n",fp);
      printf ("grddsc not converged after %d iterations\n",i-1);
    }
    else if (trace == RPLOT) {
#ifdef RFLAG
      R_plot (i-1, fp);
#else
      nrerror ("R callback not installed");
#endif
    }
    else
      nrerror ("grddsc: invalid trace value");
  }
  free_dvector (dp, 1, n); free_dvector (g, 1, n);
}

#undef STEPS
#undef EPS

#define STEPS 10  /* if trace is set, then output is produced every STEPS steps */
#define EPS 1.0e-10  /* just a small number, c.f. NRC, 423 */

void stpdsc (double p[], int n, double ftol, int *iter, double *fret,
	     double (*func)(double []), void (*dfunc)(double [], double []), int trace, int ITMAX)
     /* From a given starting point p[1..n], steepest descent is performed on a function func.
	The gradient is calculated by the routine dfunc. The convergence tolerance on the 
	function value is given by ftol. To make stpdsc working silently set trace to zero.
	Returned quantities are the location of the minimum p, the number of iterations 
	that were performed iter (less than or equal ITMAX), and the minimum value of the 
	function fret. */
{
  int i;
  double gsum, fp;
  double *g, *dp;
  
  g = dvector (1, n); dp = dvector (1, n);
  fp = (*func)(p);
  if (trace) 
  {
    if (trace == PRINT) {
      printf ("stpdsc objective function values\n");
      printf ("initial     value %f\n",fp);
    }
    else if (trace == RPLOT) {
#ifdef RFLAG
      R_plot (0, fp);
#else
      nrerror ("R callback not installed");
#endif
    }
    else
      nrerror ("stpdsc: invalid trace value");
  }
  (*dfunc)(p, g);
  vEQMIv (dp, g, n);  /* initialization of dp */
  for (i=1; i<=ITMAX; i++) 
  {
    *iter = i;
    linmin (p, dp, n, fret, func);  /* find new point p and new function value *fret */
    if (2.0*fabs(*fret-fp) <= ftol*(fabs(*fret)+fabs(fp)+EPS)) /* compare with old function value fp */
    {
      if (trace)
      {
	if (trace == PRINT) {
	  printf ("final       value %f\n",fp);
	  printf ("stpdsc converged after %d iterations\n",i);
	}
	else if (trace == RPLOT) {
#ifdef RFLAG
	  R_plot (i, fp);
#else
	  nrerror ("R callback not installed");
#endif
	}
	else
	  nrerror ("stpdsc: invalid trace value");
      }
      free_dvector (dp, 1, n); free_dvector (g, 1, n);
      return;
    }
    fp = *fret;  /* old function fp value becomes new function value *fret */
    if ((trace) && ((i % STEPS)==0)) {
      if (trace == PRINT) printf("iter %6d value %f\n",i,fp);
      else if (trace == RPLOT) {
#ifdef RFLAG
	R_plot (i, fp);
#else
	nrerror ("R callback not installed");
#endif
      }
      else
	nrerror ("stpdsc: invalid trace value");
    } 
    (*dfunc)(p, g);
    gsum = INNER (g, g, n);  /* sum of squares of gradient elements */
    if (gsum == 0.0) 
    {
      if (trace)
      {
	if (trace == PRINT) {
	  printf ("final       value %f\n",fp);
	  printf ("stpdsc converged after %d iterations\n",i);
	}
	else if (trace == RPLOT) {
#ifdef RFLAG
	  R_plot (i, fp);
#else
	  nrerror ("R callback not installed");
#endif
	}
	else
	  nrerror ("stpdsc: invalid trace value");
      }
      free_dvector (dp, 1, n); free_dvector (g, 1, n);
      return;
    }
    vEQMIv (dp, g, n);  /* update dp */
  }
  if (trace)
  {
    if (trace == PRINT) {
      printf ("final       value %f\n",fp);
      printf ("stpdsc not converged after %d iterations\n",i-1);
    }
    else if (trace == RPLOT) {
#ifdef RFLAG
      R_plot (i-1, fp);
#else
      nrerror ("R callback not installed");
#endif
    }
    else
      nrerror ("stpdsc: invalid trace value");
  }
  free_dvector (dp, 1, n); free_dvector (g, 1, n);
}

#undef STEPS
#undef EPS

#define E1 1.7182818  /* e-1.0 */
#define STEPS 100

void sann (double pb[], double *yb, long n, int itmax, int kmax, 
	   double ti, double (*func)(double []), int trace)
     /* Given a starting point pb[1..n], simulated annealing minimization is performed 
	on a function func. The starting temperature is input as ti. To make sann 
	working silently set trace to zero. sann makes in total itmax function evaluations 
	and kmax evaluations at each temperature (function evaluations for initialization 
	are not included). Returned quantities are pb (the location 
	of the minimum), and yb (the minimum value of the function func). */
{
  long i;
  int k, its, itdoc;
  double t, y, dy, ytry, scale;
  double *p, *dp, *ptry;

  p = dvector (1, n); dp = dvector (1, n); ptry = dvector (1, n); 
  *yb = (*func)(pb);  /* init best system state pb, *yb */
  copy_dvector (pb, p, 1, n); y = *yb;  /* init system state p, y */
  if (trace) 
  {
    if (trace == PRINT) {
      printf ("sann objective function values\n");
      printf ("initial       value %f\n",*yb);
    }  
    else if (trace == RPLOT) {
#ifdef RFLAG
      R_plot (0, *yb);
#else
      nrerror ("R callback not installed");
#endif
    }
    else
      nrerror ("sann: invalid trace value");
  }
  scale = 1.0/ti;
  its = itdoc = 1;
  while (its <= itmax)  /* cool down system */
  {
    t = ti/log((double)its+E1);  /* temperature annealing schedule */
    k = 1;
    while ((k <= kmax) && (its <= itmax))  /* iterate at constant temperature */
    {
      for (i=1; i<=n; i++)
	dp[i] = scale*t*gasdev();  /* random perturbation */
      for (i=1; i<=n; i++)
	ptry[i] = p[i]+dp[i];  /* new candidate point */
      ytry = (*func)(ptry); 
      dy = ytry-y;
      if ((dy <= 0.0) || (ran2() < exp(-dy/t)))  /* accept new point */
      {
	copy_dvector (ptry, p, 1, n); y = ytry;  /* update system state p, y */
	if (y <= *yb)  /* if system state is best, then update best system state pb, *yb */
	{
	  copy_dvector (p, pb, 1, n); *yb = y;
	}
      }        
      its++; k++;
    }
    if ((trace) && ((itdoc % STEPS)==0)) {
      if (trace == PRINT) printf("iter %8d value %f\n",its-1,*yb);
      else if (trace == RPLOT) {
#ifdef RFLAG
	R_plot (its-1, *yb);
#else
	nrerror ("R callback not installed");
#endif
      }
      else
	nrerror ("sann: invalid trace value");
    } 
    itdoc++;
  }
  if (trace)
  {
    if (trace == PRINT) {
      printf ("final         value %f\n",*yb);
      printf ("sann stopped after %d iterations\n",its-1);
    }
    else if (trace == RPLOT) {
#ifdef RFLAG
      R_plot (its-1, *yb);
#else
      nrerror ("R callback not installed");
#endif
    }
    else
      nrerror ("sann: invalid trace value");
  }
  free_dvector (ptry, 1, n); free_dvector (dp, 1, n); free_dvector (p, 1, n);
}

#undef E1
#undef STEPS

#define FTOL 1.0e-6  /* fractional tolerance for an early return, see "NRC, 452-454". */
#define E1 1.7182818  /* e-1.0 */
#define STEPS 100

void nrsann (double pb[], double *yb, long n, int itmax, int kmax, 
	     double ti, double (*func)(double []), int trace)
     /* Given a starting point pb[1..n], Numerical Recipes simulated annealing 
	minimization is performed on a function func. The starting temperature 
	is input as ti. To make nrsann working silently set trace to zero. nrsann 
	makes in total itmax function evaluations and at most kmax evaluations 
	at each temperature. (function evaluations for initialization 
	are not included). Returned quantities are pb (the location of the 
	minimum), and yb (the minimum value of the function func). */
{
  long i, j;
  int its, iter, itdoc;
  double t;
  double **p, *y;

  p = dmatrix (1, n+1, 1, n); y = dvector (1, n+1);
  *yb = (*func)(pb);  /* init best system state pb, *yb */
  for (i=1; i<=n+1; i++)  /* init system state p (starting simplex) and y */
  {
    for (j=1; j<=n; j++) 
      p[i][j] = pb[j]+gasdev();
    y[i] = (*func)(&p[i][0]);
  }
  if (trace) 
  {
    if (trace == PRINT) {
      printf ("nrsann objective function values\n");
      printf ("initial       value %f\n",*yb);
    }  
    else if (trace == RPLOT) {
#ifdef RFLAG
      R_plot (0, *yb);
#else
      nrerror ("R callback not installed");
#endif
    }
    else
      nrerror ("nrsann: invalid trace value");
  }
  its = itdoc = 1;
  while (its <= itmax)  /* cool down system */
  {
    t = ti/log((double)its+E1);  /* temperature annealing schedule */
    iter = IMIN(itmax-its+1,kmax);  /* make at most iter function evaluations at constant temperature */
    amebsa(p, y, n, pb, yb, FTOL, func, &iter, t, trace);  /* iterate at constant temperature */
    its += IMIN(itmax-its+1,kmax)-IMAX(0,iter);  /* in fact made MIN(itmax-its+1,kmax)-iter evaluations */
    if ((trace) && ((itdoc % STEPS)==0)) {
      if (trace == PRINT) printf("iter %8d value %f\n",its-1,*yb);
      else if (trace == RPLOT) {
#ifdef RFLAG
	R_plot (its-1, *yb);
#else
	nrerror ("R callback not installed");
#endif
      }
      else
	nrerror ("nrsann: invalid trace value");
    } 
    itdoc++;
  }
  if (trace)
  {
    if (trace == PRINT) {
      printf ("final         value %f\n",*yb);
      printf ("nrsann stopped after %d iterations\n",its-1);
    }
    else if (trace == RPLOT) {
#ifdef RFLAG
      R_plot (its-1, *yb);
#else
      nrerror ("R callback not installed");
#endif
    }
    else
      nrerror ("nrsann: invalid trace value");
  }
  free_dvector (y, 1, n+1); free_dmatrix (p, 1, n+1, 1, n);
}

#undef FTOL
#undef E1
#undef STEPS

#ifdef RFLAG
extern void call_R (char*, long, void**, char**, long*, char**, long, char**);
extern void *R_callback;

void R_plot (int its, double fp)
     /* This routine makes the callback to R, i.e., redirects the output of 
	the optimizer to R. The arguments passed to R are an integer and a float
	representing the actual number of iterations and the actual value of the 
	objective function. */
{
  void *args[2];
  char *mode[2], *values[1];
  long length[2];    
  double f[1];
  int i[1];
  
  mode[0] = "integer"; mode[1] = "double"; 
  length[0] = 1; length[1] = 1;
  args[0] = (void*)(i); i[0] = its; 
  args[1] = (void*)(f); f[0] = fp;	  
  call_R (R_callback, 2L, args, mode, length, 0L, 1L, values);
}
#endif

#define EPS3 6.055454e-06  /* macheps^{1/3} */

double numdf (double (*f)(double), double x)
     /* computes the central difference approximation to f'(x), c.f. "NRC, 186, 187" */
{
  double h, temp;
  
  if (x == 0.0) h = EPS3;
  else h = EPS3*x;
  temp = x+h;
  h = temp-x;
  return ((*f)(x+h)-(*f)(x-h))/(2.0*h);
}

#undef EPS3

