// Copyright (C) 1997-1999  Adrian Trapletti
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

//
// Linear algebra package 
//


#ifndef _LINAL_HH_
#define _LINAL_HH_


#include <iostream.h>


class vec;
class mat;


// Closures for optimizations of vector calculations, "B.Stroustrup (1997): 
// The C++ Programming Language, 3th Edition, Addison-Wesley, Reading (MA), 675-677"

struct vPLv
{
  const vec& v1;
  const vec& v2;
  
  vPLv (const vec& vv1, const vec& vv2): v1(vv1), v2(vv2) {}
};

inline vPLv operator+ (const vec& vv1, const vec& vv2)
{
  return vPLv (vv1, vv2);
}

struct vMIv
{
  const vec& v1;
  const vec& v2;
  
  vMIv (const vec& vv1, const vec& vv2): v1(vv1), v2(vv2) {}
};

inline vMIv operator- (const vec& vv1, const vec& vv2)
{
  return vMIv (vv1, vv2);
}

struct vTIv
{
  const vec& v1;
  const vec& v2;

  vTIv (const vec& vv1, const vec& vv2): v1(vv1), v2(vv2) {}
};

inline vTIv operator* (const vec& vv1, const vec& vv2)
{
  return vTIv (vv1, vv2);
}

struct vDIv
{
  const vec& v1;
  const vec& v2;

  vDIv (const vec& vv1, const vec& vv2): v1(vv1), v2(vv2) {}
};

inline vDIv operator/ (const vec& vv1, const vec& vv2)
{
  return vDIv (vv1, vv2);
}

struct vPLd
{
  const vec& v;
  double d;

  vPLd (const vec& vv, double dd): v(vv), d(dd) {}
};

inline vPLd operator+ (const vec& vv, double dd)
{
  return vPLd (vv, dd);
}

struct vMId
{
  const vec& v;
  double d;

  vMId (const vec& vv, double dd): v(vv), d(dd) {}
};

inline vMId operator- (const vec& vv, double dd)
{
  return vMId (vv, dd);
}

struct vTId
{
  const vec& v;
  double d;

  vTId (const vec& vv, double dd): v(vv), d(dd) {}
};

inline vTId operator* (const vec& vv, double dd)
{
  return vTId (vv, dd);
}

struct vDId
{
  const vec& v;
  double d;

  vDId (const vec& vv, double dd): v(vv), d(dd) {}
};

inline vDId operator/ (const vec& vv, double dd)
{
  return vDId (vv, dd);
}

struct SQRv
{
  const vec& v;

  SQRv (const vec& vv): v(vv) {}
};

inline SQRv sqr (const vec& vv)
{
  return SQRv (vv);
}


// Closures for optimizations of matrix calculations

struct mPLm
{
  const mat& ma1;
  const mat& ma2;

  mPLm (const mat& mma1, const mat& mma2): ma1(mma1), ma2(mma2) {}
};

inline mPLm operator+ (const mat& mma1, const mat& mma2)
{
  return mPLm (mma1, mma2);
}

struct mMIm
{
  const mat& ma1;
  const mat& ma2;

  mMIm (const mat& mma1, const mat& mma2): ma1(mma1), ma2(mma2) {}
};

inline mMIm operator- (const mat& mma1, const mat& mma2)
{
  return mMIm (mma1, mma2);
}

struct mTIm
{
  const mat& ma1;
  const mat& ma2;

  mTIm (const mat& mma1, const mat& mma2): ma1(mma1), ma2(mma2) {}
};

inline mTIm operator* (const mat& mma1, const mat& mma2)
{
  return mTIm (mma1, mma2);
}

struct mDIm
{
  const mat& ma1;
  const mat& ma2;

  mDIm (const mat& mma1, const mat& mma2): ma1(mma1), ma2(mma2) {}
};

inline mDIm operator/ (const mat& mma1, const mat& mma2)
{
  return mDIm (mma1, mma2);
}

struct mPLd
{
  const mat& ma;
  double d;

  mPLd (const mat& mma, double dd): ma(mma), d(dd) {}
};

inline mPLd operator+ (const mat& mma, double dd)
{
  return mPLd (mma, dd);
}

struct mMId
{
  const mat& ma;
  double d;

  mMId (const mat& mma, double dd): ma(mma), d(dd) {}
};

inline mMId operator- (const mat& mma, double dd)
{
  return mMId (mma, dd);
}

struct mTId
{
  const mat& ma;
  double d;

  mTId (const mat& mma, double dd): ma(mma), d(dd) {}
};

inline mTId operator* (const mat& mma, double dd)
{
  return mTId (mma, dd);
}

struct mDId
{
  const mat& ma;
  double d;

  mDId (const mat& mma, double dd): ma(mma), d(dd) {}
};

inline mDId operator/ (const mat& mma, double dd)
{
  return mDId (mma, dd);
}

struct mMTIm
{
  const mat& ma1;
  const mat& ma2;

  mMTIm (const mat& mma1, const mat& mma2): ma1(mma1), ma2(mma2) {}
};

inline mMTIm mul (const mat& mma1, const mat& mma2)
{
  return mMTIm (mma1, mma2);
}


// Closures for optimizations of mixed calculations

struct vOUTv
{
  const vec& v1;
  const vec& v2;

  vOUTv (const vec& vv1, const vec& vv2): v1(vv1), v2(vv2) {}
};

inline vOUTv out (const vec& vv1, const vec& vv2)
{
  return vOUTv (vv1, vv2);
}

struct mTIv
{
  const mat& ma;
  const vec& v;

  mTIv (const mat& mma, const vec& vv): ma(mma), v(vv) {}
};

inline mTIv operator* (const mat& mma, const vec& vv)
{
  return mTIv (mma, vv);
}


extern void vec2Rvec (const vec& v_src, double* v_dest);  // copy vec to already allocated R vector
extern void mat2Rmat (const mat& m_src, double* m_dest);  // copy mat to already allocated R matrix


class vec  // column vector of double
{
  friend class mat;
  
  friend istream& operator>> (istream& s, vec& v);  // deallocate v and read v from istream
  friend ostream& operator<< (ostream& s, const vec& v);  
  friend double inn (const vec& v1, const vec& v2);  // inner product 
  friend double sum (const vec& v);  // sum over all elements
  friend double mean (const vec& v);  // arithmetic mean over all elements
  
private:
  long n;  // number of elements
  double* p;  // pointer to elements p[1..n]
  
public:  
  vec (const vec& v);  // copy constructor
  vec (long nn = 0); 
  vec (double* x, long nn);  // copy constructor with an R vector as input
  ~vec (); 
  vec& operator= (const vec& v);  // assignment 
  vec& operator= (const vPLv& v);  // v3 = v1+v2; elementwise operations
  vec& operator= (const vMIv& v);  // v3 = v1-v2;
  vec& operator= (const vTIv& v);  // v3 = v1*v2; 
  vec& operator= (const vDIv& v);  // v3 = v1/v2; 
  vec& operator+= (const vec& v);  // v2 += v1;
  vec& operator-= (const vec& v);  // v2 -= v1;
  vec& operator*= (const vec& v);  // v2 *= v1; 
  vec& operator/= (const vec& v);  // v2 /= v1; 
  vec& operator+= (double d);  // v += d;
  vec& operator-= (double d);  // v -= d;
  vec& operator*= (double d);  // v *= d;
  vec& operator/= (double d);  // v /= d;
  vec& operator= (const vPLd& v);  // v2 = v1+d;
  vec& operator= (const vMId& v);  // v2 = v1-d;
  vec& operator= (const vTId& v);  // v2 = v1*d;
  vec& operator= (const vDId& v);  // v2 = v1/d;
  vec& operator= (const mTIv& ma);  // v2 = m*v1;  matrix times vector
  vec& rand (int type, double mean, double var);  // initialize vector elements randomly
  vec& zero ();  // set vector elements to zero
  vec& one ();  // set vector elements to one
  vec& operator= (const SQRv& v);  // v2 = sqr(v1);  elementwise square of vector
  void resize (long nn);  // (*this) is resized to a new size of nn elements
  
#ifdef CHECK
  double& operator() (long i);  // read and write elements
#else
  double& operator() (long i) { return p[i]; }
#endif
#ifdef CHECK
  double operator() (long i) const;  // read elements
#else
  double operator() (long i) const { return p[i]; }
#endif
  double* elem () const { return p; }  // return pointer to elements
  long rows () const { return n; }  // return number of rows
};


class mat  // matrix of double
{
  friend class vec;
  
  friend istream& operator>> (istream& s, mat& ma);  // deallocate m and read m from istream
  friend ostream& operator<< (ostream& s, const mat& ma);  
  
private:
  long n, m;  // number of rows and columns
  double** p;  // pointer to elements p[1..n][1..m]

public:  
  mat (const mat& ma);  // copy constructor
  mat (long nn = 0, long mm = 0); 
  mat (double* x, long nn, long mm);  // copy constructor with an R matrix as input
  ~mat (); 
  mat& operator= (const mat& ma);  // assignment 
  mat& operator= (const mPLm& ma);  // m3 = m1+m2; elementwise operations
  mat& operator= (const mMIm& ma);  // m3 = m1-m2;
  mat& operator= (const mTIm& ma);  // m3 = m1*m2;  
  mat& operator= (const mDIm& ma);  // m3 = m1/m2;
  mat& operator+= (const mat& ma);  // m2 += m1;
  mat& operator-= (const mat& ma);  // m2 -= m1;
  mat& operator*= (const mat& ma);  // m2 *= m1;
  mat& operator/= (const mat& ma);  // m2 /= m1;
  mat& operator+= (double d);  // m += d;
  mat& operator-= (double d);  // m -= d;
  mat& operator*= (double d);  // m *= d;
  mat& operator/= (double d);  // m /= d;
  mat& operator= (const mPLd& ma);  // m2 = m1+d;
  mat& operator= (const mMId& ma);  // m2 = m1-d;
  mat& operator= (const mTId& ma);  // m2 = m1*d;
  mat& operator= (const mDId& ma);  // m2 = m1/d;
  mat& operator= (const mMTIm& ma);  // m3 = mul(m1,m2);  matrix multiplication
  mat& operator= (const vOUTv& v);  // m = out(v1,v2);  outer product
  mat& rand (int type, double mean, double var);  // initialize matrix elements randomly
  mat& zero ();  // set matrix elements to zero
  mat& one ();  // set matrix elements to one
  mat& eye ();  // set matrix to identity matrix, only for a square matrix
  void resize (long nn, long mm);  // (*this) is resized to a new size of nn x mm elements

#ifdef CHECK
  double& operator() (long i, long j);  // read and write elements
#else
  double& operator() (long i, long j) { return p[i][j]; }
#endif
#ifdef CHECK
  double operator() (long i, long j) const;  // read elements
#else
  double operator() (long i, long j) const { return p[i][j]; }
#endif
  double** elem () const { return p; }  // return pointer to elements
  long rows () const { return n; }  // return number of rows
  long columns () const { return m; }  // return number of columns
};


class ivec  // vector with elements of type integer
{
  friend istream& operator>> (istream& s, ivec& v);  // deallocate v and read v from istream       
  friend ostream& operator<< (ostream& s, const ivec& v);
  friend ivec rperm (long n, long k); 
  /* Returns the ivec I(k) with the k elements I(1),..,I(k) sampled
     randomly from {1,..,n} without replacement. */
  
private:
  long n;  // number of elements                       
  int* p;  // pointer to elements p[1..n]                                     
  
public:
  ivec (const ivec& v);  // copy constructor
  ivec (long nn = 0); 
  ~ivec (); 
  ivec& operator= (const ivec& v);  // assignment 

#ifdef CHECK
  int& operator() (long i);  // read and write elements
#else
  int& operator() (long i) { return p[i]; }
#endif
#ifdef CHECK
  int operator() (long i) const;  // read elements
#else
  int operator() (long i) const { return p[i]; }
#endif
  int* elem () const { return p; }  // return pointer to elements
  long rows () const { return n; }  // return number of rows  
};


#endif  // _LINAL_HH_





