// Copyright (C) 1997-1999  Adrian Trapletti
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

//
// String class 
//


#include <iostream.h>
#include <ctype.h>
#include <string.h>
extern "C" {
#include <stdlib.h>
#ifdef RFLAG
#include "R.h"
#endif
}
#include "utils.hh"
#include "str.hh"


/* from B.Stroustrup (1994), The C++ Programming Language, 2nd Edition,
   Addison-Wesley, Reading, MA, 248-251 */

str::str ()  // str x;
{
#ifdef RFLAG
  p = (srep *) R_alloc (1, sizeof(srep));
#else
  p = (srep *) malloc (sizeof(srep));
#endif
  p->s = 0;
}

str::str (const char* s)  // str x = "abc"
{
#ifdef RFLAG
  p = (srep *) R_alloc (1, sizeof(srep));
  p->s = (char *) R_alloc (strlen(s)+1, sizeof(char));
#else
  p = (srep *) malloc (sizeof(srep));
  p->s = (char *) malloc ((size_t) ((strlen(s)+1)*sizeof(char)));
#endif  
  strcpy (p->s, s);
}

str::~str ()
{
  if (--p->n == 0)
  {
#ifndef RFLAG
    free ((void *) (p->s));
    free ((void *) p);
#endif 
  }
}

str& str::operator= (const char* s)
{
  if (p->n > 1)  // disconnect self
  {
    p->n--;
#ifdef RFLAG
    p = (srep *) R_alloc (1, sizeof(srep));
#else
    p = (srep *) malloc (sizeof(srep));
#endif    
  }
  else  // free old str
  {
#ifndef RFLAG
    free ((void *) (p->s));
#endif     
  }
#ifdef RFLAG
  p->s = (char *) R_alloc (strlen(s)+1, sizeof(char)); 
#else
  p->s = (char *) malloc ((size_t) ((strlen(s)+1)*sizeof(char)));  
#endif 
  strcpy (p->s, s);
  return *this;
}

str& str::operator= (const str& x)
{
  x.p->n++;  // protect against "st = st"
  if (--p->n == 0)
  {
#ifndef RFLAG
    free ((void *) (p->s));
    free ((void *) p);
#endif     
  }
  p = x.p;
  return *this;
}

istream& operator>> (istream& s, str& x)
{
  const buf_length = 256;

  int i = 0;
  char ch, buf[buf_length];
  
  while (s.get (ch))  // eat whitespace and control characters
  {
    if (!isspace (ch) && !iscntrl (ch))
    {
      s.putback (ch); break;
    }
  }
  while (s.get (ch) && (i < buf_length-1) && !isspace (ch) && !iscntrl (ch))  // read str
    buf[i++] = ch;
  if ((s) && (isspace (ch) || iscntrl (ch)))  // successful
  {
    s.putback (ch);
    buf[i] = 0; x = buf;
  }
  else  // not successful
    s.clear (ios::badbit);
  return s;
}

char& str::operator[] (int i)
{
  srep* np;
  
#ifdef CHECK
  if ((i < 0) || ((int)strlen(p->s) < i)) RTerror ("ts_util.cc", "char& str::operator[] (int i)");
#endif
  if (p->n > 1)  // clone to maintain value semantics
  {
#ifdef RFLAG
    np = (srep *) R_alloc (1, sizeof(srep));
    np->s = (char *) R_alloc (strlen(p->s)+1, sizeof(char));
#else
    np = (srep *) malloc (sizeof(srep));
    np->s = (char *) malloc ((size_t) ((strlen(p->s)+1)*sizeof(char)));
#endif    
    strcpy (np->s, p->s);
    p->n--;
    p = np;
  }
  return p->s[i];
}

const char& str::operator[] (int i) const
{
#ifdef CHECK
  if ((i < 0) || ((int)strlen(p->s) < i)) 
    RTerror ("ts_util.cc", "const char& str::operator[] (int i) const");
#endif
  return p->s[i];
}

char* str::copy () const  // copy string into char* and return char*
{
  char* s;
  
#ifdef RFLAG
  s = (char *) R_alloc (strlen(p->s)+1, sizeof(char));
#else
  s = (char *) malloc ((size_t) ((strlen(p->s)+1)*sizeof(char)));
#endif
  if (!s) RTerror ("ts_util.cc", "char* str::copy ()");
  strcpy (s, p->s);
  return s;
}




