% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bsSpike.R
\name{bsSpike}
\alias{bsSpike}
\title{Bayesian single-index regression with B-spline link and spike-and-slab prior}
\usage{
bsSpike(
  x,
  y,
  prior = list(index = list(nu = list(r1 = 1, r2 = 1), index = list(sigma_theta = 0.25)),
    link = list(basis = list(df = 21, degree = 2, delta = 0.01), beta = list(mu = NULL,
    cov = NULL)), sigma2 = list(shape = 0.001, rate = 100)),
  init = list(index = list(pi = 0.5, nu = NULL, index = NULL), link = list(beta = NULL),
    sigma2 = 0.01),
  sampling = TRUE,
  fitted = TRUE,
  monitors2 = NULL,
  niter = 10000,
  nburnin = 1000,
  thin = 1,
  thin2 = NULL,
  nchain = 1,
  setSeed = FALSE
)
}
\arguments{
\item{x}{Numeric data.frame/matrix of predictors. Each row is an observation.}

\item{y}{Numeric response vector/matrix.}

\item{prior}{Optional named list of prior settings with sublists:
\describe{
\item{\code{index}}{Spike and slab prior hyperparameters: Beta-binomial for variable selection indicator \eqn{\nu} (default \code{r1 = 1, r2 = 1}),
and normal distribution for selected variables \eqn{\theta} (default: \eqn{N(0, \sigma_{\theta}^{2})})}
\item{\code{link}}{B-spline basis and coefficient of B-spline setup.
\enumerate{
\item{\code{basis} For the basis of B-spline, \code{df} is the number of basis functions (default \code{21}), \code{degree} is the spline degree (default \code{2}) and \code{delta} is a small jitter for boundary-knot spacing control (default \code{0.01}).}
\item{\code{beta} For the coefficient of B-spline, multivariate normal prior is assigned with mean \code{mu}, and covariance \code{cov}. By default, \eqn{\mathcal{N}_p\!\big(0, \mathrm{I}_p\big)}}
}}
\item{\code{sigma2}}{Error-variance prior hyperparameters. An Inverse-Gamma prior is assigned to \eqn{\sigma^2}
where \code{shape} is shape parameter and \code{rate} is rate parameter of inverse gamma distribution.
(default \code{shape = 0.001, rate = 100})}
}}

\item{init}{Optional named list of initial values:
\describe{
\item{\code{index}}{
\enumerate{
\item{\code{pi} Initial selecting variable probability. (default: \code{0.5})}
\item{\code{nu} Initial vector of inclusion indicators . By default, each nu is randomly drawn by  \eqn{Bernoulli(1/2)}}
\item{\code{index} Initial vector of index. By default, each element of indedx vector, which is chosen by \eqn{nu}, is proposed by normal distribution.}
}}
\item{\code{link}}{Initial spline coefficients (\code{beta}). By default,
\eqn{\big(X_{\theta}^\top X_{\theta} + \rho\, \mathrm{I}\big)^{-1} X_{\theta}^\top Y} is computed,
where \eqn{X_{\theta}} is the B-spline basis design matrix.}
\item{\code{sigma2}}{Initial scalar error variance (default \code{0.01}).}
}}

\item{sampling}{Logical. If \code{TRUE} (default), run MCMC; otherwise return prepared nimble model objects without sampling.}

\item{fitted}{Logical. If \code{TRUE} (default), fitted values drawn from posterior distribution are included in the output and \code{c("Xlin", "linkFunction", "beta")} is monitored for prediction.}

\item{monitors2}{Optional character vector of additional monitor nodes. To check the names of the nodes, set \code{fit <- bsSpike(x, y, sampling = FALSE)} and then inspect the variable names stored in the model object using \code{fit$model$getVarNames()}.}

\item{niter}{Integer. Total MCMC iterations (default \code{10000}).}

\item{nburnin}{Integer. Burn-in iterations (default \code{1000}).}

\item{thin}{Integer. Thinning for monitors1 (default \code{1}).}

\item{thin2}{Integer. Optional thinning for \code{monitors2} (default \code{1}).}

\item{nchain}{Integer. Number of MCMC chains (default \code{1}).}

\item{setSeed}{Logical or numeric argument.  Further details are provided in \link[nimble]{runMCMC}.}
}
\value{
A \code{list} typically containing:
\describe{
\item{\code{model}}{Nimble model}
\item{\code{sampler}}{Nimble sampler}
\item{\code{sampling}}{Posterior draws of \eqn{\nu}, \eqn{\theta}, \eqn{\sigma^2}, and nodes for fitted values by default. Variables specified in \code{monitors2} will be added if provided.}
\item{\code{fitted}}{If \code{fitted = TRUE}, in-sample fitted values is given.}
\item{\code{input}}{List of data,input values for prior and initial values, and computation time without compiling.}
}
}
\description{
Fits a single-index model \eqn{Y_i \sim \mathcal{N}(f(X_i'\theta), \sigma^2), i = 1,\cdots,n}
where the link \eqn{f(\cdot)} is represented by B-spline link function and the
index vector \eqn{\theta} has spike-and-slab prior.
}
\details{
\strong{Model} The single–index model uses a \eqn{m}-order polynomial spline with \eqn{k} interior knots as follows:
\eqn{f(t) = \sum_{j=1}^{1+m+k} B_j(t)\,\beta_j} on \eqn{[a, b]} with \eqn{t_i = x_i' \theta, i = 1,\cdots, n}
and \eqn{\|\theta\|_2 = 1}. \eqn{\{\beta_j\}_{j=1}^{m+k}} are spline coefficient and \eqn{a_\theta} and \eqn{ b_\theta} are boundary knots where \eqn{a = min(t_i, i = 1, \cdots, n) - \delta},
and \eqn{b = max(t_i, i = 1,\cdots, n) + \delta}. \eqn{\theta} is a p-dimensional index vector subject to a spike-and-slab
prior for variable selection with binary indicator variable \eqn{\nu}.

\strong{Priors}
\itemize{
\item Slab coefficients \eqn{\theta}: Gaussian \eqn{N(0, \sigma_\theta^2)}.
\item Inclusion indicators \eqn{\nu}: Bernoulli(\eqn{\pi}).
\item Inclusion probability \eqn{\pi}: Beta(\eqn{r_1, r_2}).
\item Inverse-Gamma prior on \eqn{\sigma^2}: \eqn{\sigma^2 \sim \mathrm{IG}(a_\sigma, b_\sigma)}.
\item Conditional on \eqn{\theta} and \eqn{\sigma^2}, the link coefficients follow
\eqn{\beta = (\beta_1,\ldots,\beta_K)^\top \sim \mathcal{N}\!\big(\hat{\beta}_{\theta},\, \sigma^2 (X_{\theta}^\top X_{\theta})^{-1}\big)}.
}

\strong{Sampling}
Samplers are automatically assigned by nimble.
}
\examples{
\donttest{
set.seed(123)
n <- 200; d <- 4
theta <- c(2, 1, 1, 1); theta <- theta / sqrt(sum(theta^2))
f <- function(u) u^2 * exp(u)
sigma <- 0.5
X <- matrix(runif(n * d, -1, 1), nrow = n)
index_vals <- as.vector(X \%*\% theta)
y <- f(index_vals) + rnorm(n, 0, sigma)

# One tool version
fit <- bsSpike(X, y)

# Split version
models <- bsSpike(X, y, sampling = FALSE)
Ccompile <- compileModelAndMCMC(models)
mcmc.out <- runMCMC(Ccompile$mcmc, niter = 10000, nburnin = 1000, thin = 1,
                   nchains = 1, setSeed = TRUE, inits = models$input$init,
                   summary = TRUE, samplesAsCodaMCMC = TRUE)
}

}
\references{
Antoniadis, A., Grégoire, G., & McKeague, I. W. (2004).
Bayesian estimation in single-index models. \emph{Statistica Sinica}, 1147-1164.

Hornik, K., & Grün, B. (2014). movMF: an R package for fitting mixtures of von Mises-Fisher distributions.
\emph{Journal of Statistical Software}, 58, 1-31.

McGee, G., Wilson, A., Webster, T. F., & Coull, B. A. (2023).
Bayesian multiple index models for environmental mixtures.
\emph{Biometrics}, 79(1), 462-474.
}
