

#' @title    Draw the FROC  curves
#'@description     Draw an FROC  curves and an AFROC curves for user's specified modality and user's specified reader.
#' Using this function \strong{repeatedly}, we can draw the different reader and modality in a  \strong{same} plane simultaneously.
#' So, we can visualize the difference of modality (reader).
#'
#'    --------   To read the tables in Stan S4 class  ----------------------------
#'
#'
#'   * The  AUC denoted by AA[modalityID , readerID] are shown.
#'
#'   * The column of 2.5\% and 97.5\% means the lower and upper bounds of the 95% Credible Interval of AUCs.
#'
#'   * For example, AA[2,3] means the AUC of the 2 nd modality and the 3 rd reader.
#'

#'
#'
#'@param DrawFROCcurve TRUE or FALSE. Whether or not FROC curves are shown.
#'@param DrawAFROCcurve TRUE or FALSE. Whether or not AFROC curves are shown.
#'@param DrawCFPCTP TRUE or FALSE. Whether or not CFPCTP points are shown.
#'@param Draw.Flexible.upper_y TRUE or FALSE. Whether or not the upper bounds of vertical axis are determined automatically.
#'@param Draw.Flexible.lower_y TRUE or FALSE. Whether or not the lower bounds of vertical axis are determined automatically.
#'@param StanS4class This is an output of \code{rstan::stan()}.

#'
#'@param modalityID This is a vector indicating modalityID.
#'@param readerID  This is a vector indicating readerID.
#'
#'
#'@param Colour TRUE or FALSE. Colour of curves.
#'@param title TRUE or FALSE. If TRUE(default), then title of curves are drawn.


#'
#'@inheritParams fit_Bayesian_FROC
#'
#'@examples

# ####1#### ####2#### ####3#### ####4#### ####5#### ####6#### ####7#### ####8#### ####9####
#'#================The first example======================================
#' \donttest{

#' #1) Build the S4 class object by the following:
#'
#'
#'
#'   fit <- fit_Bayesian_FROC(BayesianFROC::dataList.Chakra.Web,ite=1111)
#'
#'
#' # The object "fit" is an S4 class object
#' # whose S4 class is an inherited class from stanfit in the rstan package.
#'
#' #<<Minor comments>>
#' #Note that return value "fit" is not an stanfit S4 object generated by rstan::stan(),
#' #but some inherited S4 class object which is an S4 object of
#' # some inherited S4 class from stanfit class. So, we can consider it as an object of
#' #an S4 class of rstan::stan().
#' #2) Now, we obtain the S4 class object named "fit".
#'   # Using this S4 class object, we draw the curves by:
#'
#'
#'
#'  DrawCurves(fit,modality = 1,reader = 4)
#'

#' #From this code, FROC curve for the first modality and fourth reader is drawn.
#' #3) By changing, e.g., the modality,
#'    #we can draw the curves for different  modalities.
#'    #This shows the comparison of modalites.
#'
#'
#'
#'  DrawCurves(fit,modality = 2,reader = 4)
#'  DrawCurves(fit,modality = 3,reader = 4)
#'
#'# Curves are overwritten for the sake of comparison.
#' #4) By repeat the running with respect to different modalities
#' #   in this manner, we can draw  AFROC (FROC) curves.
#'
#'
#'
#' #5) If you want to draw the FROC curves
#' #for reader ID =1,2,3,4 and modality ID =1,2, then the code is as follows;
#'
#'                   DrawCurves(
#'                             fit,
#'                             modalityID = c(1,2,3,4),
#'                             readerID = c(1,2)
#'                             )
#'# Each color of curves corresponds to the modality ID.
#'# So, even if curves are different readers and same modality, then color is same.
#'
#'
#'#6) If you want to see only data points, then by DrawFROCcurve = F, it will be done.
#'
#' DrawCurves(fit,
#'            DrawCFPCTP = TRUE,  # This implies data points are ploted.
#'            DrawFROCcurve = FALSE,  # From this, the curves are not drawn.
#'            modalityID = c(1,2,3,4),
#'            readerID  = c(1)
#'            )
#'
#'#7) If you use the plot in submission and it is not allowed to use color, then
#'#   by Colour =F, you can get black and white plots, e.g.,
#'
#'
#'DrawCurves(fit,
#'           DrawCFPCTP = TRUE,
#'           DrawFROCcurve = TRUE,
#'           modalityID = c(1,2,3,4),
#'           readerID  = c(1),
#'           Colour = FALSE    # From this, you can get plots without colors.
#'           )
#'
#'#'#8)  For AFROC, use DrawAFROCcurve = T
#'
#'DrawCurves(fit,
#'           DrawFROCcurve = FALSE,
#'           DrawAFROCcurve = TRUE,
#'           modalityID = c(1,2,3,4),
#'           readerID  = c(1))

#'#================The Second Example======================================

# ####1#### ####2#### ####3#### ####4#### ####5#### ####6#### ####7#### ####8#### ####9####
#'#This function is available in the case of single reader and single modality.
#'#The reason why the maintainer separate the fitting and drawing curves is, in MRMC case,
#'#It tooks a time to drawing, but in the single reader and single modality case, drawing
#'# the curve is very fast, so in fitting process the curves are also depicted, however
#'# by this function user can draw the FROC curves.


#'#First, we prepare the data endowed with this package.
#'
#'         dat  <- get(data("dataList.Chakra.1"))
#'
#'
#'
#'
#'#Second, we run the stan funtion
#'#with data named "dat"  and the author's Bayesian model.
#'
#'
#'          fit <-  fit_srsc_per_lesion(dat)
#'
#'# Drawing the curves by
#'
#'               DrawCurves(fit)
#'
#'
#'
#'
#'
#'
#'
#'
#'
#' #      Close the graphic device to avoid errors in R CMD check.
#'
#'          Close_all_graphic_devices()
#'}# dottest




#' @export
#'@inheritParams DrawCurves_MRMC_pairwise
#'@param indexCFPCTP TRUE of FALSE. If TRUE, then the cumulative false and hits are specified with its confidence level.
#'@param upper_x This is a upper bound for the axis of the horisontal coordinate of FROC curve.
#'@param upper_y This is a upper bound for the axis of the vertical coordinate of FROC curve.
#'
DrawCurves <- function(
  StanS4class,
  modalityID,
  readerID,
  title=TRUE,
  indexCFPCTP=FALSE,
  upper_x,
  upper_y,
  new.imaging.device=TRUE,
  Colour=TRUE,
  DrawFROCcurve=TRUE,
  DrawAFROCcurve=FALSE,
  DrawCFPCTP=TRUE,
  Draw.Flexible.upper_y=TRUE,
  Draw.Flexible.lower_y=TRUE
){

  fit <- StanS4class

  # if(grDevices::dev.cur() > 1) {
  #   message("\n* There are one more multiple graphics devices. I am afraid you confuse them. Please be carefull.")
  #   }
  if(fit@studyDesign=="srsc.per.image" || fit@studyDesign=="srsc.per.lesion"){

    DrawCurves_srsc(
      title=title,
      indexCFPCTP=indexCFPCTP,
      upper_x=upper_x,
      upper_y=upper_y,
      new.imaging.device=new.imaging.device,
      StanS4class=StanS4class,
      Drawcol = Colour,
      DrawFROCcurve=DrawFROCcurve,
      DrawAFROCcurve=DrawAFROCcurve,
      DrawCFPCTP=DrawCFPCTP
    )
  }

  if(fit@studyDesign=="MRMC"){


    if(missing(modalityID)){
      message("\n* WARNING:\n")
      message("\n* modalityID is missing, so we write the curve for the first modality only. If you want to write curves for, e.g., the first and the third modality, then it accomplishes by taking modalityID =c(1,3)")
      modalityID <-c(1)
      warning("* modalityID is missing, so we write the curve for the first modality only. If you want to write curves for, e.g., the first and the third modality, then it accomplishes by taking modalityID =c(1,3)")


    }

    if(missing(readerID)){
      message("*\n WARNING:\n")
      message("\n* readerID is missing, so we write the curve for the first readerID only. If you want to write curves for, e.g., the first and the third modality, then it accomplishes by taking readerID =c(1,3)")
      readerID <-c(1)
      warning("* readerID is missing, so we write the curve for the first readerID only. If you want to write curves for, e.g., the first and the third modality, then it accomplishes by taking readerID =c(1,3)")

    }


          DrawCurves_MRMC_pairwise (
            StanS4class=StanS4class,
            modalityID =modalityID,
            readerID=readerID,
            Colour=Colour,
            new.imaging.device = new.imaging.device,

            DrawFROCcurve=DrawFROCcurve,
            DrawAFROCcurve=DrawAFROCcurve,
            DrawCFPCTP=DrawCFPCTP,
            Draw.Flexible.upper_y=Draw.Flexible.upper_y,
            Draw.Flexible.lower_y=Draw.Flexible.lower_y
          )



  }



}
