% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/classPrior.R
\name{createPrior}
\alias{createPrior}
\title{Creates a standardized prior class}
\usage{
createPrior(density = NULL, sampler = NULL, lower = NULL,
  upper = NULL, best = NULL)
}
\arguments{
\item{density}{Prior density}

\item{sampler}{Sampling function for density (optional)}

\item{lower}{vector with lower bounds of parameters}

\item{upper}{vector with upper bounds of parameter}

\item{best}{vector with "best" parameter values}
}
\description{
Creates a standardized prior class
}
\details{
This is the general prior generator. It is highly recommended to not only implement the density, but also the sampler function. If this is not done, the user will have to provide explicit starting values for many of the MCMC samplers. Note the existing, more specialized prior function. If your prior can be created by those, they are preferred. Note also that priors can be created from an existing MCMC output from BT, or another MCMC sample, via \code{\link{createPriorDensity}}.
}
\note{
min and max truncate, but not re-normalize the prior density (so, if a pdf that integrated to one is truncated, the integral will in general be smaller than one). For MCMC sampling, this doesn't make a difference, but if absolute values of the prior density are a concern, one should provide a truncated density function for the prior.
}
\examples{
# Create a general prior distribution by specifying an arbitrary density function and a
# corresponding sampling function
density = function(par){
  d1 = dunif(par[1], -2,6, log =TRUE)
  d2 = dnorm(par[2], mean= 2, sd = 3, log =TRUE)
  return(d1 + d2)
}

# The sampling is optional but recommended because the MCMCs can generate automatic starting
# conditions if this is provided
sampler = function(n=1){
  d1 = runif(n, -2,6)
  d2 = rnorm(n, mean= 2, sd = 3)
  return(cbind(d1,d2))
}

prior <- createPrior(density = density, sampler = sampler, 
                     lower = c(-3,-3), upper = c(3,3), best = NULL)


# Use this prior in an MCMC 

ll <- function(x) sum(dnorm(x, log = TRUE)) # multivariate normal ll
bayesianSetup <- createBayesianSetup(likelihood = ll, prior = prior)

settings = list(iterations = 1000)
out <- runMCMC(bayesianSetup = bayesianSetup, settings = settings)

# see ?createPriorDensity for how to create a new prior from this output

}
\seealso{
\code{\link{createPriorDensity}} \cr
         \code{\link{createBetaPrior}} \cr
         \code{\link{createUniformPrior}} \cr
         \code{\link{createTruncatedNormalPrior}}\cr
         \code{\link{createBayesianSetup}}\cr
}
\author{
Florian Hartig
}
