% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trav_in_node.R
\name{trav_in_node}
\alias{trav_in_node}
\title{Traverse from one or more selected edges onto adjacent, inward nodes}
\usage{
trav_in_node(
  graph,
  conditions = NULL,
  copy_attrs_from = NULL,
  copy_attrs_as = NULL,
  agg = "sum"
)
}
\arguments{
\item{graph}{A graph object of class \code{dgr_graph}.}

\item{conditions}{An option to use filtering conditions for the traversal.}

\item{copy_attrs_from}{Providing an edge attribute name will copy those edge
attribute values to the traversed nodes. If the edge attribute already
exists, the values will be merged to the traversed nodes; otherwise, a new
node attribute will be created.}

\item{copy_attrs_as}{If an edge attribute name is provided in
\code{copy_attrs_from}, this option will allow the copied attribute values to be
written under a different node attribute name. If the attribute name
provided in \code{copy_attrs_as} does not exist in the graph's ndf, the new node
attribute will be created with the chosen name.}

\item{agg}{If an edge attribute is provided to \code{copy_attrs_from}, then an
aggregation function is required since there may be cases where multiple
edge attribute values will be passed onto the traversed node(s). To pass
only a single value, the following aggregation functions can be used:
\code{sum}, \code{min}, \code{max}, \code{mean}, or \code{median}.}
}
\value{
A graph object of class \code{dgr_graph}.
}
\description{
From a graph object of class \code{dgr_graph} with an active selection of edges
move with the edge direction to connected nodes, replacing the current edges
in the selection with those nodes traversed to. An optional filter by node
attribute can limit the set of nodes traversed to.
}
\details{
This traversal function makes use of an active selection of edges. After the
traversal, depending on the traversal conditions, there will either be a
selection of nodes or no selection at all.

Selections of edges can be performed using the following selection
(\verb{select_*()}) functions: \code{\link[=select_edges]{select_edges()}}, \code{\link[=select_last_edges_created]{select_last_edges_created()}},
\code{\link[=select_edges_by_edge_id]{select_edges_by_edge_id()}}, or \code{\link[=select_edges_by_node_id]{select_edges_by_node_id()}}.

Selections of edges can also be performed using the following traversal
(\verb{trav_*()}) functions: \code{\link[=trav_out_edge]{trav_out_edge()}}, \code{\link[=trav_in_edge]{trav_in_edge()}},
\code{\link[=trav_both_edge]{trav_both_edge()}}, or \code{\link[=trav_reverse_edge]{trav_reverse_edge()}}.
}
\examples{
# Set a seed
suppressWarnings(RNGversion("3.5.0"))
set.seed(23)

# Create a simple graph
graph <-
  create_graph() \%>\%
  add_n_nodes(
    n = 2,
    type = "a",
    label = c("asd", "iekd")) \%>\%
  add_n_nodes(
    n = 3,
    type = "b",
    label = c("idj", "edl", "ohd")) \%>\%
  add_edges_w_string(
    edges = "1->2 1->3 2->4 2->5 3->5",
    rel = c(NA, "A", "B", "C", "D"))

# Create a data frame with node ID values
# representing the graph edges (with `from`
# and `to` columns), and, a set of numeric values
df_edges <-
  data.frame(
    from = c(1, 1, 2, 2, 3),
    to = c(2, 3, 4, 5, 5),
    values = round(rnorm(5, 5), 2))

# Create a data frame with node ID values
# representing the graph nodes (with the `id`
# columns), and, a set of numeric values
df_nodes <-
  data.frame(
    id = 1:5,
    values = round(rnorm(5, 7), 2))

# Join the data frame to the graph's internal
# edge data frame (edf)
graph <-
  graph \%>\%
  join_edge_attrs(df = df_edges) \%>\%
  join_node_attrs(df = df_nodes)

# Show the graph's internal node data frame
graph \%>\% get_node_df()

# Show the graph's internal edge data frame
graph \%>\% get_edge_df()

# Perform a simple traversal from the
# edge `1`->`3` to the attached node
# in the direction of the edge; here, no
# conditions are placed on the nodes
# traversed to
graph \%>\%
  select_edges(
    from = 1,
    to = 3) \%>\%
  trav_in_node() \%>\%
  get_selection()

# Traverse from edges `2`->`5` and
# `3`->`5` to the attached node along
# the direction of the edge; both
# traversals lead to the same node
graph \%>\%
  select_edges(
    from = 2,
    to = 5) \%>\%
  select_edges(
    from = 3,
    to = 5) \%>\%
  trav_in_node() \%>\%
  get_selection()

# Traverse from the edge `1`->`3`
# to the attached node where the edge
# is incoming, this time filtering
# numeric values greater than `5.0` for
# the `values` node attribute
graph \%>\%
  select_edges(
    from = 1,
    to = 3) \%>\%
  trav_in_node(
    conditions = values > 5.0) \%>\%
  get_selection()

# Traverse from the edge `1`->`3`
# to the attached node where the edge
# is incoming, this time filtering
# numeric values less than `5.0` for
# the `values` node attribute (the
# condition is not met so the original
# selection of edge `1` -> `3` remains)
graph \%>\%
  select_edges(
    from = 1,
    to = 3) \%>\%
  trav_in_node(
    conditions = values < 5.0) \%>\%
  get_selection()

# Traverse from the edge `1`->`2` to
# the node `2` using multiple conditions
# with a single-length vector
graph \%>\%
  select_edges(
    from = 1,
    to = 2) \%>\%
  trav_in_node(
    conditions =
      grepl(".*d$", label) |
      values < 6.0) \%>\%
  get_selection()

# Create another simple graph to demonstrate
# copying of edge attribute values to traversed
# nodes
graph <-
  create_graph() \%>\%
  add_node() \%>\%
  select_nodes() \%>\%
  add_n_nodes_ws(
    n = 2,
    direction = "to") \%>\%
  clear_selection() \%>\%
  select_nodes_by_id(nodes = 2) \%>\%
  set_node_attrs_ws(
    node_attr = value,
    value = 8) \%>\%
  clear_selection() \%>\%
  select_edges_by_edge_id(edges = 1) \%>\%
  set_edge_attrs_ws(
    edge_attr = value,
    value = 5) \%>\%
  clear_selection() \%>\%
  select_edges_by_edge_id(edges = 2) \%>\%
  set_edge_attrs_ws(
    edge_attr = value,
    value = 5) \%>\%
  clear_selection() \%>\%
  select_edges()

# Show the graph's internal edge data frame
graph \%>\% get_edge_df()

# Show the graph's internal node data frame
graph \%>\% get_node_df()

# Perform a traversal from the edges to
# the central node (`1`) while also applying
# the edge attribute `value` to the node (in
# this case summing the `value` of 5 from
# both edges before adding as a node attribute)
graph <-
  graph \%>\%
  trav_in_node(
    copy_attrs_from = value,
    agg = "sum")

# Show the graph's internal node data frame
# after this change
graph \%>\% get_node_df()

}
