This package provides an R library to instrument prediction code that lets you capture inputs
to the model, predictions, prediction properties, and other metadata.
# Setup
* Install R
* You can use [R dependency management to your projects](https://rstudio.github.io/renv/articles/renv.html) to manage local r environments
* Use below commands to get started
```shell
make deps
make test
```
* If you are able to run `tests` means your local environment is properly setup

# Important make rules
Here are the available make rules that will help you in easing your development work
```shell
make all                 -> run check and clean
make clean               -> Remove intermediate files
make lint                -> Run lint
make test                -> Run test
make deps                -> Install dev dependencies
make install             -> Install package
make docs                -> Generate docs
make coverage            -> Run coverage
make check               -> Build as cran and run checks
make build               -> Run build
``` 

# Releasing a package
* For releasing and packaging related help please refer to [CRAN Releasing a package](https://r-pkgs.org/release.html)
* You need to run the below command from the `RStudio` console to push the release
```shell
devtools::release()
```
* The release process will prompt necessary checks, please read carefully and answer to continue

# How to create new environment in Domino
* Create a [personal access token](https://docs.github.com/en/authentication/keeping-your-account-and-data-secure/creating-a-personal-access-token)
* Make sure you enable `cerebrotech` `SSO` for the pat that you have generated by following the steps mentioned in the above link
* Use the above generated personal access token in the following commands
```docker
RUN R --no-save -e "install.packages(c('devtools'))"

RUN R --no-save -e "devtools::install_github('cerebrotech/r-prediction-logging', auth_token = '<github pat>')"
```

# How to use

```R
library("DominoPredictionLogging")
prediction_client <- PredictionClient(feature_names=c("min","max"),predict_names=c("prediction"))
predictionClient$record(c(1,100), c("2"))
```

## Example
```R
# This is a sample R model
# You can publish a model API by clicking on "Publish" and selecting
# "Model APIs" in your quick-start project.
 
# Load dependencies
library("jsonlite")
library("DominoPredictionLogging")
prediction_client <- PredictionClient(
    feature_names=c("min","max"),
    predict_names=c("prediction")
)
 
# Define a function to create an API
# To call model use: {"data": {"min": 1, "max": 100}}
my_model <- function(min, max) {
  random_number <- runif(1, min, max)
  predictionClient$record(c(min,max), c(random_number))
  return(list(number=random_number))
}
```

