# Copyright 2022 DARWIN EU (C)
#
# This file is part of DrugUtilisation
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

#' This function is used to summarise the dose table over multiple cohorts.
#'
#' @param cohort Cohort with drug use variables and strata
#' @param cdm cdm_reference generated by CDMConnector
#' @param strata Stratification list
#' @param drugUseVariables Name of columns with drug use variables
#' @param drugUseEstimates Estimates that we want for the columns
#' @param minCellCount Below this number counts will be supressed
#'
#' @return A summary of the drug use stratified by cohort_name and strata_name
#'
#' @export
#'
#' @examples
#' \donttest{
#' library(DrugUtilisation)
#' library(PatientProfiles)
#' library(CodelistGenerator)
#'
#' cdm <- mockDrugUtilisation()
#' cdm <- generateDrugUtilisationCohortSet(
#'   cdm, "dus_cohort", getDrugIngredientCodes(cdm, "acetaminophen")
#' )
#' cdm$dus_cohort <- cdm$dus_cohort %>%
#'   addDrugUse(cdm, 1125315)
#' result <- summariseDrugUse(cdm$dus_cohort, cdm)
#' print(result)
#'
#' cdm$dus_cohort <- cdm$dus_cohort %>%
#'   addSex(cdm) %>%
#'   addAge(cdm, ageGroup = list("<40" = c(0, 30), ">40" = c(40, 150)))
#' result <- summariseDrugUse(
#'   cdm$dus_cohort, cdm, strata = list(
#'    "age_group" = "age_group", "sex" = "sex",
#'    "age_group and sex" = c("age_group", "sex")
#'   )
#' )
#' print(result)
#' }
#'
summariseDrugUse<- function(cohort,
                            cdm,
                            strata = list(),
                            drugUseVariables = drugUseColumns(cohort),
                            drugUseEstimates = c("median", "q25", "q75"),
                            minCellCount = 5) {
  # check inputs
  checkInputs(
    cohort = cohort, cdm = cdm, strata = strata,
    drugUseVariables = drugUseVariables, drugUseEstimates = drugUseEstimates,
    minCellCount = minCellCount
  )

  # update cohort_names
  cohort <- cohort %>%
    dplyr::left_join(
      CDMConnector::cohortSet(cohort), by = "cohort_definition_id", copy = TRUE
    )

  # summarise drug use columns
  result <- PatientProfiles::summariseResult(
    table = cohort, group = list("cohort_name" = "cohort_name"),
    strata = strata, variables = list(numericVariables = drugUseVariables),
    functions = list(numericVariables = drugUseEstimates),
    minCellCount = minCellCount
  ) %>%
    dplyr::mutate(
      cdm_name = dplyr::coalesce(CDMConnector::cdmName(cdm), as.character(NA)),
      result_type = "Summary drug use"
    )

  return(result)
}

#' Obtain automatically the drug use columns
#'
#' @param cohort A cohort
#'
#' @return Name of the drug use columns
#'
#' @export
#'
#' @examples
#' \donttest{
#' library(DrugUtilisation)
#' library(CodelistGenerator)
#'
#' cdm <- mockDrugUtilisation()
#' acetaminophen <- getDrugIngredientCodes(cdm, "acetaminophen")
#' cdm <- generateDrugUtilisationCohortSet(
#'   cdm, "dus", acetaminophen
#' )
#' cdm$dus <- cdm$dus %>%
#'   addDrugUse(cdm, 1125315, acetaminophen)
#' drugUseColumns(cdm$dus)
#' }
#'
drugUseColumns <- function(cohort) {
  checkInputs(cohort = cohort)
  names <- colnames(cohort) [colnames(cohort) %in% c(
    "initial_daily_dose", "number_exposures", "duration", "cumulative_dose",
    "number_eras", "initial_quantity", "cumulative_quantity"
  )]
  return(names)
}
