\encoding{UTF-8}

\name{normalize}

\alias{normalize}

\title{Relative Coordinates and Density Reduction for TLS Point Clouds}

\description{This function obtains coordinatesrelative to the plot centre for Terrestrial Laser Scanner (TLS) point clouds (supplied as LAS files) derived from single scans (with TLS scanner point as plot centre). In addition, the point cropping process developed by Molina-Valero et al., (2019) is applied as a criterion for reducing point density homogeneously in space and proportionally to object size.}

\usage{
normalize(las,
          x.center = NULL, y.center = NULL,
          max.dist = NULL, min.height = NULL, max.height = NULL,
          algorithm.dtm = "tin", res.dtm = 0.2,
          csf = list(cloth_resolution = 0.5),
          multiple.scans = NULL,
          id = NULL, file = NULL,
          dir.data = NULL, save.result = TRUE, dir.result = NULL)}

\arguments{

  \item{las}{Character string containing the name of LAS file belonging to TLS point cloud, including .las extension (see \sQuote{Examples}). Planimetric coordinates of point cloud data must be in local, representing TLS scan point the origin with Cartesian coordinates x and y as (0, 0).}

  \item{x.center}{Planimetric x center coordinate of point cloud data.}

  \item{y.center}{Planimetric y center coordinate of point cloud data.}

  \item{max.dist}{Optional maximum horizontal distance (m) considered from the plot centre. All points farther than \code{max.dist} will be discarded after the normalization process. If this argument is not specified by the user, it will be set to NULL by default and, as a consequence, all points will be used in processing, with \code{max.dist} representing the farthest point.}

  \item{min.height}{Optional minimum height (m) considered from ground level. All points below \code{min.height} will be discarded after the normalization process. If this argument is not specified by the user, it will be set to NULL by default and, as a consequence, all points will be used in processing, with \code{min.height} representing the lowest point.}

  \item{max.height}{Optional maximum height (m) considered from ground level. All points above \code{max.height} will be discarded after the normalization process. If this argument is not specified by the user, it will be set to NULL by default and, as a consequence, all points will be used in processing, with \code{max.height} representing the highest point.}

    \item{algorithm.dtm}{Algorithm used to generate the digital terrain model (DTM) from the TLS point cloud. There are two posible options based on spatial interpolation: \sQuote{tin} and \sQuote{knnidw} (see \sQuote{Details}). If this argument is not specified by the user, it will be set to \sQuote{tin} algorithm.}

      \item{res.dtm}{numeric. Resolution of the DTM generated to normalize point cloud (see \sQuote{Details}). If this argument is not specified by the user, it will be set to 0.2 m.}

      \item{csf}{List containing parameters of CSF algorithm:}

\itemize{

  \item {\code{cloth_resolution}}: by default 0.5.}

  \item{multiple.scans}{Logical parameter indicating single scans (NULL) or multiple scans (TRUE).}


  \item{id}{Optional plot identification encoded as character string or numeric. If this argument is not specified by the user, it will be set to NULL by default and, as a consequence, the plot will be encoded as 1.}

  \item{file}{Optional file name identification encoded as character string or numeric value. If it is null, file will be encoded as \code{id} by default.}

  \item{dir.data}{Optional character string naming the absolute path of the directory where LAS files containing TLS point clouds are located. \code{.Platform$file.sep} must be used as the path separator in \code{dir.data}. If this argument is not specified by the user, it will be set to \code{NULL} by default and, as a consequence, the current working directory of the \R process will be assigned to \code{dir.data} during the execution.}

  \item{save.result}{Optional logical which indicates whether or not the output files described in \sQuote{Output Files} section must be saved in \code{dir.result}. If this argument is not specified by the user, it will be set to \code{TRUE} by default and, as consequence, the output files will be saved.}

  \item{dir.result}{Optional character string naming the absolute path of an existing directory where files described in \sQuote{Output Files} section will be saved. \code{.Platform$file.sep} must be used as the path separator in \code{dir.result}. If this argument is not specified by the user, and \code{save.result} is \code{TRUE}, it will be set to \code{NULL} by default and, as a consequence, the current working directory of the \R process will be assigned to \code{dir.result} during the execution.}}

\details{

Relative coordinates are obtained by means of a normalization process, generating a digital terrain model (DTM) from the TLS point cloud, with the ground height set at 0 m. The DTM is generated by spatial interpolation of ground points classified with the CSF algorithm (Zhang et al., (2016)). Two algorithms are available for that purpose: (i) spatial interpolation based on a Delaunay triangulation, which performs a linear interpolation within each triangle (\sQuote{tin}); (ii) spatial interpolation using a k-nearest neighbour (KNN) approach with inverse-distance weighting (IDW) (\sQuote{knnidw}). Note that normalization process is based on \CRANpkg{lidR} package functions: \code{\link{classify_ground}}, \code{\link{grid_terrain}} and \code{\link{normalize_height}}.

The point cropping process reduces the point cloud density proportionally to the likelihood that objects will receive points according to their distance from TLS and their size, which is determined by angle aperture (the farther they are, the lower the density is). The result is an approximately homogeneous point cloud in three-dimensional space (for more details see Molina-Valero et al., (2019)).}

\value{Data frame of normalized point cloud including the following columns (each row corresponds to one point):

\item{id}{Plot identification encoded as a character string or numeric in the argument \code{id}.}

\item{file}{File name identification encoded as character string or numeric, corresponding to the normalized and reduced point clouds saved. This coincides with the TXT file in the absolute path specified in \code{dir.result} (if \code{save.result} is set to \code{TRUE}).}

\item{Coordinates}{

  Cartesian (according to \url{https://en.wikipedia.org/wiki/Cartesian_coordinate_system} notation):
    \itemize{
      \item \code{x}: x axis distance (m).
      \item \code{y}: y axis distance (m).
      \item \code{z}: height (m).
    }

  Cylindrical (according to \url{https://en.wikipedia.org/wiki/Cylindrical_coordinate_system} notation):
    \itemize{
      \item \code{rho}: horizontal distance (m).
      \item \code{phi}: angular coordinate (rad).
      \item \code{z}: height (m).
    }

  Spherical (according to \url{https://en.wikipedia.org/wiki/Spherical_coordinate_system} notation):
    \itemize{
      \item \code{r}: radial distance (m).
      \item \code{theta}: polar angle (rad).
      \item \code{phi}: azimuthal angle (rad)
    }}

\item{slope}{Slope of the terrain (rad).}

\item{prob}{selection probability assigned in point cropping process (0-1]. Only the farthest will have probability of 1.}

\item{prob.select}{final selection probability assigned in point cropping process. Selected (1) and discarded point (0).}}

\section{Output Files}{

At the end of the normalization process, if the \code{save.result} argument is \code{TRUE}, the function will save the reduced point cloud as TXT file and encoded according to \code{file} argument. The format is the same as data frame described in \sQuote{Value}, except for a \code{prob} column, which is removed because all points selected after the point cropping process have a final selection probability of 1. The data frame is written without row names in \code{dir.result} directory using the \code{\link[vroom]{vroom_write}} function in the \CRANpkg{vroom} package.}

\note{
Note that \code{max.dist}, \code{min.height} and \code{max.height} arguments may be useful for optimizing computing time as well as for removing unnecessary and/or outlier points. These values may be selected more appropriately when inventory data are already available, or the user has some knowledge about autoecology of scanned tree species.

Note also that the linear interpolation algorithm (\sQuote{tin} in this package) showed the highest accuracy in Liang et al., (2018) in DTM generation with single-scans. In this work a DTM resolution of 0.2 m was also considered adequately for square plots of 32 x 32 m.}

\references{
Liang, X., Hyyppä, J., Kaartinen, H., Lehtomäki, M., Pyörälä, J., Pfeifer, N., ... & Wang, Y. (2018). International benchmarking of terrestrial laser scanning approaches for forest inventories. \emph{ISPRS journal of photogrammetry and remote sensing}, \bold{144}, 137-179. \doi{https://doi.org/10.1016/j.isprsjprs.2018.06.021}

Molina-Valero J. A., Ginzo-Villamayor M. J., Novo Pérez M. A., Álvarez-González J. G., & Pérez-Cruzado C. (2019). Estimación del área basimétrica en masas maduras de Pinus sylvestris en base a una única medición del escáner laser terrestre (TLS). \emph{Cuadernos de la Sociedad Espanola de Ciencias Forestales}, \bold{45(3)}, 97-116. \doi{https://doi.org/10.31167/csecfv0i45.19887}.

Zhang, W., Qi, J., Wan, P., Wang, H., Xie, D., Wang, X., & Yan, G. (2016). An easy-to-use airborne LiDAR data filtering method based on cloth simulation. \emph{Remote Sensing}, \bold{8(6)}, 501. \doi{https://doi.org/10.3390/rs8060501}.}

\seealso{
\code{\link{tree.detection}}, \code{\link{tree.detection.multiple}}.
}

\author{
Juan Alberto Molina-Valero,
María José Ginzo Villamayor,
Manuel Antonio Novo Pérez,
Adela Martínez-Calvo,
Juan Gabriel Álvarez-González,
Fernando Montes and
César Pérez-Cruzado.}

\examples{

\donttest{

# Establishment of working directories (optional)
# By default here we propose the current working directory of the R process

dir.data <- getwd()
dir.result <- getwd()


# Loading example data (LAS file) to dir.data

download.file("https://www.dropbox.com/s/2c3d320o3srcawb/1.las?raw=1",
              destfile = file.path(dir.data, "1.las"),
              method = "wininet", mode = "wb")


# Normalizing the whole point cloud data without considering arguments

pcd <- normalize(las = "1.las", dir.data = dir.data, dir.result = dir.result)


# Using all the arguments

pcd <- normalize(las = "1.las",
                 max.dist = 15, min.height = 0.25, max.height = 25,
                 id = "1", file = "1.txt",
                 dir.data = dir.data, dir.result = dir.result)

  }

}
