#' @export
#' @title Structural Function
#' @aliases structural_function
#' @description A structured function is a kind of kinked (piecewise) function generated by connecting two
#' functions through a transition region.
#' This function calculates the value of a structured function.
#' @param theta the track switching parameter, which is a scalar.
#' @param transition.interval a 2-vector.
#' @param f1 the first function (or a value).
#' @param f2 the second function (or a value).
#' @param ... parameters of f1 and f2.
#' @return The value of the structural function.
#' @examples
#' x <- seq(1, 5, 0.1)
#' y <- c()
#' for (theta in x) y <- c(y, structural_function(theta, c(2, 3), log, sqrt, theta))
#' plot(x, y)
#' lines(x, log(x), col = "blue")
#' lines(x, sqrt(x), col = "red")
#'
#' ####
#' f <- function(theta) {
#'   p <- c(1, 1)
#'   structural_function(
#'     theta,
#'     c(15, 20),
#'     function(p) CD_A(alpha = 5, Beta = c(0.6, 0.4), p),
#'     function(p) CD_A(alpha = 15, Beta = c(0.3, 0.7), p),
#'     p
#'   )
#' }
#'
#' tmp <- sapply(1:25, f)
#' matplot(t(tmp), type = "l")

structural_function <- function(theta, transition.interval, f1, f2, ...) {
  if (is.function(f1)) value1 <- f1(...) else value1 <- f1
  if (is.function(f2)) value2 <- f2(...) else value2 <- f2

  if (theta <= transition.interval[1]) {
    return(value1)
  }
  if (theta >= transition.interval[2]) {
    return(value2)
  }
  result <- (transition.interval[2] - theta) / diff(transition.interval) * value1 +
    (theta - transition.interval[1]) / diff(transition.interval) * value2
  return(result)
}
