%-#########################################################################################
%- This program is distributed in the hope that it will be useful,
%- but WITHOUT ANY WARRANTY; without even the implied warranty of
%- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
%- GNU General Public License for more details.
%-#########################################################################################

\name{GenSA}
\alias{GenSA}
\title{
    Generalized Simulated Annealing Function
}
\description{
    Function which searchs for global minimum of a very complex non-linear objective function with a very large number of optima. 
}

\usage{
    GenSA(par, fn, lower, upper, control=list(), ...)
}

\details{
    The default values of the control components are set for a complex optimization problem.
        For usual optimization problem with medium complexity, GenSA can find a reasonable solution quickly so
        the user is recommended to let GenSA stop earlier by setting \code{threshold.stop} if \code{threshold.stop} is the expected
        function value, or by setting \code{max.time} if the user just want to run GenSA for \code{max.time} seconds, or
        by setting \code{max.call} if the user just want to run GenSA within \code{max.call} function calls. Please refer to the examples
        below. For very complex optimization problems, the user is recommended to increase \code{maxit} and \code{temp}.
}
\arguments{
    \item{par}{
        Vector. Initial values for the components to be optimized. Default is \code{NULL}, in which case, default values will be generated automatically.
    }
    \item{fn}{
        A function to be minimized, with first argument the vector of parameters over which minimization is to take place. It should return
            a scalar result.
    }
    \item{lower}{
        Vector with length of \code{par}. Lower bounds for components.
    }
    \item{upper}{
        Vector with length of \code{par}. Upper bounds for components.
    }
    \item{...}{
        allows the user to pass additional arguments to the function \code{fn}.
    }
    \item{control}{
        The argument is a list that can be used to control the behavior of the algorithm:
            \describe{
                \item{\code{maxit}}{
                    Integer. Maximum number of iterations of the algorithm. 
                }
                \item{\code{threshold.stop}}{
                    Numeric. The program will stop when the expected objective function value \code{threshold.stop} is reached. Default value is \code{NULL}
                }
                \item{\code{nb.stop.improvement}}{
                    Integer. The program will stop when there is no any improvement in \code{nb.stop.improvement} steps. 
                }
                \item{\code{smooth}}{
                    Logical.\code{TRUE} when the objective function is smooth, or differentiable almost everywhere in the
                        region of \code{par}, \code{FALSE} otherwise. Default value is \code{TRUE}.
                }
                \item{\code{max.call}}{
                    Integer. Maximum number of call of the objective function. Default is set to 1e7.
                }
                \item{\code{max.time}}{
                    Numeric. Maximum running time in seconds.
                }
                \item{\code{temperature}}{
                    Numeric. Initial value for temperature.
                }
                \item{\code{visiting.param}}{
                    Numeric. Parameter for visiting distribution.
                }
                \item{\code{acceptance.param}}{
                    Numeric. Parameter for acceptance distribution.
                }
                \item{\code{verbose}}{
                    Logical. \code{TRUE} means that messages from the algorithm are shown. Default is \code{FALSE}.
                }
                \item{\code{simple.function}}{
                    Logical. \code{FALSE} means that the objective function has only a few local minima. Default is \code{FALSE} which means that the objective function is complicated with many local minima.
                }
                \item{\code{trace.fn}}{
                    Character string. The trace matrix will be written in the file \code{trace.fn}. Default is \code{NULL} which means that the trace matrix will be stored in the returned value rather than in a file.
                }
            }
    }
}


\value{
    The returned value is a list with the following fields:
        \describe{
            \item{value:}{
                Numeric. The value of \code{fn} corresponding to \code{par}.
            }
            \item{par:}{
                Vector. The best set of parameters found.
            }
            \item{trace.mat:}{
                A matrix which contains the history of the algorithm. (By columns: Step number, temperature,
                        current objective function value, current minimal objective function value).
            }
            \item{counts:}{
                Integer. Total number of calls of the objective function.
            }
        }
}

\keyword{ package, Generalized Simulated Annealing, optimization }
\seealso{
    optim
}

\references{
    Xiang Y, Gubian S, Suomela B, Hoeng. "Generalized Simulated Annealing for Global Optimization: the GenSA Package". \href{http://journal.r-project.org/archive/2013-1/xiang-gubian-suomela-etal.pdf}{The R Journal, 5(1):13-29, June 2013}.

        Tsallis C (1988). "Possible generalization of Boltzmann-Gibbs statistics." Journal of Statistical Physics, 52, 479--487.

        Tsallis C, Stariolo DA (1996). "Generalized Simulated Annealing." Physica A, 233, 395--406.

        Xiang Y, Sun DY, Fan W, Gong XG (1997). "Generalized Simulated Annealing Algorithm and Its Application to the Thomson Model." Physics Letters A, 233, 216--220.

        Xiang Y, Gong XG (2000a). "Efficiency of Generalized Simulated Annealing." PHYSICAL REVIEW E, 62, 4473.

        Xiang Y, Sun DY, Gong XG (2000). "Generalized Simulated Annealing Studies on Structures and Properties of Nin (n=2-55) Clusters." Journal of Physical Chemistry A, 104, 2746--2751.
}


\examples{
    library(GenSA)
# Try Rastrgin function (The objective function value for global minimum
# is 0 with all components of par are 0.)
        Rastrigin <- function(x) {
            sum(x^2 - 10 * cos(2 * pi  * x)) + 10 * length(x)
        }
# Perform the search on a 30 dimensions rastrigin function. Rastrigin
# function with dimension 30 is known as the most
# difficult optimization problem according to "Yao X, Liu Y, Lin G (1999).
# \Evolutionary Programming Made Faster."
# IEEE Transactions on Evolutionary Computation, 3(2), 82-102.

# GenSA will stop after finding the targeted function value 0 with
# absolute tolerance 1e-13
    set.seed(1234) # The user can use any seed.
        dimension <- 30
        global.min <- 0
        tol <- 1e-13
        lower <- rep(-5.12, dimension)
        upper <- rep(5.12, dimension)
        out <- GenSA(lower = lower, upper = upper, fn = Rastrigin,
                control=list(threshold.stop=global.min+tol,verbose=TRUE))
        out[c("value","par","counts")]

# GenSA will stop after running for about 2 seconds
# Note: The time for solving this problem by GenSA may vary
# depending on the computer used. 
        set.seed(1234) # The user can use any seed.
        dimension <- 30
        global.min <- 0
        tol <- 1e-13
        lower <- rep(-5.12, dimension)
        upper <- rep(5.12, dimension)
        out <- GenSA(lower = lower, upper = upper, fn = Rastrigin,
                control=list(max.time=2))
        out[c("value","par","counts")]
}


\author{
    Yang Xiang, Sylvain Gubian, Brian Suomela, Julia Hoeng, PMP SA.
}


