package de.fhg.iais.kd.djm;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Vector;

import org.apache.log4j.Logger;

import com.trilead.ssh2.Session;
import com.trilead.ssh2.StreamGobbler;

public class ClientConnection extends Thread {

	// private static Logger logger;
	private static Logger logger = Logger.getLogger(ClientConnection.class);
	private Socket connectedSocket;
	private DataOutputStream out;
	private BufferedWriter bw;
	public enum Mode {
		execFile, execCondorJob, execGlobusJob,  checkAllResults, checkResult, getJobs, stop, restart, deleteFiles, deleteJob, stopJob, restartJob, uploadFiles, downloadFiles
	};

	public ClientConnection(Socket connectedSocket) {
		this.connectedSocket = connectedSocket;
	}

	public void notNull(Object param, String name) throws XMLFormatException {
		if (param == null) {
			try{
				bw.write("<" + name + "> not specified\n");
				bw.flush();
				bw.close();
			}catch(IOException e) {};
			throw new XMLFormatException("<" + name + "> not specified");
		}
	}

	public void run() {

		try {
			
			// Get input from the client
			PerformDocument performDoc = new PerformDocument();
			
			DataInputStream in = new DataInputStream(connectedSocket.getInputStream());
			out = new DataOutputStream(connectedSocket.getOutputStream());
			bw = new BufferedWriter(new OutputStreamWriter(out));

			PerformDocParser parser = new PerformDocParser();
				
			try{
				performDoc = parser.parseInputDoc(in);
			}
			catch(Exception e)//parsing exception
			{
				logger.error(e.getLocalizedMessage());
				bw.write("Error: " +e.getLocalizedMessage()+ "\n");
				bw.flush();
				bw.close();
				return;
	    		
			}

			
			//needed tags for all cases
			notNull(performDoc.getUsername(), "username");
			notNull(performDoc.getMode(), "mode");

			Mode mode = performDoc.getMode();
			if(mode==null)
				return;
			
			switch (mode) {
				// stop the server
				case stop: {
					logger.debug("stopping Server");
					DistributedJobManagement.shutdown = true;
					break;
				}
					// restart the server
				case restart: {
					logger.debug("restarting Server");
					// TODO not everyone should be able to restart the server
					DistributedJobManagement.sshConnections = new HashMap<String[], SSHClient>(); 	//mapping ip,username to SShClient
					DistributedJobManagement.jobs = new HashMap<Integer, JobDescription>();			//mapping jobID to JobDescription
					DistributedJobManagement.userJobs = new HashMap<String, ArrayList<Integer>>();	//mapping user to his jobs
					break;
				}
					// check result
				case checkAllResults: {
					checkAllResults(performDoc);
					break;
	
				}
				case checkResult: {
					checkResult(performDoc);
					break;
				}
				case execFile: {
					execFile(performDoc);
					break;
				}
				case execCondorJob: {
					execCondorJob(performDoc);
					
					break;
				}
				case execGlobusJob:{
					execGlobusJob(performDoc);
					break;
				}
				
				case getJobs: {
					getJobs(performDoc);
					break;
				}
				case deleteFiles: {
					deleteFiles(performDoc);
					break;
				}
				case deleteJob: {
					deleteJob(performDoc);
					break;
				}
				case stopJob: {
					stopJob(performDoc);
					break;
				}
				case restartJob: {
					restartJob(performDoc);
					break;
				}
				case uploadFiles: {
					//send ok thus client starts sending the files
					out.writeChars("ok\n");
					out.flush();
					FileTransfer.downloadFilesTo(performDoc.getInputFiles(), performDoc.getRemoteDir(), in);
					break;
				}
				case downloadFiles: {
					FileTransfer.setFileSizesAndSend(out, performDoc.getInputFiles());
					FileTransfer.uploadFiles(out, performDoc.getInputFiles(), performDoc.getUsername());
					break;
				}
				// case :

			}// end switch
			if(connectedSocket!=null)
				connectedSocket.close();
		} catch (XMLFormatException e)
		{
			logger.info(e.getMessage());
		} catch (IOException ioe) {
			logger.error("IOException in ClientConnection: " + ioe);
		//	ioe.printStackTrace();
			try {
				bw.write("Error: IOException in ClientConnection: " + ioe.getLocalizedMessage() + "\n");
				bw.flush();
				bw.close();
				 connectedSocket.close();
			} catch (IOException e) {}
			 
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	private void restartJob(PerformDocument performDoc) throws Exception {
		if(performDoc.getId()==-1)
		{
			logger.error("no id specified in restartJob");
			return;
		}
				
		logger.debug("restart job "+ performDoc.getId());
		 
		 //get Job
		 int jobID = performDoc.getId();
		 JobDescription job = DistributedJobManagement.jobs.get(jobID);
		 if(job==null)
		 {
			 logger.error("no Jobdescription found for id " + jobID);
			 return;
		 }
		
		//create an new SshClient or load an old one if available
		SSHClient sshClient = createSshClient(performDoc.getKey(), job.getUsername(), job.getExecIp());
			
			if(sshClient==null){
				String err = "Error, no sshKey added, and key at ~/.ssh/id_rsa doesnt exists";
				bw.write(err +"\n");
				bw.flush();
				bw.close();
				throw new IOException(err);
			}
			
			//delete old job
			
			if(!stopJob(performDoc)){
				return;
			}
			
			//copy entries from job to performDoc
			
			performDoc.setExecutable(job.getExecutable());
			performDoc.setLocalVar(job.getLocalVar());
			performDoc.setExecIp(job.getExecIp());
			performDoc.setArguments(job.getArguments());
			performDoc.setRemoteDir(job.getRemoteDir());
			performDoc.setUsername(job.getUsername());
			performDoc.setOutputFiles(job.getOutputFiles());
			performDoc.setInputFiles(job.getInputFiles());
			performDoc.setToDelete(job.getToDelete());
			
		 switch(job.getJobType()){
		 case condorJob:{
			execCondorJob(performDoc, jobID);
			 break;
		 }
		 case globusJob:{
			 execGlobusJob(performDoc, jobID);
			 break;
		 }
		 case fileExecution:{
			execFile(performDoc, jobID);
			 break;
		 }
		 }
		
	}

	private boolean stopJob(PerformDocument performDoc) throws Exception {
		
		if(performDoc.getId()==-1)
		{
			logger.info("no id specified in stopJob");
			return false;
		}
				
		logger.debug("stop job "+ performDoc.getId());
		 
		 //get Job
		 int jobID = performDoc.getId();
		 JobDescription job = DistributedJobManagement.jobs.get(jobID);
		 if(job==null)
		 {
			 logger.error("no Jobdescription found for id " + jobID);
			 return false;
		 }
		 Thread thread = job.getThreadId();
		 thread.stop();
		
		//create an new SshClient or load an old one if available
		SSHClient sshClient = createSshClient(performDoc.getKey(), job.getUsername(), job.getExecIp());
			
			if(sshClient==null){
				String err = "Error, no sshKey added, and key at ~/.ssh/id_rsa doesnt exists";
				bw.write(err +"\n");
				bw.flush();
				bw.close();
				throw new IOException(err);
			}
			
		 switch(job.getJobType()){
		 case condorJob:{
			 if(job.getCondorId()!=-1)
				 sshClient.send("condor_rm " + job.getCondorId() + "\n"); 
			 else
				 logger.error("cannot stop job, because no condor cluster id is saved");
			 break;
		 }
		 case globusJob:{
			 if(job.getGlobusId()==null)
				 logger.error("cannot stop job, because no globus job id is saved");
			 else
				 sshClient.send("cd " + job.getRemoteDir() + " && globusrun-ws -kill -j "  + job.getGlobusEprFile() + "\n");
			 break;
		 }
		 case fileExecution:{
			 if(job.getArguments().startsWith("CMD BATCH --vanilla ")){
				String[] processes =sshClient.send("ps aux | grep \"R -f " + job.getArguments().substring(20) +"\"").split("\n");
				for(int i=0;i<processes.length; i++){
					//if grep doesnt fount itself
					if(!processes[i].contains("grep")){
						if(!processes[i].contains(job.getArguments().substring(20))){
							//something is wrong, line doesnt contains the search string
							logger.error("cannot stop Job, Return code: " + processes[i]);
							return false;
						}else {
							String[] entries =processes[i].split(" +");
							if(entries.length<11){
								//too few elements in the string, must be an error
								logger.error("cannot stop Job, Return code is " + processes[i]);
								return false;
							}else {
								//kill the job
								String ret = sshClient.send("kill " + entries[1]);
								if(ret.length()>0){
									logger.error("cannot kill the process in stop Job, Return code is " + ret);
									return false;
								}
							}
						}
					}
				}
			 }
				
			 else{
				 logger.error("cannot stop job, because job has file execuition mode and is not a R CMD BATCH job");
				 return false;
			 }
			 
			 break;
		 }
		 }
		 
		 deleteJob(performDoc);
		 return true;
	}

	private void execGlobusJob(PerformDocument performDoc) throws IOException, XMLFormatException {
			int id = DistributedJobManagement.getNewID();
			execGlobusJob(performDoc, id);
	}
	private void execGlobusJob(PerformDocument performDoc, int id) throws IOException, XMLFormatException {
		logger.debug("mode: execGlobusJob");
		 notNull(performDoc.getExecIp(), "execIp");
		 notNull(performDoc.getExecutable(), "executable");
		 notNull(performDoc.getArguments(), "arguments");
		 notNull(performDoc.getRemoteDir(), "remoteDir");

		//create an new SshClient or load an old one if available
		SSHClient sshClient = createSshClient(performDoc.getKey(), performDoc.getUsername(), performDoc.getExecIp());
		
		if(sshClient==null){
			String err = "Error, no sshKey added, and key at ~/.ssh/id_rsa doesnt exists";
			bw.write(err +"\n");
			bw.flush();
			bw.close();
			throw new IOException(err);
		}

		try {
			JobDescription job = new JobDescription(JobDescription.JobType.globusJob, 
					JobDescription.Status.waitingForInputFiles, performDoc.getLocalVar(), performDoc
							.getExecutable(),
							performDoc.getExecIp(),
					performDoc.getArguments(), performDoc
							.getRemoteDir(), performDoc.getUsername(),
					performDoc.getOutputFiles(), performDoc.getInputFiles(), Thread.currentThread());
			// add job to running jobs
			DistributedJobManagement.jobs.put(id, job);

			// add id to userJobs mapping
			synchronized (DistributedJobManagement.userJobs) {
				if (DistributedJobManagement.userJobs
						.containsKey(performDoc.getUsername())) {
					ArrayList<Integer> ids = DistributedJobManagement.userJobs
							.get(performDoc.getUsername());
					ids.add(id);
				} else {
					ArrayList<Integer> ids = new ArrayList<Integer>();
					ids.add(id);
					DistributedJobManagement.userJobs.put(performDoc
							.getUsername(), ids);
				}
			}

			// return ID to client

			 logger.debug("created new Job with id:" + id +
			 "\nrunning jobs: "+ DistributedJobManagement.jobs.size());
			 bw.write(id+ "\n");
			 bw.flush();
			 if(!performDoc.isBlock())
			 {
				 bw.close();
				 connectedSocket.close();
			 }

			
			//wait until all files are copied
			waitForInputFiles(performDoc, sshClient, job);

			String globusJobFile=  "globus" + System.currentTimeMillis();
			String globusStdoutFile=  "globus" + System.currentTimeMillis() + ".stdout";
			String globusStderrFile=  "globus" + System.currentTimeMillis() + ".stderr";
			String globusEpr = System.currentTimeMillis() + ".epr";
			
			createGlobusJobFile(job.getExecutable(), job.getArguments(), performDoc.getRemoteDir(), globusStdoutFile, globusStderrFile, job.getInputFiles(), job.getOutputFiles(), globusJobFile, sshClient, performDoc.getRemoteDir());
			
			//add these files  to delete them afterwards
			ArrayList<String> toDelete = job.getToDelete();
			toDelete.add(globusJobFile);
			toDelete.add(globusStdoutFile);
			toDelete.add(globusStderrFile);
			toDelete.add(globusEpr);
			for( String file : performDoc.getToDelete())
				toDelete.add(file);
			job.setToDelete(toDelete);

			// execute command on execIp
			
			Session sess = sshClient.getConnection().openSession();
			 
			job.setGlobusEprFile(globusEpr);
			
			String msg = "cd " + performDoc.getRemoteDir() + " && globusrun-ws -submit -S -f " + globusJobFile + " -o " + globusEpr;
			sess.execCommand(msg);
			InputStream stdout = new StreamGobbler(sess.getStdout());
			InputStream stderr = new StreamGobbler(sess.getStderr());

			BufferedReader br = new BufferedReader(new InputStreamReader(stdout));
			BufferedReader stderrReader = new BufferedReader(new InputStreamReader(
					stderr));

			logger.debug("Globus output:");
			String result="";
			while (true) {
				String line = stderrReader.readLine();
				if (line == null)
					break;
				else{
					result+=line+"\n";
					if(line.contains("Current job state: Active")){
						job.setStatus(JobDescription.Status.submitted);
						logger.debug("job " + id + " submitted");
					}
					else if(line.contains("Job ID: ")){
						//Job ID: uuid:d17c2852-c6aa-11dd-bde1-000475ac6435
						try{
							String globusId = line.substring(line.lastIndexOf(" "));
							//save GlobusId
							job.setGlobusId(globusId);
						}catch (Exception e){
							logger.error("cannot extract Globus job id from line: " + line);
						}
					}
					else if(line.contains("Submitting job...Failed.") || line.contains("error")){
						logger.error("Globus job failed");
						String err= line;
						line="";
						while((line = stderrReader.readLine())!=null){
							err += "   " + line;
						}
						job.setStatus(JobDescription.Status.errorNew);
						job.setError(err);
						if (performDoc.isBlock()) {
							bw.write("Job had an error: " + err + "\n");
							bw.flush();
							bw.close();
						}
						connectedSocket.close();
						sess.close();
						return;
					}
					else
						logger.debug(line);
				}
			}

			while (true) {
				String line = br.readLine();
				if (line == null)
					break;
				result+="Error: following line on stdout: " +line+"\n";
			}
		 
			sess.close();
			
			String res = waitForOutputFileSetSizesAndId(sshClient, job, id, globusStderrFile);
			if(res!="")
			{
				//found an error
				job.setStatus(JobDescription.Status.errorNew);
				logger.error("job " + id + " has an error\n" + res);
				job.setError(res);
				
				if (performDoc.isBlock()) {
					bw.write("Job had an error: " + result.replaceAll("\n", " ") + "\n");
					bw.flush();
					bw.close();
				}
			}
			
			else if(!result.contains("Current job state: Done") || result.contains("Error") || result.contains("ERROR"))
			{
				//Error 
				job.setStatus(JobDescription.Status.errorNew);
				job.setError("Error: " + result.replaceAll("\n", " "));
				logger.debug("job " + id + " has error: " + result);
				
				if (performDoc.isBlock()) {
					bw.write("Job had an error: " + result.replaceAll("\n", " ") + "\n" );
					bw.flush();
					bw.close();
				}

			}
			else
			{
				job.setStatus(JobDescription.Status.done);
				logger.debug("job " + id + " has finished");
				
				if (performDoc.isBlock()) {
					String send = "";
					for (int i = 0; i < performDoc.getOutputFiles().size(); i++)
						send = send + performDoc.getOutputFiles().get(i).getFilename() + "\n";

					bw.write(send);
					bw.flush();
					bw.close();
				}
			}
		} catch (Exception e) {
			logger.error("Error: cannot connect to host " + performDoc.getExecIp());
			e.printStackTrace();
			bw.write("Error: cannot connect to host " + performDoc.getExecIp() + "\n");
			 bw.flush();
			 bw.close();
			 connectedSocket.close();
		}
		
	}

	private void deleteFiles(PerformDocument performDoc) throws IOException {
		if(performDoc.getId()==-1)
		{
			logger.info("no id specified in deleteFiles");
			return;
		}
				
		logger.debug("delete files of job "+ performDoc.getId());
		 
		 //get Job
		 int jobID = performDoc.getId();
		 JobDescription job = DistributedJobManagement.jobs.get(jobID);
		 if(job==null)
		 {
			 logger.error("no Jobdescription found for id " + jobID);
			 return;
		 }
		//create an new SshClient or load an old one if available
		SSHClient sshClient = createSshClient(performDoc.getKey(), job.getUsername(), job.getExecIp());
			
		if(sshClient==null){
			String err = "Error, no sshKey added, and key at ~/.ssh/id_rsa doesnt exists";
			throw new IOException(err);
		}
		
		 try{
			 
			 if(!performDoc.getUsername().equals(job.getUsername()))
			 {
				logger.info("People != owner of job ("+ jobID +" ) are not allowed to delete the job\n");
				return;
			 }
			 //delete input and output Files on the server
			 for(OutputFile file : job.getOutputFiles())
			 {
				 if(!file.getFilename().startsWith("gsiftp://") && !file.getFilename().startsWith("file://"))
				 {
					 logger.debug("delete output file "+ file.getFilename());
					 String res=sshClient.send("cd " + job.getRemoteDir() + " && rm "+ new File(file.getFilename()).getName());
				 	 if(res.length()>0)
				 		 logger.error("Error: " + res);
				 }
			 }
			 for(InputFile file : job.getInputFiles())
			 {
				 if(!file.getFilename().startsWith("gsiftp://") && !file.getFilename().startsWith("file://"))
				 {
					 logger.debug("delete input file "+ file.getFilename());
					 String res=sshClient.send("cd " + job.getRemoteDir() + " && rm "+ new File(file.getFilename()).getName());
				 	 if(res.length()>0)
				 		 logger.error("Error: " + res);
				 }
			 }
			 for(String file : job.getToDelete()){
				 logger.debug("delete generated file "+ file);
				 String res=sshClient.send("cd " + job.getRemoteDir() + " && rm " + file);
			 	 if(res.length()>0)
			 		 logger.error("Error: " + res);
			 }
		 }catch(Exception e)
		 {
			 logger.error("cannot delete files on server:\n" + e.getLocalizedMessage());
		 }
		// delete job in hashmap
		DistributedJobManagement.jobs.remove(jobID);
		// delete id in userJobs
		ArrayList<Integer> ids = DistributedJobManagement.userJobs.get(performDoc.getUsername());
		if (!ids.isEmpty())
			ids.remove((Object) jobID);
		DistributedJobManagement.userJobs.put(performDoc.getUsername(), ids);
		
	}
	private void deleteJob(PerformDocument performDoc){
		if(performDoc.getId()==-1)
		{
			logger.info("no id specified in deleteFiles");
			return;
		}
				
		logger.debug("delete files of job "+ performDoc.getId());
		 
		 //get Job
		 int jobID = performDoc.getId();
		 JobDescription job = DistributedJobManagement.jobs.get(jobID);
		 if(job==null)
		 {
			 logger.error("no Jobdescription found for id " + jobID);
			 return;
		 }
		// delete job in hashmap
			DistributedJobManagement.jobs.remove(jobID);
			// delete id in userJobs
			ArrayList<Integer> ids = DistributedJobManagement.userJobs.get(performDoc.getUsername());
			if (!ids.isEmpty())
				ids.remove((Object) jobID);
			DistributedJobManagement.userJobs.put(performDoc.getUsername(), ids);
			
	}

	private void getJobs(PerformDocument performDoc) throws IOException {
		String ownerJobs = "";
		ArrayList<Integer> ids = DistributedJobManagement.userJobs.get(performDoc.getUsername());
		if(ids!=null){
			for (int i = 0; i < ids.size(); i++) {
				int id = ids.get(i);
				JobDescription jd = DistributedJobManagement.jobs.get(id);
				ownerJobs = ownerJobs + id + " " + jd.getLocalVar() + "\n";
			}
		}
		else
			ownerJobs="\n";
		System.err.println(ownerJobs);
		bw.flush();
		bw.close();
		
	}

	private void execCondorJob(PerformDocument performDoc) throws IOException, XMLFormatException {
		int id = DistributedJobManagement.getNewID();
		execCondorJob(performDoc, id);
	}
	
	private void execCondorJob(PerformDocument performDoc, int id) throws IOException, XMLFormatException {
		logger.debug("mode: execCondorJob");
		 notNull(performDoc.getExecIp(), "execIp");
		 notNull(performDoc.getExecutable(), "executable");
		 notNull(performDoc.getArguments(), "arguments");
		 notNull(performDoc.getRemoteDir(), "remoteDir");

		if (performDoc.getOutputFiles().size() == 0) {
			bw.write("<outputFile> not specified\n");
			bw.flush();
			bw.close();
			throw new IOException("<outputFile> not specified");
		}
		//create an new SshClient or load an old one if available
		SSHClient sshClient = createSshClient(performDoc.getKey(), performDoc.getUsername(), performDoc.getExecIp());
		if(sshClient==null){
			String err = "Error, no sshKey added, and key at ~/.ssh/id_rsa doesnt exists";
			bw.write(err +"\n");
			bw.flush();
			bw.close();
			throw new IOException(err);
		}

		try {	
			JobDescription job = new JobDescription(JobDescription.JobType.condorJob,
					JobDescription.Status.waitingForInputFiles, performDoc.getLocalVar(), performDoc
							.getExecutable(),
							performDoc.getExecIp(),
					performDoc.getArguments(), performDoc
							.getRemoteDir(), performDoc.getUsername(),
					performDoc.getOutputFiles(), performDoc.getInputFiles(), Thread.currentThread());
			// add job to running jobs
			DistributedJobManagement.jobs.put(id, job);

			// add id to userJobs mapping
			synchronized (DistributedJobManagement.userJobs) {
				if (DistributedJobManagement.userJobs
						.containsKey(performDoc.getUsername())) {
					ArrayList<Integer> ids = DistributedJobManagement.userJobs
							.get(performDoc.getUsername());
					ids.add(id);
				} else {
					ArrayList<Integer> ids = new ArrayList<Integer>();
					ids.add(id);
					DistributedJobManagement.userJobs.put(performDoc
							.getUsername(), ids);
				}
			}

			// return ID to client

			 logger.debug("created new Job with id:" + id +
			 "\nrunning jobs: "+ DistributedJobManagement.jobs.size());
			 bw.write(id+ "\n");
			 bw.flush();
			 if(!performDoc.isBlock())
			 {
				 bw.close();
				 connectedSocket.close();
			 }

			
			//wait until all files are copied
			waitForInputFiles(performDoc, sshClient, job);
			
			job.setStatus(JobDescription.Status.submitted);
			logger.debug("job " + id + " submitted");

			String condorErrorFile=  "condor" + System.currentTimeMillis() +".err";
			String condorFile =  "condor" + System.currentTimeMillis();
			job.setJobFile(condorFile);


			//add these files  to delete them afterwards
			ArrayList<String> toDelete = job.getToDelete();
			toDelete.add(condorFile);
			toDelete.add(condorErrorFile);
			for( String file : performDoc.getToDelete())
				toDelete.add(file);
			job.setToDelete(toDelete);
			
			createCondorJobFile(job.getExecutable(), job.getArguments(), condorErrorFile, job.getInputFiles(), condorFile, sshClient, job.getRemoteDir());
			
			// execute command on execIp
			String result = sshClient.send("cd " + performDoc.getRemoteDir() + " && condor_submit " + condorFile);
			//Submitting job(s).
			//1 job(s) submitted to cluster 526.
			if(!result.contains(" submitted to cluster "))
			{
				//Error 
				job.setStatus(JobDescription.Status.errorNew);
				job.setError(result.replaceAll("\n", " "));
				logger.debug("job " + id + " has error: " + job.getError() + "|");
				if (performDoc.isBlock()) {
					bw.write("Job had an error: " + job.getError() + "\n" );
					bw.flush();
					bw.close();
				}

			}
			else
			{
				try{
					int clusterId = Integer.valueOf(result.substring(result.lastIndexOf(" ")+1, result.length()-2));
					job.setCondorId(clusterId);
				} catch(Exception e){
					logger.error("cannot extract Condor cluster id");
				}
				String res = waitForOutputFileSetSizesAndId(sshClient, job, id, condorErrorFile);
				if(res!="")
				{
					//found an error
					job.setStatus(JobDescription.Status.errorNew);
					logger.error("job " + id + " has an error\n" + res);
					job.setError(res);
					
					if (performDoc.isBlock()) {
						bw.write("Job had an error: " + result.replaceAll("\n", " ") + "\n");
						bw.flush();
						bw.close();
					}
				}
				else
				{
					job.setStatus(JobDescription.Status.done);
					logger.debug("job " + id + " has finished");
					
					if (performDoc.isBlock()) {
						String send = "";
						for (int i = 0; i < performDoc.getOutputFiles().size(); i++)
							send = send + performDoc.getOutputFiles().get(i).getFilename() + "\n";

						bw.write(send);
						bw.flush();
						bw.close();
					}
				}
			}
		} catch (Exception e) {
			logger.error("Error: cannot connect to host " + performDoc.getExecIp());
			e.printStackTrace();
			bw.write("Error: cannot connect to host " + performDoc.getExecIp() + "\n");
			 bw.flush();
			 bw.close();
			 connectedSocket.close();
		}
		
	}
	private void execFile(PerformDocument performDoc) throws IOException, XMLFormatException {
		int id = DistributedJobManagement.getNewID();
		execFile(performDoc, id);
	}
	private void execFile(PerformDocument performDoc, int jobId) throws IOException, XMLFormatException {
		logger.debug("mode: exec File");
		 notNull(performDoc.getExecIp(), "execIp");
		 notNull(performDoc.getExecutable(), "executable");
		 notNull(performDoc.getArguments(), "arguments");
		 notNull(performDoc.getRemoteDir(), "remoteDir");

		if (performDoc.getOutputFiles().size() == 0) {
			bw.write("<outputFile> not specified\n");
			bw.flush();
			bw.close();
			throw new IOException("<outputFile> not specified");
		}
		//create an new SshClient or load an old one if available
		SSHClient sshClient = createSshClient(performDoc.getKey(), performDoc.getUsername(), performDoc.getExecIp());
		
		if(sshClient==null){
			String err = "Error, no sshKey added, and key at ~/.ssh/id_rsa doesnt exists";
			bw.write(err +"\n");
			bw.flush();
			bw.close();
			throw new IOException(err);
		}

		try {
			
			JobDescription job = new JobDescription(JobDescription.JobType.fileExecution,
					JobDescription.Status.waitingForInputFiles, performDoc.getLocalVar(), performDoc
							.getExecutable(),
							performDoc.getExecIp(),
					performDoc.getArguments(), performDoc
							.getRemoteDir(), performDoc.getUsername(),
					performDoc.getOutputFiles(), performDoc.getInputFiles(), Thread.currentThread());
			// add job to running jobs
			DistributedJobManagement.jobs.put(jobId, job);

			// add id to userJobs mapping
			synchronized (DistributedJobManagement.userJobs) {
				if (DistributedJobManagement.userJobs
						.containsKey(performDoc.getUsername())) {
					ArrayList<Integer> ids = DistributedJobManagement.userJobs
							.get(performDoc.getUsername());
					ids.add(jobId);
				} else {
					ArrayList<Integer> ids = new ArrayList<Integer>();
					ids.add(jobId);
					DistributedJobManagement.userJobs.put(performDoc
							.getUsername(), ids);
				}
			}

			// return ID to client
			 logger.debug("created new Job with id:" + jobId +
			 "running jobs: "+ DistributedJobManagement.jobs.size());
			 bw.write(jobId+ "\n");
			 bw.flush();
			 if(!performDoc.isBlock())
			 {
				 bw.close();
				 connectedSocket.close();
			 }
		
			
			//wait until all files are copied to execution host
			waitForInputFiles(performDoc, sshClient, job);
			
			job.setStatus(JobDescription.Status.submitted);
			logger.debug("job " + jobId + " submitted");
			// execute command on execIp
			String result = sshClient.send("cd " + performDoc.getRemoteDir() + " && " + performDoc.getExecutable()
					+ " " + performDoc.getArguments());

			if(result!="")
			{
				//Error 
				job.setStatus(JobDescription.Status.errorNew);
				job.setError(result);
				logger.debug("job " + jobId + " has error: " + result.replaceAll("\n", " "));
				
				if (performDoc.isBlock()) {
					bw.write("Job had an error: " + result.replaceAll("\n", " ") + "\n");
					bw.flush();
					bw.close();
				}

			}
			else
			{
			
				waitForOutputFileSetSizesAndId(sshClient, job, jobId, null);
				
				job.setStatus(JobDescription.Status.done);
				logger.debug("job " + jobId + " has finished");
				
				
				if (performDoc.isBlock()) {
					String send = "";
					for (int i = 0; i < performDoc.getOutputFiles().size(); i++)
						send = send + performDoc.getOutputFiles().get(i).getFilename() + "\n";

					bw.write(send);
					bw.flush();
					bw.close();
				}
			}

		} catch (Exception e) {
			logger.error("Error: cannot connect to host " + performDoc.getExecIp());
			e.printStackTrace();
			bw.write("Error: cannot connect to host " + performDoc.getExecIp() + "\n");
			 bw.flush();
			 bw.close();
			 connectedSocket.close();
		}
		
	}

	private void checkResult(PerformDocument performDoc) throws IOException, XMLFormatException {
		notNull(performDoc.getId(), "id");
		
		  logger.debug("check result of id: " + performDoc.getId());
		  JobDescription jd = DistributedJobManagement.jobs.get(performDoc.getId());
		  
		  if(jd==null) { 
			  bw.write("no job found\n"); 
			  bw.flush();
		  	  bw.close(); 
		  	  return; 
		  }
		  if(!jd.getUsername().equals(performDoc.getUsername()))
		  {
			  bw.write("you are not the owner of this job, aborting\n"); 
			  bw.flush();
		  	  bw.close();
		  }
		  logger.debug(jd.status);
		  
		  switch(jd.status){
		  				  
			  case done:
			  {
				  logger.debug("job " + performDoc.getId() + " done"); 
				  ArrayList<OutputFile> outputFiles = jd.outputFiles; 
				  if(outputFiles.size()==0) { 
					  bw.write("job done, no output added\n"); 
					  bw.flush(); 
					  bw.close(); 
					  return; 
				  }
				  String send="job done\n"; 
				  for(int i=0;i<outputFiles.size();i++)
					  send = send + outputFiles.get(i).convertToString() + "\n";
				  bw.write(send); 
				  bw.flush(); 
				  bw.close();
				  break;
			  }
			  case submitted: 
			  {
				  bw.write("submitted\n");
				  bw.flush();
			  bw.close(); 
			  break;
			  }
			  case error:
			  {
				  bw.write("ERROR: Job had an error : " + jd.getError()+ "\n");
				  bw.flush(); 
				  bw.close(); 
				  break; 
			  }
			  case errorNew:
			  {
				  jd.status = JobDescription.Status.error;
				  bw.write("ERROR: Job had an error : " + jd.getError()+ "\n");
				  bw.flush(); 
				  bw.close(); 
				  break; 
			  }
		  }
		
	}

	private void checkAllResults(PerformDocument performDoc) throws IOException {
		logger.debug("checking result. Jobs in database: " + DistributedJobManagement.jobs.size());
		String jobsDone = "";
		synchronized (DistributedJobManagement.userJobs) {
			ArrayList<Integer> userIds = DistributedJobManagement.userJobs.get(performDoc.getUsername());
			if (userIds == null)
				return;
			for (Integer jobID : userIds) {
				// for each Job of this user check if status is done
				JobDescription jd = DistributedJobManagement.jobs.get(jobID);
				if (jd.status.equals(JobDescription.Status.done)) {
					jd.status=JobDescription.Status.uploadResult;
//					logger.debug("job " + jobID + " done");
					ArrayList<OutputFile> outputFiles = jd.outputFiles;
					for (int i = 0; i < outputFiles.size(); i++)
						jobsDone += outputFiles.get(i).convertToString() + "\n";
				}
				else if (jd.status.equals(JobDescription.Status.errorNew)) {
					jd.status=JobDescription.Status.error;
					bw.write("Job had Error: "+ jd.getError()+ " Job ID: " + jobID + "\n");
					logger.error("Job had Error: "+ jd.getError()+ " Job ID: " + jobID + "\n");
					bw.flush();
					return;
				}
			}
			
			/*	//remove last &
			if(jobsDone.length()>0 && jobsDone.contains("&"))
				jobsDone=jobsDone.substring(0,jobsDone.length()-2);*/
			bw.write(jobsDone);
			if(jobsDone!="")
				logger.debug("Output files: " +jobsDone);
			bw.flush();
			bw.close();
			
			
		}
		
	}
	/**
	 * 
	 * @param executable
	 * @param arguments
	 * @param remDir
	 * @param stdout
	 * @param stderr
	 * @param inputFiles
	 * @param outputFiles
	 * @param globusJobFile
	 * @param sshClient
	 * @throws Exception
	 */
	
	private void createGlobusJobFile(String executable, String arguments, String remDir, String stdout, String stderr, ArrayList<InputFile> inputFiles, ArrayList<OutputFile> outputFiles, String globusJobFile, SSHClient sshClient, String remoteDir) throws Exception {
		logger.debug("creating remote file "+globusJobFile);
		String[] argsSplit = arguments.split(" ");
		String job ="<job>\n"+
		"<executable>"+executable+"</executable>\n"+
		"<directory>"+remDir+"</directory>\n";
		for(int i=0;i<argsSplit.length;i++)
			job= job + "<argument>" + argsSplit[i] + "</argument>\n";
		job = job + "<stdout>"+stdout+"</stdout>\n"+
		"<stderr>"+stderr+"</stderr>\n";
		/*
		if(inputFiles.size()!=0){
			job = job + "<fileStageIn>\n";
			for(int i=0;i<inputFiles.size();i++){
				job=job + "<transfer>\n"+
				"<sourceUrl>";	
				String filename = inputFiles.get(i).getFilename();
				if(filename.startsWith("file:/") || filename.startsWith("gsiftp:/"))
					job=job.concat(filename);
				else
					job=job.concat("file://" +filename);
				job=job + "</sourceUrl>\n"+
				"<destinationUrl>file://"+remDir + "/" + new File(inputFiles.get(i).getFilename()).getName()+"</destinationUrl> \n"+
				"</transfer>\n";
			}
			job = job + "</fileStageIn>\n";
		}
		if(outputFiles.size()!=0){
			job = job + "<fileStageOut>\n";
			for(int i=0;i<outputFiles.size();i++){
				job=job + "<transfer>\n"+
				"<sourceUrl>" +
				"file://"+remDir + "/" + new File(outputFiles.get(i).getFilename()).getName()+ 
				"</sourceUrl>\n"+
				"<destinationUrl>";
				String filename = outputFiles.get(i).getFilename();
				if(filename.startsWith("file:/") || filename.startsWith("gsiftp:/"))
					job=job.concat(filename);
				else
					job=job.concat("file://" +filename);
				job=job + "</destinationUrl> \n"+
				"</transfer>\n";
			}
			job = job + "</fileStageOut>\n";
		}
		if(inputFiles.size()!=0){
			job = job + "<fileCleanUp>\n";
			for(int i=0;i<inputFiles.size();i++){
				job=job + "<deletion>\n<file>\n"+
				"file://"+remDir + "/" + new File(inputFiles.get(i).getFilename()).getName()+
				"</file>\n</deletion>\n";
			}
			//delete GlobusJobFile	
			job = job + "<deletion>\n<file>\nfile://"+ globusJobFile+"</file>\n</deletion>\n";
			job = job + "</fileCleanUp>\n";
		}
		*/
		
		job = job + "</job>\n";

		String result = sshClient.send("mkdir -p " + remoteDir + " && cd " + remoteDir + " && echo \"" + job + "\" > " + globusJobFile + "\n");
//		logger.debug("echo \"" + job + "\" > " + condorFile + "\n");
		if(result.length()!=0)
		{
			logger.error("Error, cannot create GlobusJob File: \n" + result);
		}
	}

	/**
	 * creates a condor Job File on execution host
	 * @param executable
	 * @param arguments
	 * @param errFile
	 * @param inputFiles
	 * @param condorFile 
	 * @param sshClient
	 * @throws Exception 
	 */
	private void createCondorJobFile(String executable, String arguments, String errFile, ArrayList<InputFile> inputFiles, String condorFile, SSHClient sshClient, String remDir) throws Exception {
		logger.debug("creating remote file "+condorFile);
		
		String job ="Executable     = "+ executable +
				"\nUniverse       = vanilla\n"+
				"should_transfer_files = YES\n"+
				"when_to_transfer_output = ON_EXIT\n"+
				"arguments      = "+ arguments +"\n"+
				"Error          = "+ errFile + "\n"+
				"transfer_input_files =";
		for(int i=0;i<inputFiles.size();i++){
			job=job.concat(inputFiles.get(i).getFilename());
			if(i<inputFiles.size()-1)
				job=job.concat(", ");
		}
		job=job.concat("\nQueue\n");
				
		String result = sshClient.send("cd " + remDir + " && echo \"" + job + "\" > " + condorFile + "\n");
//		logger.debug("echo \"" + job + "\" > " + condorFile + "\n");
		if(result.length()!=0)
		{
			logger.error("Error, cannot create Condor File: \n" + result);
		}
	}

	/**
	 * creates new SSHClient if none exists for that user and ip
	 * @param key
	 * @param username
	 * @param Ip
	 * @return SSHClient
	 */
	private SSHClient createSshClient(String key, String username, String Ip) {
		//create new ssh connection if key exists
		SSHClient sshClient=null;
		if(!DistributedJobManagement.sshConnections.containsKey(new String[]{Ip, username}))
		{
			// checking ssh key
			if (key == null) {
				String home = System.getenv("HOME");
				if (home != null){
					String rsa = home + "/.ssh/id_rsa";
					String dsa = home + "/.ssh/id_dsa";
					if(new File(rsa).exists())
						key = rsa;
					else if(new File(dsa).exists())
						key = dsa;
					else
						key="";
						
				}
				else
					key = "";
			}
			if(!key.equals(""))
			{
				logger.debug("specified or default key is availible");
				// get MySsh connection for host
				sshClient = DistributedJobManagement.sshConnections
						.get(new String[]{Ip, username});
				if (sshClient == null) {
					// if none exists, create a new one
					sshClient = new SSHClient(Ip, username, key);
					sshClient.connect();
					DistributedJobManagement.sshConnections.put(new String[]{Ip, username}, sshClient);
				}
			}
			else
			{
				logger.debug("no key found, aborting");
				return null;
			}
		}
		else
		{
			logger.debug("found sshClient in database");
			sshClient=DistributedJobManagement.sshConnections.get(new String[]{Ip, username});
		}
		return sshClient;
	}

	/**
	 * 
	 * @param sshClient
	 * @param job
	 * @param id
	 * @param condorErrFile if not null, this file is checked remotely if size is =0, otherwise an error occured
	 * @throws Exception
	 */
	private String waitForOutputFileSetSizesAndId(SSHClient sshClient, JobDescription job, int id, String condorErrFile) throws Exception {
		//TODO optimize, dont send an ssh command for every file
		for(OutputFile outputFile : job.getOutputFiles())
		{
			outputFile.setId(id);
			boolean outputFound = false;
			boolean errorOccured = false;
			while(!outputFound)
			{
				String[] stdout = sshClient.send("ls -l "+ job.getRemoteDir()).split("\n");
				for(String line : stdout){
					if(line.contains(new File(outputFile.getFilename()).getName())){						
						String[] split = line.split(" +");//split at one or more " "
						outputFound=true;
						try{
							outputFile.setSize(Integer.parseInt(split[4]));
							break;
						}catch (Exception e){
							throw new IOException("ls -l returns wrong output on serverside \nreturned string is:"+line + "\n");
						}
					}
					//check if condorErrFile has size 0
					else if(condorErrFile!=null && line.contains(condorErrFile)){
						String[] split = line.split(" +");//split at one or more " "
						try{
							if(Integer.parseInt(split[4])!=0)
							{
								//error
								errorOccured=true;
								break;
							}
						}catch (Exception e){
							throw new IOException("ls -l returns wrong output on serverside \nreturned string is:"+line + "\n");
						}
					}
					
				}
				if(errorOccured)
				{
					String result = sshClient.send("cat " + condorErrFile +"\n" );
					logger.error("Condor error occured: " + result);
					return result;
				}
				logger.debug("waiting for output file " + outputFile.getFilename()+ " " + id);
				Thread.sleep(1000);
			}
		}
		return "";
	}

	/**
	 * waits until all files are available on serverside, ie tests if actual size (with s -l on server) is the same as the expected size in performDoc
	 * @param performDoc
	 * @param job 
	 * @throws IOException if ls-l returns something wrong
	 */
	private void waitForInputFiles(PerformDocument performDoc, SSHClient sshClient, JobDescription job) throws Exception{
		if(performDoc.getInputFiles().size()==0)
			return;
		boolean allFinished=false;
		boolean[] finished = new boolean[performDoc.getInputFiles().size()];
		for(int i=0;i<finished.length;i++)
			finished[i]=false;
		
		while(!allFinished)
		{
			for(int i=0;i<performDoc.getInputFiles().size();i++){
				if(finished[i]==false)
				{
					InputFile inputFile = performDoc.getInputFiles().get(i);
					if(inputFile.getSize()==0)
						finished[i]=true;
					else{
						String stdout = sshClient.send("cd " + job.getRemoteDir() + " && ls -l "+ inputFile.getFilename());
						String size;
						if(stdout.contains("Error")){
							size="-1";
						}
						else {
							String[] split = stdout.split(" ");
							size = split[4];
						}
						try{
							
							//now check is file has the right size
							if(Integer.parseInt(size) == inputFile.getSize())
							{
								//System.out.println("file: "+ inputFile.getFilename() + " size: "+ size + " needed size: "+ inputFile.getSize());
								finished[i]=true;
								
							}
							else
								logger.debug("File "+ performDoc.getInputFiles().get(i).filename + " has size on server: "+ size + " needed size: "+ inputFile.getSize());
						}catch (Exception e){
							throw new IOException("ls -l returns wrong output on serverside \nreturned string is:"+stdout);
						}
					}
					//check if all finished
					allFinished=true;
					for(int j=0;j<finished.length;j++)
						if(finished[j]==false)
							allFinished=false;
				}
			}
			if(allFinished==false)
				Thread.sleep(1000);
		}
	}

}
