#' Extract results, conduct posterior inference and compute performance metrics for MCMC samples of models from the IMIFA family
#'
#' This function post-processes simulations generated by \code{\link{mcmc_IMIFA}} for any of the IMIFA family of models. It can be re-ran at little computational cost in order to extract different models explored by the sampler used for \code{sims}, without having to re-run the model itself. New results objects using different numbers of clusters and different numbers of factors (if visited by the model in question), or using different model selection criteria (if necessary) can be generated with ease. The function also performs post-hoc corrections for label switching, as well as post-hoc Procrustes rotation, to ensure sensible posterior parameter estimates and constructs credible intervals.
#' @param sims An object of class "\code{IMIFA}" generated by \code{\link{mcmc_IMIFA}}.
#' @param burnin Optional additional number of iterations to discard. Defaults to 0.
#' @param thinning Optional interval for extra thinning to be applied. Defaults to 1.
#' @param G If this argument is not specified, results will be returned with the optimal number of clusters. If different numbers of clusters were explored in \code{sims} for the "\code{MFA}" or "\code{MIFA}" methods, supplying an integer value allows pulling out a specific solution with \code{G} clusters, even if the solution is sub-optimal. Similarly, this allows retrieval of samples corresponding to a solution, if visited, with \code{G} clusters for the "\code{OMFA}", "\code{OMIFA}", "\code{IMFA}" and "\code{IMIFA}" methods.
#' @param Q If this argument is non specified, results will be returned with the optimal number of factors. If different numbers of factors were explored in \code{sims} for the "\code{FA}", "\code{MFA}", "\code{OMFA}" or "\code{IMFA}" methods, this allows pulling out a specific solution with \code{Q} factors, even if the solution is sub-optimal. Similarly, this allows retrieval of samples corresponding to a solution, if visited, with \code{Q} factors for the "\code{IFA}", "\code{MIFA}", "\code{OMIFA}" and "\code{IMIFA}" methods.
#' @param criterion The criterion to use for model selection, if model selection is required by the method in \code{sims}. Note that these are \emph{all} calculated, this argument merely indicates which one will form the basis of the construction of the output.
#' @param G.meth If the object in \code{sims} arises from the "\code{OMFA}", "\code{OMIFA}", "\code{IMFA}" or "\code{IMIFA}" methods, this argument determines whether the optimal number of clusters is given by the mode or median of the posterior distribution of \code{G}. Defaults to "\code{Mode}".
#' @param Q.meth If the object in \code{sims} arises from the "\code{IFA}", "\code{MIFA}", "\code{OMIFA}" or "\code{IMIFA}" methods, this argument determines whether the optimal number of latent factors is given by the mode or median of the posterior distribution of \code{Q}. Defaults to "\code{Mode}".
#' @param dat The actual data set on which \code{\link{mcmc_IMIFA}} was originally run. This is necessary for computing error metrics between the estimated and empirical covariance matrix/matrices. If this is not supplied, the function will attempt to find the data set if it is still available in the global environment.
#' @param conf.level The confidence level to be used throughout for credible intervals for all parameters of inferential interest. Defaults to 0.95.
#' @param zlabels For any method that performs clustering, the true labels can be supplied if they are known in order to compute clustering performance metrics. This also has the effect of ordering the MAP labels (and thus the ordering of cluster-specific parameters) to most closely correspond to the true labels if supplied.
#'
#' @return An object of class "\code{Results_IMIFA}" to be passed to \code{\link{plot.Results_IMIFA}} for visualising results. Dedicated \code{print} and \code{summary} functions exist for objects of this class. The object is a list of lists, some of the most important components of which are:
#' \describe{
#' \item{Clust}{Everything pertaining to clustering performance can be found here for all but the "\code{FA}" and "\code{IFA}" methods. More detail is given if \code{zlabels}} are supplied.
#' \item{Error}{Error metrics (e.g. MSE) between the empirical and estimated covariance matrix/matrices.}
#' \item{GQ.results}{Everything pertaining to model choice can be found here, incl. posterior summaries for the estimated number of groups and estimated number of factors, if applicable to the method employed. Information criterion values are also accessible here.}
#' \item{Means}{Posterior summaries for the means.}
#' \item{Loadings}{Posterior summaries for the factor loadings matrix/matrices.}
#' \item{Scores}{Posterior summaries for the latent factor scores.}
#' \item{Uniquenesses}{Posterior summaries for the uniquenesses.}
#' }
#' @export
#' @importFrom Rfast "med" "rowMaxs" "standardise" "colMaxs" "rowVars" "rowmeans" "Order" "cova"
#' @importFrom abind "adrop"
#' @importFrom MCMCpack "procrustes"
#' @importFrom e1071 "matchClasses" "classAgreement"
#' @importFrom mclust "classError"
#' @importFrom matrixStats "rowMedians" "rowQuantiles"
#' @importFrom utils "head"
#'
#' @seealso \code{\link{mcmc_IMIFA}}, \code{\link{plot.Results_IMIFA}}
#' @references Murphy, K., Gormley, I.C. and Viroli, C. (2017) Infinite Mixtures of Infinite Factor Analysers: Nonparametric Model-Based Clustering via Latent Gaussian Models, \code{https://arxiv.org/abs/1701.07010}
#'
#' @examples
#' # data(coffee)
#' # data(olive)
#'
#' # Run a MFA model on the coffee data over a range of clusters and factors.
#' # simMFAcoffee  <- mcmc_IMIFA(coffee, method="MFA", range.G=2:3, range.Q=0:3, n.iters=1000)
#'
#' # Accept all defaults to extract the optimal model.
#' # resMFAcoffee  <- get_IMIFA_results(simMFAcoffee)
#'
#'
#' # Instead let's get results for a 3-cluster model, allowing Q be chosen by \code{aic.mcmc}.
#' # resMFAcoffee2 <- get_IMIFA_results(simMFAcoffee, G=3, criterion="aic.mcmc")
#'
#' # Run an IMIFA model on the olive data, accepting all defaults.
#' # simIMIFAolive <- mcmc_IMIFA(olive, method="IMIFA", n.iters=10000)
#'
#' # Extract optimum results
#' # Estimate \code{G} & \code{Q} by the \emph{median} of their posterior distributions
#' # Construct 90% credible intervals.
#' # resIMIFAolive <- get_IMIFA_results(simIMIFAolive, G.meth="median",
#' #                                    Q.meth="median", conf.level=0.9)
#' # summary(resIMIFAolive)
get_IMIFA_results              <- function(sims = NULL, burnin = 0L, thinning = 1L, G = NULL, Q = NULL, criterion = c("bicm", "aicm", "bic.mcmc", "aic.mcmc", "log.iLLH", "dic"),
                                           G.meth = c("mode", "median"), Q.meth = c("mode", "median"), dat = NULL, conf.level = 0.95, zlabels = NULL) {
  UseMethod("get_IMIFA_results")
}

#' @export
get_IMIFA_results.IMIFA        <- function(sims = NULL, burnin = 0L, thinning = 1L, G = NULL, Q = NULL, criterion = c("bicm", "aicm", "bic.mcmc", "aic.mcmc", "log.iLLH", "dic"),
                                           G.meth = c("mode", "median"), Q.meth = c("mode", "median"), dat = NULL, conf.level = 0.95, zlabels = NULL) {

  defopt         <- options()
  options(warn=1)
  on.exit(suppressWarnings(options(defopt)), add=TRUE)
  if(missing(sims))               stop("Simulations must be supplied")
  if(class(sims) != "IMIFA")      stop("Object of class 'IMIFA' must be supplied")
  if(!exists(deparse(substitute(sims)),
             envir=.GlobalEnv))   stop(paste0("Object ", match.call()$sims, " not found\n"))
  if(class(sims) != "IMIFA")      stop(paste0("Simulations object of class 'IMIFA' must be supplied"))
  burnin         <- as.integer(burnin)
  thinning       <- as.integer(thinning)
  if(any(c(length(thinning),
           length(burnin)) > 1))  stop("'burnin' and 'thinning' must be of length 1")
  store          <- seq(from=burnin + 1, to=attr(sims, "Store"), by=thinning)
  if(length(store) < 10)          stop("Not enough stored samples to proceed")
  n.store        <- length(store)
  tmp.store      <- store
  label.switch   <- attr(sims, "Label.Switch")
  method         <- attr(sims, "Method")
  alpha.step     <- attr(sims, "Alph.step")
  learn.d        <- attr(sims, "Disc.step")
  inf.G          <- is.element(method, c("IMIFA", "IMFA", "OMIFA", "OMFA"))
  inf.Q          <- !is.element(method, c("FA", "MFA", "OMFA", "IMFA"))
  n.fac          <- attr(sims, "Factors")
  n.grp          <- attr(sims, "Groups")
  n.obs          <- attr(sims, "Obs")
  n.var          <- attr(sims, "Vars")
  sw             <- attr(sims, "Switch")
  cent           <- attr(sims, "Center")
  scaling        <- attr(sims, "Scaling")
  scal.meth      <- attr(scaling, "Method")
  conf.level     <- as.numeric(conf.level)
  if(any(length(conf.level) != 1,
     !is.numeric(conf.level),
     (conf.level <= 0   ||
      conf.level >= 1)))          stop("'conf.level' must be a single number between 0 and 1")
  conf.levels    <- c((1 - conf.level)/2, (1 + conf.level)/2)
  criterion      <- match.arg(criterion)
  if(all(!is.element(method, c("FA", "MFA", "OMFA", "IMFA")),
     !is.element(criterion,  c("log.iLLH",
     "aicm", "bicm"))))           stop(paste0("'criterion' should be one of 'aicm', 'bicm' or 'log.iLLH' for the ", method, " method"))
  recomp         <- any(burnin  > 0,
                    thinning    > 1)

  G.T            <- !missing(G)
  Q.T            <- !missing(Q)
  G.ind          <- Q.ind      <- 1L
  if(inf.G)  {
    GQs          <- length(sims[[G.ind]])
    GQ1          <- GQs > 1
    G.store      <- matrix(unlist(lapply(seq_len(GQs), function(gq) sims[[G.ind]][[gq]]$G.store[store])), nrow=GQs, ncol=n.store, byrow=TRUE)
    G.meth       <- ifelse(missing(G.meth), "mode", match.arg(G.meth))
    G.tab        <- if(GQ1) lapply(apply(G.store, 1, function(x) list(table(x, dnn=NULL))), "[[", 1) else table(G.store, dnn=NULL)
    G.prob       <- if(GQ1) lapply(G.tab, prop.table) else prop.table(G.tab)
    G.mode       <- if(GQ1) unlist(lapply(G.tab, function(gt) as.numeric(names(gt[gt == max(gt)])[1]))) else as.numeric(names(G.tab[G.tab == max(G.tab)])[1])
    G.med        <- if(GQ1) ceiling(matrixStats::rowMedians(G.store) * 2)/2 else ceiling(Rfast::med(G.store) * 2)/2
    if(!G.T) {
      G          <- switch(G.meth, mode=G.mode, floor(G.med))
    }
    G.CI         <- if(GQ1) round(matrixStats::rowQuantiles(G.store, probs=conf.levels)) else round(stats::quantile(G.store, conf.levels))
  }
  if(G.T)    {
    G            <- as.integer(G)
    if(any(length(G) != 1,
           !is.integer(G)))       stop("'G' must be an integer of length 1")
    if(!inf.G) {
      if(!is.element(method, c("FA", "IFA"))) {
        if(!is.element(G, n.grp)) stop("This 'G' value was not used during simulation")
        G.ind    <- which(n.grp == G)
      } else if(G > 1)            message(paste0("Forced G=1 for the ", method, " method"))
    } else   {
      if(all(!inf.Q, GQ1)) {
        if(!Q.T)                  stop(paste0("'G' cannot be supplied without 'Q' for the ", method, " method if a range of Q values were explored"))
        tmpQ     <- which(n.fac == unique(Q))
      } else {
        tmpQ     <- Q.ind
      }
      if(length(tmpQ > 0)  && !is.element(G,
         unique(G.store[tmpQ,]))) stop("This 'G' value was not visited during simulation")
    }
  }
  G              <- if(any(inf.G, all(G.T, !is.element(method, c("FA", "IFA"))))) G else 1L
  if(Q.T)    {
    Q            <- as.integer(Q)
    if(!is.integer(Q))            stop("'Q' must of integer type")
    if(G.T)  {
      if(length(Q) == 1)     Q <- rep(Q, G)
      if(length(Q) != G)          stop(paste0("'Q' must be supplied for each group, as a scalar or vector of length G=", G))
    } else if(length(n.grp)    != 1 && all(!is.element(length(Q),
              c(1,  n.grp))))     stop("'Q' must be a scalar if G=1, 'G' is not suppplied, or a range of G values were explored")
    if(all(is.element(method, c("FA", "MFA", "OMFA", "IMFA")))) {
      if(length(unique(Q)) != 1)  stop(paste0("'Q' cannot vary across groups for the ", method, " method"))
      Q          <- unique(Q)
      if(!is.element(Q,   n.fac)) stop("This 'Q' value was not used during simulation")
      Q.ind      <- which(n.fac == Q)
    }
    if(inf.Q)  {
      if(any((Q  != 0) + (Q *
        (n.var - Q)   <= 0) > 1)) stop(paste0("'Q' must be less than the number of variables ", n.var))
      Qtmp       <- if(inf.G) Rfast::rowMaxs(sims[[1]][[1]]$Q.store[seq_len(G),, drop=FALSE], value=TRUE) else switch(method, MIFA=Rfast::rowMaxs(sims[[ifelse(G.T, which(G == n.grp), G.ind)]][[1]]$Q.store, value=TRUE), max(sims[[1]][[1]]$Q.store))
      if(any(Q * (Qtmp - Q) < 0)) stop(paste0("'Q' can't be greater than the maximum number of factors stored in ", ifelse(method == "IFA", "", "any group of "), match.call()$sims))
    }
  }
  if(inf.G)    {
    tmp.store    <- if(GQ1) lapply(seq_len(GQs), function(gq) store[which(G.store[gq,] == G[ifelse(G.T, 1, gq)])]) else store[which(G.store == G)]
    GQ.temp1     <- list(G = G, G.Mode = G.mode, G.Median = G.med, Stored.G = G.store,
                         G.CI = G.CI, G.Probs = G.prob, G.Counts = G.tab)
  }
  G.range        <- ifelse(G.T, 1, length(n.grp))
  Q.range        <- ifelse(any(Q.T, all(!is.element(method, c("OMFA", "IMFA")), inf.Q)), 1, length(n.fac))
  crit.mat       <- matrix(NA, nrow=G.range, ncol=Q.range)

  # Retrieve log-likelihoods and/or tune G &/or Q according to criterion
    if(all(G.T, Q.T)) {
      dimnames(crit.mat) <- list(paste0("G", G),     if(inf.Q) "IFA" else paste0("Q", Q))
    } else if(G.T)    {
      dimnames(crit.mat) <- list(paste0("G", G),     if(inf.Q) "IFA" else paste0("Q", n.fac))
    } else if(Q.T)    {
      dimnames(crit.mat) <- list(paste0("G", n.grp), if(inf.Q) "IFA" else paste0("Q", Q))
    } else {
      dimnames(crit.mat) <- list(paste0("G", n.grp), if(inf.Q) "IFA" else paste0("Q", n.fac))
    }
    rownames(crit.mat)   <- switch(method, IMFA=, IMIFA="IM", OMFA=, OMIFA="OM", rownames(crit.mat))
    aicm         <- bicm       <- log.iLLH <-
    aic.mcmc     <- bic.mcmc   <- dic      <- crit.mat
    log.N        <- log(n.obs)
    for(g in seq_len(G.range))   {
      gi                 <- ifelse(G.T, G.ind, g)
      for(q in seq_len(Q.range)) {
        qi               <- ifelse(Q.T, Q.ind, q)
        log.likes        <- if(is.element(method, c("OMFA", "IMFA")) && GQ1) sims[[gi]][[qi]]$ll.store[tmp.store[[qi]]] else sims[[gi]][[qi]]$ll.store[tmp.store]
        ll.max           <- 2 * max(log.likes, na.rm=TRUE)
        ll.var           <- ifelse(length(log.likes) != 1, 2 * stats::var(log.likes, na.rm=TRUE), 0)
        ll.mean          <- mean(log.likes, na.rm=TRUE)
        aicm[g,q]        <- ll.max  - ll.var * 2
        bicm[g,q]        <- ll.max  - ll.var * log.N
        log.iLLH[g,q]    <- ll.mean - ll.var * (log.N - 1)
        if(!inf.Q) {
          qk             <- ifelse(G.T, 1, qi)
          K              <- switch(method, OMFA=, IMFA=G[qk] - 1 + G[qk] * .dim(n.fac[qi], n.var), attr(sims[[gi]][[qi]], "K"))
          aic.mcmc[g,q]  <- ll.max  - K * 2
          bic.mcmc[g,q]  <- ll.max  - K * log.N
          dic[g,q]       <- (ll.max - ll.mean) * 3 - ll.mean
        }
      }
    }
    crit         <- get(criterion)
    crit.max     <- which(crit == max(crit), arr.ind=TRUE)

  # Control for supplied values of G &/or Q
    if(!any(Q.T, G.T)) {
      G.ind      <- crit.max[1]
      Q.ind      <- crit.max[2]
      if(!inf.G) {
        G        <- n.grp[G.ind]
      }
      if(!inf.Q) {
        Q        <- n.fac[Q.ind]
      }
    } else if(all(G.T, !Q.T)) {
      Q.ind      <- which(crit == max(crit))
      if(!inf.Q) {
        Q        <- n.fac[Q.ind]
      }
    } else if(all(Q.T, !G.T)) {
      G.ind      <- which(crit == max(crit))
      if(!inf.G) {
        G        <- n.grp[G.ind]
      }
    }
    G            <- ifelse(inf.G, ifelse(G.T, G, G[Q.ind]), ifelse(length(n.grp) == 1, n.grp, G))
    Gseq         <- seq_len(G)
    G.ind        <- ifelse(all(length(n.grp) == 1, !inf.G), which(n.grp == G), G.ind)
    GQ.temp2     <- list(AICMs = aicm, BICMs = bicm, LogIntegratedLikelihoods = log.iLLH)
    if(is.element(method, c("OMFA", "IMFA")) &&
       GQ1)      {
      tmp.store  <- tmp.store[[Q.ind]]
    }
    if(!inf.Q)   {
      Q          <- if(length(n.fac)   > 1)  Q else  n.fac
      Q.ind      <- if(all(!Q.T, length(n.fac) > 1)) Q.ind else which(n.fac == Q)
      Q          <- stats::setNames(if(length(Q) != G) rep(Q, G) else Q, paste0("Group ", Gseq))
      if(all(inf.G, Q.T))  GQ.temp1$G <- rep(G, GQs)
      GQ.temp1   <- if(is.element(method, c("OMFA", "IMFA")) && GQ1) lapply(GQ.temp1, "[[", Q.ind) else if(inf.G) GQ.temp1
      GQ.temp3   <- c(GQ.temp2, list(AIC.mcmcs = aic.mcmc, BIC.mcmcs = bic.mcmc, DICs = dic))
      GQ.res     <- switch(method, OMFA=, IMFA=c(GQ.temp1, list(Q = Q), GQ.temp3), c(list(G = G, Q = Q), GQ.temp3))
    }
    clust.ind    <- !any(is.element(method,   c("FA", "IFA")),
                     all(is.element(method, c("MFA", "MIFA")), G == 1))
    sw.mx        <- ifelse(clust.ind, sw["mu.sw"],  TRUE)
    sw.px        <- ifelse(clust.ind, sw["psi.sw"], TRUE)
    if(inf.Q) {
      Q.store    <- sims[[G.ind]][[Q.ind]]$Q.store[Gseq,tmp.store, drop=FALSE]
      Q.meth     <- ifelse(missing(Q.meth), "mode", match.arg(Q.meth))
    }
    if(length(tmp.store) <= 1)    stop(paste0("Not enough samples stored to proceed", ifelse(any(G.T, Q.T), paste0(": try supplying different Q or G values"), "")))

  # Retrieve dataset
    dat.nam      <- gsub("[[:space:]]", "", ifelse(missing(dat), attr(sims, "Name"), deparse(substitute(dat))))
    nam.dat      <- gsub("\\[.*", "", dat.nam)
    data.x       <- exists(nam.dat, envir=.GlobalEnv)
    pattern      <- c("(", ")")
    if(!data.x) {                 warning(paste0("Object ", nam.dat, " not found in .GlobalEnv: can't compute empirical covariance and error metrics"), call.=FALSE)
    } else      {
      dat        <- as.data.frame(get(nam.dat))
      nam.x      <- gsub(".*\\[(.*)\\].*", "(\\1)",  dat.nam)
      if(any(unlist(vapply(seq_along(pattern), function(p) grepl(pattern[p], nam.dat, fixed=TRUE), logical(1))),
         !identical(dat.nam, nam.dat) && (any(grepl("[[:alpha:]]", gsub('c', '', nam.x))) || grepl(":",
         nam.x, fixed=TRUE)))) {  warning("Extremely inadvisable to supply 'dat' subsetted by any means other than row/column numbers or c() indexing:\n can't compute empirical covariance and error metrics, best to create new data object", call.=FALSE)
      } else  {
        spl.ind          <- if(grepl("(,", nam.x, fixed=TRUE)) sapply(gregexpr("\\(,", nam.x), utils::head, 1L) else sapply(gregexpr("\\)", nam.x), utils::head, 1L)
        spl.tmp          <- c(substring(nam.x, 1, spl.ind), substring(nam.x, spl.ind + 2L, nchar(nam.x)))
        neg.r            <- grepl("-", spl.tmp[1], fixed=TRUE) || grepl("!", spl.tmp[1], fixed=TRUE)
        neg.c            <- grepl("-", spl.tmp[2], fixed=TRUE) || grepl("!", spl.tmp[2], fixed=TRUE)
        rowx             <- as.numeric(unlist(strsplit(gsub('\\(', '', gsub(',', '', unlist(regmatches(spl.tmp[1], gregexpr('\\(?[0-9,.]+', spl.tmp[1]))))), '')))
        rowx             <- if(any(spl.ind <= 0, sum(rowx) %in% 0)) seq_len(nrow(dat)) else rowx
        colseq           <- ifelse(neg.c, -1, 1) * suppressWarnings(as.numeric(unlist(strsplit(gsub('\\(', '', gsub(',', '', unlist(regmatches(spl.tmp[2], gregexpr('\\(?[0-9,.]+', spl.tmp[2]))))), ''))))
        rowseq           <- rep(neg.r, nrow(dat))
        rowseq[rowx]     <- !rowseq[rowx]
        dat              <- subset(dat, select=if(any(spl.ind <= 0, sum(colseq) %in% 0)) seq_len(ncol(dat)) else colseq, subset=rowseq, drop=!grepl("drop=F", dat.nam))
      }
      dat        <- dat[complete.cases(dat),]
      dat        <- dat[vapply(dat, is.numeric, logical(1))]
      dat        <- if(is.logical(scaling)) Rfast::standardise(as.matrix(dat), center=cent, scale=scaling) else scale(dat, center=cent, scale=scaling)
      varnames   <- colnames(dat)
      if(!identical(dim(dat),
         c(n.obs, n.var)))        warning("Dimensions of data don't match those in the dataset supplied to mcmc_IMIFA():\n be careful using subsetted data, best to create new object", call.=FALSE)
      n.obs      <- nrow(dat)
    }

# Manage Label Switching & retrieve cluster labels/mixing proportions
  if(clust.ind) {
    label.miss   <- missing(zlabels)
    if(!label.miss)   {
      z.nam      <- gsub("[[:space:]]", "", deparse(substitute(zlabels)))
      nam.z      <- gsub("\\[.*", "", z.nam)
      nam.zx     <- gsub(".*\\[(.*)\\].*", "\\1)", z.nam)
      if(!exists(nam.z,
         envir=.GlobalEnv))       stop(paste0("Object ", match.call()$zlabels, " not found\n"))
      if(any(unlist(vapply(seq_along(pattern), function(p) grepl(pattern[p], nam.z, fixed=TRUE), logical(1))),
         !identical(z.nam,   nam.z)   && (any(grepl("[[:alpha:]]", gsub('c', '', nam.zx))) || grepl(":",
         nam.zx, fixed=TRUE))))   stop("Extremely inadvisable to supply 'zlabels' subsetted by any means other than row/column numbers or c() indexing: best to create new object")
     if(length(zlabels) != n.obs) stop(paste0("'zlabels' must be a factor of length N=",  n.obs))
    }
    if(sw["mu.sw"])   {
      mus        <- sims[[G.ind]][[Q.ind]]$mu[,,tmp.store, drop=FALSE]
    }
    if(sw["l.sw"])    {
      lmats      <- sims[[G.ind]][[Q.ind]]$load
      if(inf.Q) {
        lmats    <- as.array(lmats)
      }
      lmats      <- lmats[,,,tmp.store, drop=FALSE]
    }
    if(sw["psi.sw"])  {
      psis       <- sims[[G.ind]][[Q.ind]]$psi[,,tmp.store, drop=FALSE]
    }
    if(sw["pi.sw"])   {
      pies       <- sims[[G.ind]][[Q.ind]]$pi.prop[,tmp.store, drop=FALSE]
    }
    z            <- as.matrix(sims[[G.ind]][[Q.ind]]$z.store[,tmp.store])
    if(!label.switch) {
      z.temp     <- try(factor(z[,1], labels=Gseq), silent=TRUE)
      if(inherits(z.temp, "try-error")) {
        z.temp   <- factor(z[,1], levels=Gseq)
      }
      for(sl in seq_along(tmp.store)) {
        sw.lab   <- .lab.switch(z.new=z[,sl], z.old=z.temp, Gs=Gseq)
        z[,sl]   <- sw.lab$z
        z.perm   <- sw.lab$z.perm
        if(!identical(as.integer(z.perm), Gseq)) {
          if(sw["mu.sw"])  {
            mus[,Gseq,sl]      <- mus[,z.perm,sl]
          }
          if(sw["l.sw"])   {
            lmats[,,Gseq,sl]   <- lmats[,,z.perm,sl]
          }
          if(sw["psi.sw"]) {
            psis[,Gseq,sl]     <- psis[,z.perm,sl]
          }
          if(sw["pi.sw"])  {
            pies[Gseq,sl]      <- pies[z.perm,sl]
          }
          if(inf.Q)        {
            Q.store[Gseq,sl]   <- Q.store[z.perm,sl]
          }
        }
      }
    }
    post.z       <- apply(z, 1, function(x) factor(which.max(tabulate(x)), levels=Gseq))
    uncertain    <- 1 - Rfast::colMaxs(matrix(apply(z, 1, tabulate, nbins=G)/length(tmp.store), nrow=G, ncol=n.obs), value=TRUE)
    if(sw["pi.sw"])    {
      pi.prop    <- pies[Gseq,seq_along(tmp.store), drop=FALSE]
      var.pi     <- Rfast::rowVars(pi.prop)
      ci.pi      <- matrixStats::rowQuantiles(pi.prop, probs=conf.levels)
      post.pi    <- Rfast::rowmeans(pi.prop)
    } else {
      post.pi    <- stats::setNames(prop.table(tabulate(post.z, nbins=G)), paste0("Group ", Gseq))
    }
    if(!label.miss) {
      zlabels    <- factor(zlabels, labels=seq_along(unique(zlabels)))
      levs       <- levels(zlabels)
      if(length(levs) == G) {
        sw.lab   <- .lab.switch(z.new=post.z, z.old=zlabels, Gs=Gseq)
        post.z   <- stats::setNames(factor(sw.lab$z, levels=Gseq), names(post.z))
        l.perm   <- sw.lab$z.perm
        z.tmp    <- apply(z, 2, factor, levels=l.perm)
        z        <- provideDimnames(apply(z.tmp, 2, function(x) as.numeric(levels(as.factor(x)))[as.numeric(x)]), base=dimnames(z.tmp))
        if(sw["mu.sw"])    mus <- mus[,l.perm,, drop=FALSE]
        if(sw["l.sw"])   lmats <- lmats[,,l.perm,, drop=FALSE]
        if(sw["psi.sw"])  psis <- psis[,l.perm,, drop=FALSE]
        gnames   <- paste0("Group ", l.perm)
        index    <- order(gnames)
        post.pi  <- stats::setNames(post.pi[index], gnames[index])
        if(sw["pi.sw"]) {
         rownames(pi.prop)     <- gnames
         pi.prop <- pi.prop[index,, drop=FALSE]
         var.pi  <- stats::setNames(var.pi[index],  gnames[index])
         rownames(ci.pi)       <- gnames
         ci.pi   <- ci.pi[index,,   drop=FALSE]
        }
        if(inf.Q)   {
         rownames(Q.store)     <- gnames
         Q.store <- Q.store[index,, drop=FALSE]
        }
      }
      tab        <- table(post.z, zlabels, dnn=list("Predicted", "Observed"))
      tab.stat   <- c(e1071::classAgreement(tab), mclust::classError(post.z, zlabels))
      if(nrow(tab) != ncol(tab))     {
        tab.stat <- tab.stat[-seq_len(2)]
        names(tab.stat)[4]     <- "error.rate"
      } else {
        names(tab.stat)[6]     <- "error.rate"
      }
      if(tab.stat$error.rate   == 0) {
        tab.stat$misclassified <- NULL
      }
      tab.stat   <- c(list(confusion.matrix = tab), tab.stat)
      uncert.obs <- which(uncertain >= 1/G)
      attr(uncertain, "Obs")   <- if(sum(uncert.obs) != 0) uncert.obs
      if(!label.miss && (length(levs) == G)) {
        names(tab.stat)[1]     <- "confusion.matrix.matched"
      }
      class(tab.stat)          <- "listof"
    }
    sizes        <- tabulate(post.z, nbins=G)
    if(any(sizes == 0))           warning("Empty group exists in modal clustering:\n examine trace plots and try supplying a lower G value to tune.imifa() or re-running the model", call.=FALSE)
    if(alpha.step != "fixed") {
      alpha      <- sims[[G.ind]][[Q.ind]]$alpha
      post.alpha <- mean(alpha)
      var.alpha  <- stats::var(alpha)
      ci.alpha   <- stats::quantile(alpha, conf.levels)
      rate       <- switch(alpha.step, metropolis=sims[[G.ind]][[Q.ind]]$rate, 1)
      DP.alpha   <- list(alpha = alpha, post.alpha = post.alpha, var.alpha = var.alpha, ci.alpha = ci.alpha, acceptance.rate = rate)
      class(DP.alpha)          <- "listof"
    }
    cluster      <- list(map = post.z, z = z, uncertainty = uncertain)
    cluster      <- c(cluster, list(post.sizes = sizes, post.pi = post.pi/sum(post.pi)),
                      if(sw["pi.sw"]) list(pi.prop = pi.prop, var.pi = var.pi, ci.pi = ci.pi),
                      if(!label.miss) list(perf = tab.stat),
                      if(alpha.step != "fixed") list(DP.alpha = DP.alpha),
                      if(is.element(method, c("IMFA", "IMIFA"))) list(lab.rate = sims[[G.ind]][[Q.ind]]$lab.rate))
    attr(cluster, "Z.init")    <- attr(sims[[G.ind]], "Z.init")
    attr(cluster, "Init.Meth") <- attr(sims, "Init.Z")
    attr(cluster, "Label.Sup") <- !label.miss
    post.z       <- as.numeric(levels(post.z))[post.z]
    z.ind        <- lapply(Gseq, function(g) post.z == g)
  }
  if(inf.Q)   {
    G1           <- G > 1
    Q.tab        <- if(G1) lapply(apply(Q.store, 1, function(x) list(table(x, dnn=NULL))), "[[", 1) else table(Q.store, dnn=NULL)
    Q.prob       <- if(G1) lapply(Q.tab, prop.table) else prop.table(Q.tab)
    Q.mode       <- if(G1) unlist(lapply(Q.tab, function(qt) as.numeric(names(qt[qt == max(qt)])[1]))) else as.numeric(names(Q.tab[Q.tab == max(Q.tab)])[1])
    Q.med        <- if(G1) ceiling(matrixStats::rowMedians(Q.store) * 2)/2 else ceiling(Rfast::med(Q.store) * 2)/2
    if(!Q.T)  {
      Q          <- switch(Q.meth, mode=Q.mode, floor(Q.med))
    } else    {
      Q          <- if(G.T) Q else stats::setNames(rep(Q, G), paste0("Group ", Gseq))
    }
    leder.b      <- .ledermann(n.obs, n.var)
    if(any(unlist(Q) > leder.b))  warning(paste0("Estimate of Q", ifelse(clust.ind, " in one or more of the groups ", " "), "is greater than the suggested Ledermann upper bound (", leder.b, "):\nsolution may be invalid"), call.=FALSE)
    Q.CI         <- if(G1) round(matrixStats::rowQuantiles(Q.store, probs=conf.levels)) else round(stats::quantile(Q.store, conf.levels))
    GQ.temp4     <- list(Q = Q, Q.Mode = Q.mode, Q.Median = Q.med, Stored.Q = Q.store,
                         Q.CI = Q.CI, Q.Probs = Q.prob, Q.Counts = Q.tab)
    GQ.res       <- if(inf.G) c(GQ.temp1, GQ.temp4) else c(list(G = G), GQ.temp4)
    GQ.res       <- c(GQ.res, GQ.temp2)
    attr(GQ.res, "Q.big") <- attr(sims[[G.ind]][[Q.ind]], "Q.big")
  }

# Retrieve (unrotated) scores
  no.score       <- all(Q == 0)
  if(no.score)   {
    if(sw["s.sw"])                message("Scores & loadings not stored as model has zero factors")
    sw["s.sw"]   <- FALSE
  }
  if(sw["s.sw"]) {
    eta          <- sims[[G.ind]][[Q.ind]]$eta
    if(inf.Q) {
      eta        <- as.array(eta)
    }
    eta          <- eta[,,tmp.store, drop=FALSE]
  }

# Loop over g in G to extract other results
  result         <- list(list())
  e.store        <- list()
  mse   <- mae   <- medse  <-
  medae <- rmse  <-#cvrmse <-
  nrmse <- emp.T <- est.T  <- rep(NA, G)
  for(g in Gseq) {
    Qg           <- Q[g]
    Qgs          <- seq_len(Qg)
    sw["l.sw"]   <- attr(sims, "Switch")["l.sw"]
    if(Qg == 0)  {
      if(all(sw["l.sw"],
             !no.score))          message(paste0("Loadings ", ifelse(G > 1, paste0("for group ", g, " not stored as it"), " not stored as model"), " has zero factors"))
      sw["l.sw"] <- FALSE
    }
    store        <- if(inf.Q) seq_along(tmp.store)[which(Q.store[g,] >= Qg)] else seq_along(tmp.store)
    n.store      <- length(store)

  # Retrieve (unrotated) loadings
    if(sw["l.sw"]) {
      if(clust.ind)  {
        lmat     <- abind::adrop(lmats[,,g,store, drop=FALSE], drop=3)
        l.temp   <- abind::adrop(lmat[,,1, drop=FALSE], drop=3)
      } else {
        lmat     <- sims[[G.ind]][[Q.ind]]$load
        if(inf.Q) {
          lmat   <- as.array(lmat)
        }
        lmat     <- lmat[,,store, drop=FALSE]
        l.temp   <- abind::adrop(lmat[,,1, drop=FALSE], drop=3)
      }
    }

  # Loadings matrix / identifiability / error metrics / etc.
    if(all(sw["s.sw"], clust.ind)) {
      etag       <- eta[z.ind[[g]],,, drop=FALSE]
    }
    if(sw["l.sw"])      {
      for(p in seq_len(n.store))   {
        p2                 <- store[p]
        rot                <- MCMCpack::procrustes(X=as.matrix(lmat[,,p]), Xstar=l.temp)$R
        lmat[,,p]          <- lmat[,,p]  %*% rot
        if(sw["s.sw"])  {
          if(clust.ind) {
            etag[,,p2]     <- etag[,,p2] %*% rot
          } else {
            eta[,,p2]      <- eta[,,p2]  %*% rot
          }
        }
      }
    }
    if(all(sw["s.sw"], clust.ind)) {
      eta[z.ind[[g]],,]    <- etag
    }

  # Retrieve means, uniquenesses & empirical covariance matrix
    if(clust.ind) {
      if(sw["mu.sw"])  {
        mu       <- as.matrix(mus[,g,store])
      }
      if(sw["psi.sw"]) {
        psi      <- as.matrix(psis[,g,store])
      }
      if(all(data.x, sizes[g] > 1)) {
        cov.emp  <- provideDimnames(Rfast::cova(dat[z.ind[[g]],, drop=FALSE]), base=list(varnames, varnames))
      }
    } else {
      post.mu    <- sims[[G.ind]][[Q.ind]]$post.mu
      post.psi   <- sims[[G.ind]][[Q.ind]]$post.psi
      if(sw["mu.sw"])  {
        mu       <- sims[[G.ind]][[Q.ind]]$mu[,store]
      }
      if(sw["psi.sw"]) {
        psi      <- sims[[G.ind]][[Q.ind]]$psi[,store]
      }
      cov.emp    <- sims[[G.ind]][[Q.ind]]$cov.emp
    }
    emp.T[g]     <- exists("cov.emp", envir=environment())

  # Compute posterior means and % variation explained
    if(sw["mu.sw"])  {
      post.mu    <- Rfast::rowmeans(mu)
      var.mu     <- Rfast::rowVars(mu)
      ci.mu      <- matrixStats::rowQuantiles(mu,  probs=conf.levels)
    }
    if(sw["psi.sw"]) {
      post.psi   <- Rfast::rowmeans(psi)
      var.psi    <- Rfast::rowVars(psi)
      ci.psi     <- matrixStats::rowQuantiles(psi, probs=conf.levels)
    }
    if(sw["l.sw"])   {
      lmat       <- lmat[,Qgs,, drop=FALSE]
      post.load  <- rowMeans(lmat, dims=2)
      var.load   <- apply(lmat, c(1, 2), var)
      ci.load    <- apply(lmat, c(1, 2), stats::quantile, conf.levels)
      var.exp    <- sum(colSums(post.load * post.load))/n.var
      class(post.load)   <- "loadings"
    } else if(emp.T[g]) {
      var.exp    <- ifelse(exists("z.ind", envir=.GlobalEnv) && sum(z.ind[[g]]) == 0, 0, max(0, (sum(diag(cov.emp)) - sum(post.psi))/n.var))

    }

  # Calculate estimated covariance matrices & compute error metrics
    if(clust.ind) {
      if(all(sw["psi.sw"], any(sw["l.sw"], Qg == 0))) {
        cov.est  <- if(Qg > 0)    tcrossprod(post.load) + diag(post.psi) else diag(post.psi)
        if(data.x)      {
          dimnames(cov.est)    <- list(varnames, varnames)
        }
      } else if(g == 1) {
        if(all(!sw["l.sw"], Qg  > 0, !sw["psi.sw"]))  {
                                  warning("Loadings & Uniquenesses not stored: can't estimate covariance matrix and compute error metrics", call.=FALSE)
        } else if(all(Qg > 0,
                  !sw["l.sw"])) { warning("Loadings not stored: can't estimate covariance matrix and compute error metrics", call.=FALSE)
        } else if(!sw["psi.sw"])  warning("Uniquenesses not stored: can't estimate covariance matrix and compute error metrics", call.=FALSE)
      }
    } else     {
      cov.est    <- sims[[G.ind]][[Q.ind]]$cov.est
      if(all(recomp, sw["psi.sw"], any(sw["l.sw"], Qg == 0))) {
        cov.est  <- replace(cov.est, is.numeric(cov.est), 0)
        for(r in seq_len(n.store))    {
         sigma   <- if(Qg > 0)    tcrossprod(lmat[,,r]) + diag(psi[,r]) else diag(psi[,r])
         cov.est <- cov.est + sigma/n.store
        }
      } else if(all(recomp,  g == 1)) {
        if(all(!sw["l.sw"], Qg  > 0, !sw["psi.sw"]))  {
                                  warning("Loadings & Uniquenesses not stored: can't re-estimate covariance matrix", call.=FALSE)
        } else if(all(Qg > 0,
                  !sw["l.sw"])) { warning("Loadings not stored: can't re-estimate covariance matrix", call.=FALSE)
        } else if(!sw["psi.sw"])  warning("Uniquenesses not stored: can't re-estimate covariance matrix", call.=FALSE)
      }
    }
    est.T[g]     <- exists("cov.est", envir=environment())

    if(all(emp.T[g], est.T[g])) {
      error      <- cov.emp - cov.est
      sq.error   <- error * error
      abs.error  <- abs(error)
      mse[g]     <- mean(sq.error)
      mae[g]     <- mean(abs.error)
      medse[g]   <- Rfast::med(sq.error)
      medae[g]   <- Rfast::med(abs.error)
      rmse[g]    <- sqrt(mse[g])
      nrmse[g]   <- rmse[g]/(max(cov.emp) - min(cov.emp))
     #cvrmse[g]  <- rmse[g]/mean(cov.emp)
      if(any(all(scal.meth != "none", cent) &&
                 sum(round(diag(cov.est))   !=
                 round(diag(cov.emp)))      != 0,
         sum(abs(post.psi  - (1 - post.psi)) < 0) != 0,
         var.exp  > 1))           warning(paste0(ifelse(G == 1, "C", paste0("Group ", g, "'s c")), "hain may not have fully converged"), call.=FALSE)
    }

    results      <- list(if(sw["mu.sw"])  list(means     = mu,
                                               var.mu    = var.mu,
                                               ci.mu     = ci.mu),
                         if(sw["l.sw"])   list(loadings  = lmat,
                                               post.load = post.load,
                                               var.load  = var.load,
                                               ci.load   = ci.load),
                         if(sw["psi.sw"]) list(psi       = psi,
                                               var.psi   = var.psi,
                                               ci.psi    = ci.psi),
                         if(sw.mx)        list(post.mu   = post.mu),
                         if(sw.px)        list(post.psi  = post.psi),
                         if(any(sw["l.sw"],
                                sw.px))   list(var.exp   = var.exp),
                         if(emp.T[g])     list(cov.emp   = cov.emp),
                         if(est.T[g])     list(cov.est   = cov.est))
    result[[g]]  <- unlist(results, recursive=FALSE)
    attr(result[[g]], "Store") <- n.store
    e.store[[g]] <- store
  }
  if(sw["s.sw"])   {
    eta.store    <- unique(unlist(e.store))
    eta          <- eta[,seq_len(max(Q)),eta.store, drop=FALSE]
    scores       <- list(eta = eta, post.eta = rowMeans(eta, dims=2), var.eta = apply(eta, c(1, 2), var),
                         ci.eta  = apply(eta, c(1, 2), stats::quantile, conf.levels))
    attr(scores, "Eta.store")  <- eta.store
  }
  names(result)  <- paste0("Group", Gseq)
  class(GQ.res)                <- "listof"
  attr(GQ.res, "Criterion")    <- criterion
  attr(GQ.res, "Factors")      <- n.fac
  attr(GQ.res, "Groups")       <- n.grp
  attr(GQ.res, "Supplied")     <- c(Q=Q.T, G=G.T)
  err.T                        <- vapply(Gseq, function(g) all(emp.T[g], est.T[g]), logical(1))
  if(any(err.T))   {
    errors       <- lapply(list(MSE = mse, MAE = mae, MEDSE = medse,
                                MEDAE = medae, RMSE = rmse, NRMSE = nrmse
                                #,CVRMSE = cvrmse
                                ), stats::setNames, paste0("Group ", Gseq))
    if(G > 1)      {
      errors     <- c(errors, list(Averages = unlist(lapply(errors, mean, na.rm=TRUE))))
      class(errors)            <- "listof"
    } else {
      errors     <- stats::setNames(unlist(errors), names(errors))
    }
  }
  if(sw["mu.sw"])  {
    post.mu      <- do.call(cbind, lapply(result, "[[", "post.mu"))
    var.mu       <- do.call(cbind, lapply(result, "[[", "var.mu"))
    ci.mu        <- Filter(Negate(is.null), lapply(result, "[[", "ci.mu"))
    means        <- list(post.mu = post.mu, var.mu = var.mu, ci.mu = ci.mu)
  }
  if(sw["l.sw"])   {
    post.load    <- Filter(Negate(is.null), lapply(result, "[[", "post.load"))
    var.load     <- Filter(Negate(is.null), lapply(result, "[[", "var.load"))
    ci.load      <- Filter(Negate(is.null), lapply(result, "[[", "ci.load"))
    loads        <- list(post.load = post.load, var.load = var.load, ci.load = ci.load)
  }
  if(sw["psi.sw"]) {
    post.psi     <- do.call(cbind, lapply(result, "[[", "post.psi"))
    var.psi      <- do.call(cbind, lapply(result, "[[", "var.psi"))
    ci.psi       <- Filter(Negate(is.null), lapply(result, "[[", "ci.psi"))
    psis         <- list(post.psi = post.psi, var.psi = var.psi, ci.psi = ci.psi)
  }
  result         <- c(result, if(exists("cluster", envir=environment())) list(Clust = cluster),
                      if(any(err.T))   list(Error        = errors),  list(GQ.results = GQ.res),
                      if(sw["mu.sw"])  list(Means        =  means),
                      if(sw["l.sw"])   list(Loadings     =  loads),
                      if(sw["s.sw"])   list(Scores       = scores),
                      if(sw["psi.sw"]) list(Uniquenesses =   psis))

  class(result)                <- "Results_IMIFA"
  attr(result, "Alpha")        <- if(alpha.step == "fixed") attr(sims, "Alpha")
  attr(result, "Conf.Level")   <- conf.level
  attr(result, "Errors")       <- any(err.T)
  attr(result, "Method")       <- method
  if(is.element(method, c("IMFA", "IMIFA"))) {
    attr(result, "Alph.step")  <- alpha.step
    attr(result, "Disc.step")  <- learn.d
    attr(result, "Discount")   <- if(!learn.d) attr(sims, "Discount")
    attr(result, "Ind.Slice")  <- attr(sims, "Ind.Slice")
  }
  attr(result, "Name")         <- attr(sims, "Name")
  attr(result, "Obs")          <- n.obs
  attr(result, "Store")        <- tmp.store
  attr(result, "Switch")       <- sw
  attr(result, "Uni.Meth")     <- attr(sims, "Uni.Meth")
  attr(result, "Vars")         <- n.var
  cat(print.Results_IMIFA(result))
  return(result)
}
