% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summarize.R
\name{summarize}
\alias{summarize}
\title{Compute summary statistics}
\usage{
summarize(
  formula,
  data,
  na.action = stats::na.pass,
  na.rm = FALSE,
  level = 0.95,
  which = c("observed", "missing", "mean", "sd", "min", "median", "max", "correlation"),
  skip.reference = TRUE
)
}
\arguments{
\item{formula}{[formula] on the left hand side the outcome(s) and on the right hand side the grouping variables.
E.g. Y1+Y2 ~ Gender + Gene will compute for each gender and gene the summary statistics for Y1 and for Y2.
Passed to the \code{stats::aggregate} function.}

\item{data}{[data.frame] dataset (in the wide format) containing the observations.}

\item{na.action}{[function] a function which indicates what should happen when the data contain 'NA' values.
Passed to the \code{stats::aggregate} function.}

\item{na.rm}{[logical] Should the summary statistics be computed by omitting the missing values.}

\item{level}{[numeric,0-1] the confidence level of the confidence intervals.}

\item{which}{[character vector] name of the summary statistics to kept in the output.
Can be any of, or a combination of: \code{"observed"} (number of observations with a measurement),
\code{"missing"} (number of observations with a missing value), \code{"mean"}, \code{"mean.lower"}, \code{"mean.upper"},
\code{"sd"}, \code{"min"},
\code{"median"}, \code{"median.lower"}, \code{"median.upper"},
\code{"max"}.}

\item{skip.reference}{[logical] should the summary statistics for the reference level of categorical variables be omitted?}
}
\value{
a data frame containing summary statistics (in columns) for each outcome and value of the grouping variables (rows). It has an attribute \code{"correlation"} when it was possible to compute the correlation matrix for each outcome with respect to the grouping variable.
}
\description{
Compute summary statistics (similar to the SAS macro procmean).
This is essentially an interface to the \code{stats::aggregate} function.
}
\details{
Confidence intervals for the mean are computed via \code{stats::t.test}
and confidence intervals for the median are computed via \code{asht::medianTest}.
}
\examples{
## simulate data in the wide format
set.seed(10)
d <- sampleRem(1e2, n.times = 3)

## add a missing value
d2 <- d
d2[1,"Y2"] <- NA

## run summarize
summarize(Y1 ~ 1, data = d)
summarize(Y1+Y2 ~ X1, data = d)

summarize(Y1 ~ X1, data = d2)
summarize(Y1+Y2 ~ X1, data = d2, na.rm = TRUE)

## long format
dL <- reshape(d, idvar = "id", direction = "long",
             v.names = "Y", varying = c("Y1","Y2","Y3"))
summarize(Y ~ time + X1, data = dL)

## compute correlations (single time variable)
e.S <- summarize(Y ~ time + X1 | id, data = dL, na.rm = TRUE)
e.S
attr(e.S, "correlation")

## compute correlations (composite time variable)
dL$time2 <- dL$time == 2
dL$time3 <- dL$time == 3
e.S <- summarize(Y ~ time2 + time3 + X1 | id, data = dL, na.rm = TRUE)
e.S
attr(e.S, "correlation")
}
