#' Set taxonomy summary configuration
#' @description This function set taxonomy summary configuration by assigning treatment column number, facet column number, replication column number, treatment mapping color, treatment order and facet order.
#' @param taxobj Taxonomic summary object generated by \code{\link{tax_summary}}
#' @param treat_location Numeric. Treatment column number in meta file/group information.
#' @param facet_location Numeric, default:NULL. Facet column number in meta file/group information.
#' @param rep_location Numeric. Replication column number in meta file/group information. By default:"auto",will automaticlly assigned the rep_location.
#' @param subject_location Numeric, default:NULL.Subject column number in meta file/group information (used for pairwise experiment).
#' @param treat_col Named character string, default:NULL. A set of aesthetic character to map treatment to.
#' @param treat_order Character string, default:NULL. The character string indicating treatment displaying order.
#' @param facet_order Character string, default:NULL. The character string indicating facet displaying order.
#'
#' @return object_config returns taxonomy summary object with configuration.
#' @export
#'
#' @author  Wang Ningqi<2434066068@qq.com>
#' @examples
#' {
#'   ### Data preparation ###
#'   data(testotu)
#'   groupinformation <- data.frame(
#'     group = c(rep("a", 10), rep("b", 10)),
#'     factor1 = rnorm(10),
#'     factor2 = rnorm(mean = 100, 10),
#'     subject = factor(c(1:10, 1:10)),
#'     group2 = c(rep("e", 5), rep("f", 5), rep("e", 5), rep("f", 5))
#'   )
#'
#'   ### Packaging metafile, community data, and taxonomy table ###
#'   test_object <- tax_summary(
#'     groupfile = groupinformation,
#'     inputtable = testotu[, 2:21],
#'     reads = TRUE,
#'     taxonomytable = testotu[, c(1, 22)]
#'   )
#'
#'   ### Object configuration ###
#'   test_object_plan1 <- object_config(
#'     taxobj = test_object,
#'     treat_location = 1,
#'     rep_location = 4
#'   )
#'
#'   ### Facet configuration ###
#'   test_object_plan2 <- object_config(
#'     taxobj = test_object,
#'     treat_location = 1,
#'     rep_location = 4,
#'     facet_location = 5
#'   )
#' }
object_config=function(taxobj,treat_location,facet_location=NULL,rep_location="auto",subject_location=NULL,treat_col=NULL,treat_order=NULL,facet_order=NULL){
  gp=methods::slot(taxobj,"groupfile")
  cond=unique(gp[,treat_location])
  if(treat_location>ncol(taxobj$groupfile)){
    stop("Invalid 'treat_location',Please check again!")
  }
  if(rep_location=="auto"){
    for(i in cond){
      gp$rep_auto[gp[,treat_location]==i]=1:as.numeric(table(gp[,treat_location])[i])
    }
    rep_location=ncol(gp)
  }else if(rep_location>ncol(gp)){
    stop("Invalid 'rep_location',Please check again!")
  }
  gp[, rep_location] <- as.numeric(gp[, rep_location]) %>% suppressWarnings()
  if(length(which(is.na(gp[,rep_location])==TRUE))>1){
    stop("Invalid 'rep_location',please confirm replication column in you metadata is in 'numeric'!")
  }
  if(!is.null(treat_order)){
    fit=which(cond %in%treat_order)
    if( length(fit)!=length(cond)){
      warning('treat_order does not match your treatment, please check!')
      return(NULL)
    }
  }
  if(is.null(treat_col)){
    n_treat=length(cond)
    if(n_treat==2){
      treat_col=color_scheme("Plan1",names=cond)
    }else if (n_treat>2&n_treat<=8){
      treat_col=color_scheme("Plan7",names=cond)[cond]
    }else if (n_treat>8&n_treat<=12){
      treat_col=color_scheme("Plan8",names=cond)[cond]
    }else{
      warning("Too many treatments! Colors can not be assigned automaticlly")
    }
  }
  configuration=list(treat_location,facet_location,rep_location,treat_col,treat_order,facet_order,subject_location)
  names(configuration)=c("treat_location","facet_location","rep_location","treat_col","treat_order","facet_order","subject_location")
  configuration <- list(
    treat_location   = treat_location,
    facet_location   = facet_location,
    rep_location     = rep_location,
    treat_col        = treat_col,
    treat_order      = treat_order,
    facet_order      = facet_order,
    subject_location = subject_location
  )
  taxobj <- methods::new("LorMe",
                         groupfile = gp,
                         data      = methods::slot(taxobj, "data"),
                         configuration    = c(methods::slot(taxobj, "configuration"), configuration))
  if(!is.null(methods::slot(taxobj,"tree"))){
    tree=methods::slot(taxobj,"tree")
    taxobj <- add_tree(taxobj, tree)
  }

  return(taxobj)
}
