\name{residuals.marssMLE}
\alias{residuals.marssMLE}

\title{ MARSS Residuals }
\description{
  Calculates the residuals, residuals variance, and standardized residuals for both the one-step-ahead (conditioned on data up to t-1) and smoothed (conditioned on all the data) residuals.
}
\usage{
\method{residuals}{marssMLE}(object,..., conditioning=c("T", "t-1"), normalize=FALSE, silent=FALSE)
}
\arguments{
  \item{object}{ An object of class \code{\link{marssMLE}}.}
  \item{...}{ Additional arguments to be passed to the residuals functions. For conditioning="T", \code{Harvey=TRUE} can be passed into to use the Harvey et al (1998) algorithm. }
  \item{conditioning}{ "T" for smoothed residuals and "t-1" for one-step-ahead residuals. }
  \item{normalize}{ TRUE/FALSE }
  \item{silent}{ If TRUE, do not print inversion warnings. }
}
\value{

\strong{Smoothed residuals}

If \code{conditioning="T"}, a list with the following components  
  \item{model.residuals}{ The smoothed model residuals (data minus model predicted values) as a n x T matrix. The residuals are conditioned on all the data and the set of model parameters.  Called the smoothations. This is different than the Kalman filter innovations which are conditioned on the data up to \eqn{t-1}.}
  \item{state.residuals}{ The smoothed state residuals.}
  \item{residuals}{ The residuals as a (n+m) x T matrix with \code{model.residuals} on top and \code{state.residuals} below. }
  \item{var.residuals}{ The variance of the model residuals and state residuals as a (n+m) x (n+m) x T matrix with the model residuals variance in rows/columns 1 to n and state residuals variances in rows/columns n+1 to n+m. }
  \item{std.residuals}{ The Cholesky standardized residuals as a (n+m) x T matrix. This is \code{residuals} multiplied by the inverse of the Cholesky decomposition of \code{var.residuals}. }
  \item{mar.residuals}{ The marginal standardized residuals as a (n+m) x T matrix. This is \code{residuals} multiplied by the inverse of the diagonal matrix formed by the square-root of the diagonal of \code{var.residuals}. }
    \item{E.obs.residuals}{ The expected value of the model residuals conditioned on the observed data. Returned as a n x T matrix.  For observed data, this will be the observed model residuals. For unobserved data, this will be 0 if \eqn{\mathbf{R}}{R} is diagonal but non-zero if \eqn{\mathbf{R}}{R} is non-diagonal. See \code{\link{MARSSresiduals.tT}}. }
    \item{var.obs.residuals}{ The variance of the model residuals conditioned on the observed data. Returned as a n x n x T matrix.  For observed data, this will be 0. See \code{\link{MARSSresiduals.tT}}. }
  \item{msg}{ Any warning messages. This will be printed unless Object$control$trace = -1 (suppress all error messages). }

  
\strong{One-step-ahead residuals}

If \code{conditioning="t-1"}, a list with the following components  
  \item{residuals}{ The model residuals (data minus model predicted values), as a n x T matrix. The residuals are conditioned on the data up to time \eqn{t-1} and the set of model parameters.  Called the innovations. Residuals associated with missing data will appear as NA. }
  \item{var.residuals}{ The variance of the model residuals as a n x n x T matrix. }
  \item{std.residuals}{ The Cholesky standardized model residuals as a n x T matrix. This is \code{residuals} multiplied by the inverse of the Cholesky decomposition of \code{var.residuals}. }
  \item{mar.residuals}{ The marginal standardized model residuals as a n x T matrix. This is \code{residuals} multiplied by the inverse of the diagonal matrix formed by the square-root of the diagonal of \code{var.residuals}. }
  \item{msg}{ Any warning messages. This will be printed unless Object$control$trace = -1 (suppress all error messages). }

}
\details{

For smoothed residuals, see \code{\link{MARSSresiduals.tT}}.

For one-step-ahead residuals, see \code{\link{MARSSresiduals.tt1}}.

\strong{Standardized residuals}

\code{std.residuals} are Cholesky standardized residuals. These are the residuals muliplied by the inverse of the Cholesky decomposition of the variance matrix of the residuals: 
\deqn{ \hat{\Sigma}_t^{-1/2} \hat{\mathbf{v}}_t.}{ hatSigma_t^{-1/2} hatv_t.}
These residuals are uncorrelated. 

The interpretation of the Cholesky standardized residuals is not straight-forward when the Q and R variance-covariance matrices are non-diagonal.  The residuals which were generated by a non-diagonal variance-covariance matrices are transformed into orthogonal residuals in MVN(0,I) space.  For example, if v is 2x2 correlated errors with variance-covariance matrix R. The transformed residuals (from this function) for the i-th row of v is a combination of the row 1 effect and the row 1 effect plus the row 2 effect.  So in this case, row 2 of the transformed residuals would not be regarded as solely the row 2 residual but rather how different row 2 is from row 1, relative to expected.  If the errors are highly correlated, then the Cholesky standardized residuals can look rather non-intuitive.

\code{mar.residuals} are the marginal standardized residuals. These are the residuals muliplied by the inverse of the diagonal matrix formed from the square-root of the diagonal of the variance matrix of the residuals: 
\deqn{ \textrm{dg}(\hat{\Sigma}_t)^{-1/2} \hat{v}_t,}{ dg(hatSigma_t)^{-1/2} hatv_t,} where 'dg(A)' is the square matrix formed from the diagonal of A, aka \code{diag(diag(A))}. These residuals will be correlated if the variance matrix is non-diagonal. 

\strong{Normalized residuals}

If \code{normalize=FALSE}, the unconditional variance of \eqn{V_t} and \eqn{W_t} are R and Q and the model is assumed to be written as
\deqn{ y_t = Z x_t + a + v_t}
\deqn{ x_t = B x_{t-1} + u + w_t}
If normalize=TRUE, the model is assumed to be written
\deqn{ y_t = Z x_t + a + Hv_t}
\deqn{ x_t = B x_{t-1} + u + Gw_t}
with the variance of \eqn{V_t} and \eqn{W_t} equal to I (identity).

\strong{Missing or left-out data}

See the discussion of smoothed residuals for missing and left-out data in \code{\link{MARSSresiduals.tT}}.

}

\author{ 
  Eli Holmes, NOAA, Seattle, USA.  

  eli(dot)holmes(at)noaa(dot)gov
} 
\seealso{ \code{\link{MARSSresiduals.tT}}, \code{\link{MARSSresiduals.tt1}}, \code{\link{plot.marssMLE}} }
\examples{
  dat <- t(harborSeal)
  dat <- dat[c(2,11),]
  MLEobj <- MARSS(dat)
  
  #state smoothed residuals
  state.resids1 <- residuals(MLEobj)$state.residuals
  #this is the same as
  states <- MLEobj$states
  Q <- coef(MLEobj,type="matrix")$Q
  state.resids2 <- states[,2:30]-states[,1:29]-matrix(coef(MLEobj,type="matrix")$U,2,29)
  #compare the two
  cbind(t(state.resids1[,-30]), t(state.resids2))

  #normalize to variance of 1
  state.resids1 <- residuals(MLEobj, normalize=TRUE)$state.residuals
  state.resids2 <- (solve(t(chol(Q))) \%*\% state.resids2)
  cbind(t(state.resids1[,-30]), t(state.resids2))

  #one-step-ahead standardized residuals
  residuals(MLEobj, conditioning="t-1")$std.residuals
}
\references{
Holmes, E. E. 2014. Computation of standardized residuals for (MARSS) models. Technical Report. arXiv:1411.0045.

See also the discussion and references in \code{\link{MARSSresiduals.tT}} and \code{\link{MARSSresiduals.tt1}}.
}

