% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/approximate_algorithm.R
\name{approx_horseshoe}
\alias{approx_horseshoe}
\title{Run approximate MCMC algorithm for horseshoe prior}
\usage{
approx_horseshoe(
  y,
  X,
  burn = 1000,
  iter = 5000,
  auto.threshold = TRUE,
  threshold = 0,
  tau = 1,
  s = 0.8,
  sigma2 = 1,
  w = 1,
  alpha = 0.05,
  a = 0.2,
  b = 10,
  t = 10,
  adapt_p0 = 0,
  adapt_p1 = -4.6 * 10^(-4)
)
}
\arguments{
\item{y}{Response vector, \eqn{y \in \mathbb{R}^{N}}.}

\item{X}{Design matrix, \eqn{X \in \mathbb{R}^{N \times p}}.}

\item{burn}{Number of burn-in samples. The default is 1000.}

\item{iter}{Number of samples to be drawn from the posterior. The default is
5000.}

\item{auto.threshold}{Argument for setting whether to use an algorithm that
automatically updates the threshold using adaptive probability.}

\item{threshold}{Threshold to be used in the approximate MCMC algorithm.
This argument is ignored when auto.threshold=TRUE. If you select
auto.threshold = FALSE and threshold = 0 (This is the default value for the
threshold argument), the threshold is set to
\eqn{\sqrt{p \times min(N, p)}} as suggested in Johndrow et al. (2020). Or,
you can set your custom value directly through this argument. For more
information about \eqn{\delta}, browseVignettes("Mhorseshoe") and 4.1 of
Johndrow et al. (2020).}

\item{tau}{Initial value of the global shrinkage parameter \eqn{\tau} when
starting the algorithm. The default is 1.}

\item{s}{\eqn{s^{2}} is the variance of tau's MH proposal distribution.
0.8 is a good default. If set to 0, the algorithm proceeds by fixing the
global shrinkage parameter \eqn{\tau} to the initial setting value.}

\item{sigma2}{Initial value of error variance \eqn{\sigma^{2}}. The default
is 1.}

\item{w}{A hyperparameter of gamma prior for \eqn{\sigma^{2}}. The default
is 1.}

\item{alpha}{\eqn{100(1-\alpha)\%} credible interval setting argument.}

\item{a}{A tuning parameter of the rejection sampler, where the default
value is \eqn{a = 1/5}.}

\item{b}{A tuning parameter of the rejection sampler, where the default
value is \eqn{b = 10}.}

\item{t}{Threshold update cycle for adaptive probability algorithm when
auto.threshold is set to TRUE. The default is 10.}

\item{adapt_p0}{A tuning parameter \eqn{p_{0}} of the adaptive probability,
\eqn{p(t) = exp[p_{0} + p_{1}t]}. The default is \eqn{0}.}

\item{adapt_p1}{A tuning parameter \eqn{a_{1}} of the adaptive probability,
\eqn{p(t) = exp[p_{0} + p_{1}t]}. The default is \eqn{-4.6 \times 10^{-4}}.}
}
\value{
\item{BetaHat}{Posterior mean of \eqn{\beta}.}
\item{LeftCI}{Lower bound of \eqn{100(1-\alpha)\%} credible interval for
\eqn{\beta}.}
\item{RightCI}{Upper bound of \eqn{100(1-\alpha)\%} credible interval for
\eqn{\beta}.}
\item{Sigma2Hat}{Posterior mean of \eqn{\sigma^{2}}.}
\item{TauHat}{Posterior mean of \eqn{\tau}.}
\item{LambdaHat}{Posterior mean of \eqn{\lambda_{j},\ j=1,2,...p.}.}
\item{ActiveMean}{Average number of elements in the active set per iteration
in this algorithm.}
\item{BetaSamples}{Posterior samples of \eqn{\beta}.}
\item{LambdaSamples}{Posterior samples of local shrinkage parameters.}
\item{TauSamples}{Posterior samples of global shrinkage parameter.}
\item{Sigma2Samples}{Posterior samples of \eqn{sigma^{2}}.}
\item{ActiveSet}{\eqn{\mathbb{R}^{iter \times p}} Matrix indicating active
elements as 1 and non-active elements as 0 per iteration of the MCMC
algorithm.}
}
\description{
The approximate MCMC algorithm for the horseshoe prior
}
\details{
This function implements the approximate algorithm introduced in Section
2.2 of Johndrow et al. (2020) and the method proposed in this package, which
improves computation speed when p >> N. The approximate algorithm introduces
a threshold and uses only a portion of the total \eqn{p} columns for matrix
multiplication, reducing the computational cost compared to the existing
MCMC algorithms for the horseshoe prior. The "auto.threshold" argument
determines whether the threshold used in the algorithm will be updated by
the adaptive method proposed in this package.
}
\examples{
# Making simulation data.
set.seed(123)
N <- 200
p <- 100
true_beta <- c(rep(1, 10), rep(0, 90))

X <- matrix(1, nrow = N, ncol = p) # Design matrix X.
for (i in 1:p) {
  X[, i] <- stats::rnorm(N, mean = 0, sd = 1)
}

y <- vector(mode = "numeric", length = N) # Response variable y.
e <- rnorm(N, mean = 0, sd = 2) # error term e.
for (i in 1:10) {
  y <- y + true_beta[i] * X[, i]
}
y <- y + e

# Run with auto.threshold set to TRUE
result1 <- approx_horseshoe(y, X, burn = 0, iter = 100,
                            auto.threshold = TRUE)

# Run with fixed custom threshold
result2 <- approx_horseshoe(y, X, burn = 0, iter = 100,
                            auto.threshold = FALSE, threshold = 1/(5 * p))

# posterior mean
betahat <- result1$BetaHat

# Lower bound of the 95\% credible interval
leftCI <- result1$LeftCI

# Upper bound of the 95\% credible interval
RightCI <- result1$RightCI

}
\references{
Johndrow, J., Orenstein, P., & Bhattacharya, A. (2020).
Scalable Approximate MCMC Algorithms for the Horseshoe Prior. In Journal
of Machine Learning Research, 21, 1-61.
}
