% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MittagLefflerDist.R, R/MittagLefflerFunc.R
\name{MittagLeffleR}
\alias{MittagLeffleR}
\alias{dml}
\alias{pml}
\alias{qml}
\alias{rml}
\alias{MittagLeffleR}
\alias{mlf}
\title{Mittag-Leffler Distribution}
\usage{
dml(x, tail, scale = 1, log = FALSE, second.type = FALSE)

pml(q, tail, scale = 1, second.type = FALSE, lower.tail = TRUE,
  log.p = FALSE)

qml(p, tail, scale = 1, second.type = FALSE, lower.tail = TRUE,
  log.p = FALSE)

rml(n, tail, scale = 1, second.type = FALSE)

mlf(z, a, b = 1, g = 1)
}
\arguments{
\item{x, q}{vector of quantiles.}

\item{tail}{tail parameter.}

\item{scale}{scale parameter.}

\item{log, log.p}{logical; if TRUE, probabilities p are given as log(p).}

\item{second.type}{logical; if FALSE (default), 
first type of Mittag-Leffler distribution is assumed.}

\item{lower.tail}{logical; if TRUE, probabilities are \eqn{P[X \le x]}
otherwise, \eqn{P[X > x]}}

\item{p}{vector of probabilities.}

\item{n}{number of observations. If length(n) > 1, the length is taken
to be the number required.}

\item{z}{The argument (real-valued)}

\item{a, b, g}{Parameters of the Mittag-Leffler distribution; see Garrappa}
}
\value{
\code{dml} returns the density,
        \code{pml} returns the distribution function,
        \code{qml} returns the quantile function, and
        \code{rml} generates random variables.

\code{mlf} returns the value of the Mittag-Leffler function.
}
\description{
Probability density, cumulative distribution
function, quantile function and random variate generation for the
two types of Mittag-Leffler distribution, as well as the 1, 2 and 3
parameter Mittag-Leffler function.
}
\details{
The generalized (two-parameter) Mittag-Leffer function is defined by the
power series
    \deqn{E_{\alpha,\beta} (z) = \sum_{k=0}^\infty  z^k / \Gamma(\alpha
    k + \beta) }
for complex \eqn{z} and complex \eqn{\alpha, \beta} with
\eqn{Real(\alpha) > 0} (only implemented for real valued parameters)

The **first type** of Mittag-Leffler distribution assumes the Mittag-Leffler
function as its tail function, so that the CDF is given by
\deqn{F(q; \alpha, \tau) = 1 - E_{\alpha,1} (-(q/\tau)^\alpha)}
for \eqn{q \ge 0}, tail parameter \eqn{0 < \alpha \le 1},
and scale parameter \eqn{\tau > 0}.
Its PDF is given by
\deqn{f(x; \alpha, \tau) = x^{\alpha - 1} 
E_{\alpha,\alpha} [-(x/\tau)^\alpha] / \tau^\alpha.}
As \eqn{\alpha} approaches 1 from below, the Mittag-Leffler converges
(weakly) to the exponential
distribution. For \eqn{0 < \alpha < 1}, it is (very) heavy-tailed, i.e.
has infinite mean.

The **second type** of Mittag-Leffler distribution is defined via the
Laplace transform of its density f:
\deqn{\int_0^\infty \exp(-sx) f(x; \alpha, 1) dx = E_{\alpha,1}(-s)}
It is light-tailed, i.e. all its moments are finite.
At scale \eqn{\tau}, its density is 
\deqn{f(x; \alpha, \tau) = f(x/\tau; \alpha, 1) / \tau.}
}
\examples{
dml(1, 0.8)
dml(1, 0.6, second.type=TRUE)
pml(2, 0.7, 1.5)
qml(p = c(0.25, 0.5, 0.75), tail = 0.6, scale = 100)
rml(1000, 0.7, 1)

## Fitting Mittag-Leffler distribution to observations X by Maximum
## Likelihood

mlml <- function(X) {
  log_l <- function(theta) {
    #transform parameters so can do optimization unconstrained
    theta[1] <- 1/(1+exp(-theta[1]))
    theta[2] <- exp(theta[2])
    -sum(log(dml(X,theta[1],theta[2])))
  }
  ml_theta <- stats::optim(c(0.5,0.5), fn=log_l)$par
  ml_a <- 1/(1 + exp(-ml_theta[1]))
  ml_d <- exp(ml_theta[2])
  print(paste("tail =", ml_a, "scale =", ml_d))
}
mlml(rml(n = 100, tail = 0.9, scale = 2))
mlf(2,0.7)
}
\references{
Haubold, H. J., Mathai, A. M., & Saxena, R. K. (2011). Mittag-Leffler
Functions and Their Applications. Journal of Applied Mathematics, 2011, 
1–51. \url{http://doi.org/10.1155/2011/298628}

Garrappa, R. (2015). Numerical Evaluation of Two and Three Parameter
Mittag-Leffler Functions. SIAM Journal on Numerical Analysis, 53(3),
 1350–1369. \url{http://doi.org/10.1137/140971191}

Mittag-Leffler distribution. (2017, May 3).
In Wikipedia, The Free Encyclopedia.
\url{https://en.wikipedia.org/w/index.php?title=Mittag-Leffler_distribution&oldid=778429885}

The Mittag-Leffler function. MathWorks File Exchange.
\url{https://au.mathworks.com/matlabcentral/fileexchange/48154-the-mittag-leffler-function}
}
