% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/powerOneWayTests.R
\name{powerOneWayTests}
\alias{powerOneWayTests}
\title{Power Simulation for One-Factorial Single Hypothesis Tests}
\usage{
powerOneWayTests(mu, n = 10, errfn = c("Normal", "Lognormal", "Exponential",
  "Chisquare", "TDist", "Cauchy", "Weibull"), parms = list(mean = 0, sd = 1),
  test = c("kruskalTest", "leTest", "vanWaerdenTest", "normalScoresTest",
  "spearmanTest", "cuzickTest", "jonckheereTest", "johnsonTest", "oneway.test",
  "adKSampleTest", "bwsKSampleTest", "bwsTrendTest", "mackWolfeTest"),
  alternative = c("two.sided", "greater", "less"), var.equal = TRUE,
  dist = NULL, alpha = 0.05, FWER = TRUE, replicates = 1000, p = NULL)
}
\arguments{
\item{mu}{numeric vector of group means.}

\item{n}{number of replicates per group. If \code{n} is a scalar, then
a balanced design is assumed. Otherwise, \code{n} must be a vector of same
length as \code{mu}.}

\item{errfn}{the error function. Defaults to \code{"Normal"}.}

\item{parms}{a list that denotes the arguments for the error function.
Defaults to \code{list(mean=0, sd=1)}.}

\item{test}{the test for which the power analysis is
to be performed. Defaults to \code{"kwManyOneConoverTest"}.}

\item{alternative}{the alternative hypothesis. Defaults to \code{"two.sided"},
ignored if the selected error function does not use this argument.}

\item{var.equal}{a logical variable indicating whether to treat the variances
         in the samples as equal.  \code{"TRUE"}, then a simple F test for
         the equality of means in a one-way analysis of variance is
         performed.  If \code{"FALSE"}, an approximate method of Welch (1951)
         is used, which generalizes the commonly known 2-sample Welch
         test to the case of arbitrarily many samples. Defaults to \code{"TRUE"}; only relevant,
if \code{test = "oneway.test"}, otherwise ignored.}

\item{dist}{the test distribution. Only relevant for
\code{\link{kruskalTest}}. Defaults's to \code{NULL}.}

\item{alpha}{the nominal level of Type I Error.}

\item{FWER}{logical, indicates whether the family-wise error should be computed.
Defaults to \code{TRUE}.}

\item{replicates}{the number of Monte Carlo replicates or runs. Defaults to \code{1000}.}

\item{p}{the a-priori known peak as an ordinal number of the treatment
group including the zero dose level, i.e. \eqn{p = \{1, \ldots, k\}}.
Defaults to \code{NULL}. Only relevant, if \code{"mackWolfeTest"} is selected.}
}
\value{
An object with class \code{powerOneWayPMCMR}.
}
\description{
Performs power simulation for one-factorial
single hypothesis tests.
}
\details{
The linear model of a one-way ANOVA can be written as: 

\deqn{
X_{ij} = \mu_i + \epsilon_{ij}
}

For each Monte Carlo run, the function simulates \eqn{\epsilon_{ij}} based on the given error function and
the corresponding parameters. Then the specified test is performed.
Finally, Type I and Type II error rates are calculated.
}
\examples{
\dontrun{
set.seed(12)
mu <- c(0, 0, 1, 2)
n <- c(5, 4, 5, 5)
parms <- list(mean=0, sd=1)
powerOneWayTests(mu, n, parms, test = "cuzickTest",
alternative = "two.sided", replicates = 1E4)

## Compare power estimation for
## one-way ANOVA with balanced design
## as given by functions
## power.anova.test, pwr.anova.test
## and powerOneWayTest

groupmeans <- c(120, 130, 140, 150)
SEsq <- 500  # within-variance
n <- 10
k <- length(groupmeans)
df <- n * k - k
SSQ.E <- SEsq * df
SSQ.A <- n * var(groupmeans) * (k - 1)
sd.errfn <- sqrt(SSQ.E / (n * k - 1))
R2 <- c("R-squared" = SSQ.A / (SSQ.A + SSQ.E))
cohensf <- sqrt(R2 / (1 - R2))
names(cohensf) <- "Cohens f"

## R stats power function 
power.anova.test(groups = k,
                 between.var = var(groupmeans),
                 within.var = SEsq,
                 n = n)

## pwr power function
pwr.anova.test(k = k, n = n, f = cohensf, sig.level=0.05)

## this Monte-Carlo based estimation  
set.seed(200)
powerOneWayTests(mu = groupmeans,
                 n = n,
                 parms = list(mean=0, sd=sd.errfn),
                 test = "oneway.test",
                 var.equal = TRUE,
                 replicates = 5E3)

## Compare with effect sizes
R2
cohensf

}

}
\seealso{
\code{\link{powerMCTests}},
\code{\link[pwr]{pwr.anova.test}}, 
\code{\link[stats]{power.anova.test}}
}
\concept{
TestPower
}
\keyword{misc}
