% Generated by roxygen2 (4.0.2): do not edit by hand
\name{linear.model.MLE}
\alias{linear.model.MLE}
\title{Maximum Likelihood estimation for the geostatistical linear Gaussian model}
\usage{
linear.model.MLE(formula, coords, data, kappa, fixed.rel.nugget = NULL,
  start.cov.pars, method = "BFGS", low.rank = FALSE, knots = NULL,
  messages = TRUE)
}
\arguments{
\item{formula}{an object of class "\code{\link{formula}}" (or one that can be coerced to that class): a symbolic description of the model to be fitted.}

\item{coords}{an object of class \code{\link{formula}} indicating the geographic coordinates.}

\item{data}{a data frame containing the variables in the model.}

\item{kappa}{shape parameter of the Matern covariance function.}

\item{fixed.rel.nugget}{fixed value for the relative variance of the nugget effect; default is \code{fixed.rel.nugget=NULL} if this should be included in the estimation.}

\item{start.cov.pars}{a vector of length two with elements corresponding to the starting values of \code{phi} and the relative variance of the nugget effect \code{nu2}, respectively, that are used in the optimization algorithm. If \code{nu2} is fixed through \code{fixed.rel.nugget}, then start.cov.pars represents the starting value for \code{phi} only.}

\item{method}{method of optimization. If \code{method="BFGS"} then the \code{\link{maxBFGS}} function is used; otherwise \code{method="nlminb"} to use the \code{\link{nlminb}} function. Default is \code{method="BFGS"}.}

\item{low.rank}{logical; if \code{low.rank=TRUE} a low-rank approximation of the Gaussian spatial process is used when fitting the model. Default is \code{low.rank=FALSE}.}

\item{knots}{if \code{low.rank=TRUE}, \code{knots} is a matrix of spatial knots that are used in the low-rank approximation. Default is \code{knots=NULL}.}

\item{messages}{logical; if \code{messages=TRUE} then status messages are printed on the screen (or output device) while the function is running. Default is \code{messages=TRUE}.}
}
\value{
An object of class "PrevMap".
The function \code{\link{summary.PrevMap}} is used to print a summary of the fitted model.
The object is a list with the following components:

\code{estimate}: estimates of the model parameters; use the function \code{\link{coef.PrevMap}} to obtain estimates of covariance parameters on the original scale.

\code{covariance}: covariance matrix of the ML estimates.

\code{log.lik}: maximum value of the log-likelihood.

\code{y}: response variable.

\code{D}: matrix of covariates.

\code{coords}: matrix of the observed sampling locations.

\code{method}: method of optimization used.

\code{kappa}: fixed value of the shape parameter of the Matern function.

\code{knots}: matrix of the spatial knots used in the low-rank approximation.

\code{const.sigma2}: adjustment factor for \code{sigma2} in the low-rank approximation.

\code{fixed.rel.nugget}: fixed value for the relative variance of the nugget effect.

\code{call}: the matched call.
}
\description{
This function performs maximum likelihood estimation for the geostatistical linear Gaussian Model.
}
\details{
This function estimates the parameters of a geostatistical linear Gaussian model, specified as
\deqn{Y = d'\beta + S(x) + Z,}
where \eqn{Y} is the measured outcome, \eqn{d} is a vector of coavariates, \eqn{\beta} is a vector of regression coefficients, \eqn{S(x)} is a stationary Gaussian spatial process and \eqn{Z} are independent zero-mean Gaussian variables with variance \code{tau2}. More specifically, \eqn{S(x)} has an isotropic Matern covariance function with variance \code{sigma2}, scale parameter \code{phi} and shape parameter \code{kappa}. In the estimation, the shape parameter \code{kappa} is treated as fixed. The relative variance of the nugget effect, \code{nu2=tau2/sigma2}, can be fixed though the argument \code{fixed.rel.nugget}; if \code{fixed.rel.nugget=NULL}, then the variance of the nugget effect is also included in the estimation.

\bold{Low-rank approximation.}
In the case of very large spatial data-sets, a low-rank approximation of the Gaussian spatial process \eqn{S(x)} can be computationally beneficial. Let \eqn{(x_{1},\dots,x_{m})} and \eqn{(t_{1},\dots,t_{m})} denote the set of sampling locations and a grid of spatial knots covering the area of interest, respectively. Then \eqn{S(x)} is approximated as \eqn{\sum_{i=1}^m K(\|x-t_{i}\|; \phi, \kappa)U_{i}}, where \eqn{U_{i}} are zero-mean mutually independent Gaussian variables with variance \code{sigma2} and \eqn{K(.;\phi, \kappa)} is the isotropic Matern kernel (see \code{\link{matern.kernel}}). Since the resulting approximation is no longer a stationary process, the parameter \code{sigma2} is adjusted by a factor\code{constant.sigma2}. See \code{\link{adjust.sigma2}} for more details on the the computation of the adjustment factor \code{constant.sigma2} in the low-rank approximation.
}
\examples{
data(loaloa)
# Empirical logit transformation
loaloa$logit <- log((loaloa$NO_INF+0.5)/(loaloa$NO_EXAM-loaloa$NO_INF+0.5))
fit.MLE <- linear.model.MLE(logit ~ 1,coords=~LONGITUDE+LATITUDE,
                data=loaloa, start.cov.pars=c(0.2,0.15),
                 kappa=0.5)
summary(fit.MLE)

# Low-rank approximation
data(data_sim)
n.subset <- 200
data_subset <- data_sim[sample(1:nrow(data_sim),n.subset),]

# Logit transformation
data_subset$logit <- log(data_subset$y+0.5)/
                                 (data_subset$units.m-data_subset$y+0.5)
knots <- as.matrix(expand.grid(seq(-0.2,1.2,length=8),seq(-0.2,1.2,length=8)))

fit <- linear.model.MLE(formula=logit~1,coords=~x1+x2,data=data_subset,
                             kappa=2,start.cov.pars=c(0.15,0.1),low.rank=TRUE,
                             knots=knots)
summary(fit,log.cov.pars=FALSE)
}
\author{
Emanuele Giorgi \email{e.giorgi@lancaster.ac.uk}

Peter J. Diggle \email{p.diggle@lancaster.ac.uk}
}
\seealso{
\code{\link{shape.matern}}, \code{\link{summary.PrevMap}}, \code{\link{coef.PrevMap}}, \code{\link{matern}}, \code{\link{matern.kernel}}, \code{\link{maxBFGS}}, \code{\link{nlminb}}.
}

