% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/autoencoder_functions.R
\name{atseq2feature_seq2seq}
\alias{atseq2feature_seq2seq}
\title{Feature Extraction by action and time sequence autoencoder}
\usage{
atseq2feature_seq2seq(atseqs, K, weights = c(1, 0.5),
  cumulative = FALSE, log = TRUE, rnn_type = "lstm", n_epoch = 50,
  method = "last", step_size = 1e-04, optimizer_name = "rmsprop",
  samples_train, samples_valid, samples_test = NULL, pca = TRUE,
  gpu = FALSE, parallel = FALSE, seed = 12345, verbose = TRUE,
  return_theta = TRUE)
}
\arguments{
\item{atseqs}{a list of two elements, first element is the list of \code{n} action sequences, Each element 
is an action sequence in the form of a vector of actions. The second element is the list of \code{n} 
timestamp sequences corresponding to the action sequences. Each element is a numeric sequence in the form 
of a vector of timestamps associated with actions, with the timestamp of the first event (e.g., "start") of 0.}

\item{K}{the number of features to be extracted.}

\item{weights}{a vector of 2 elements for the weight of the loss of action sequences
(categorical_crossentropy) and time sequences (mean squared error), respectively. 
The total loss is calculated as the weighted sum of the two losses.}

\item{cumulative}{logical. If TRUE, the sequence of cumulative time up to each event is
used as input to the neural network. If FALSE, the sequence of inter-arrival time (gap 
time between an event and the previous event) will be used as input to the neural network.
Default is FALSE.}

\item{log}{logical. If TRUE, for the timestamp sequences, input of the neural net is
the base-10 log of the original sequence of times plus 1 (i.e., log10(t+1)). If FALSE,
the original sequence of times is used.}

\item{rnn_type}{the type of recurrent unit to be used for modeling
response processes. \code{"lstm"} for the long-short term memory unit. 
\code{"gru"} for the gated recurrent unit.}

\item{n_epoch}{the number of training epochs for the autoencoder.}

\item{method}{the method for computing features from the output of an
recurrent neural network in the encoder. Available options are 
\code{"last"} and \code{"avg"}.}

\item{step_size}{the learning rate of optimizer.}

\item{optimizer_name}{a character string specifying the optimizer to be used
for training. Availabel options are \code{"sgd"}, \code{"rmsprop"}, 
\code{"adadelta"}, and \code{"adam"}.}

\item{samples_train}{vectors of indices specifying the
training, validation and test sets for training autoencoder.}

\item{samples_valid}{vectors of indices specifying the
training, validation and test sets for training autoencoder.}

\item{samples_test}{vectors of indices specifying the
training, validation and test sets for training autoencoder.}

\item{pca}{logical. If TRUE, the principal components of features are
returned. Default is TRUE.}

\item{gpu}{logical. If TRUE, use gpu for training when available. Default is FALSE.}

\item{parallel}{logical. If TRUE, allow cpu parallel computing. Default is FALSE.}

\item{seed}{random seed.}

\item{verbose}{logical. If TRUE, training progress is printed.}

\item{return_theta}{logical. If TRUE, extracted features are returned.}
}
\value{
\code{tseq2feature_seq2seq} returns a list containing
  \item{theta}{a matrix containing \code{K} features or principal features. Each column is a feature.}
  \item{train_loss}{a vector of length \code{n_epoch} recording the trace of training losses.}
  \item{valid_loss}{a vector of length \code{n_epoch} recording the trace of validation losses.}
  \item{test_loss}{a vector of length \code{n_epoch} recording the trace of test losses. Exists only if \code{samples_test} is not \code{NULL}.}
}
\description{
\code{atseq2feature_seq2seq} extract features from action and timestamp sequences by a 
sequence autoencoder.
}
\details{
This function trains a sequence-to-sequence autoencoder using keras. The encoder
of the autoencoder consists of a recurrent neural network.
The decoder consists of another recurrent neural network followed by a fully connected layer
with softmax activation for actions and another fully connected layer with ReLU activation 
for times. The outputs of the encoder are the extracted features.

The output of the encoder is a function of the encoder recurrent neural network.
It is the last latent state of the encoder recurrent neural network if \code{method="last"}
and the average of the encoder recurrent neural network latent states if \code{method="avg"}.
}
\examples{
\donttest{
n <- 50
data(cc_data)
samples <- sample(1:length(cc_data$seqs$time_seqs), n)
atseqs <- sub_seqs(cc_data$seqs, samples)
action_and_time_seq2seq_res <- atseq2feature_seq2seq(atseqs, 5, rnn_type="lstm", n_epoch=5, 
                                   samples_train=1:40, samples_valid=41:50)
features <- action_and_time_seq2seq_res$theta
plot(action_and_time_seq2seq_res$train_loss, col="blue", type="l",
     ylim = range(c(action_and_time_seq2seq_res$train_loss, 
                    action_and_time_seq2seq_res$valid_loss)))
lines(action_and_time_seq2seq_res$valid_loss, col="red", type = 'l')
}
}
\seealso{
\code{\link{chooseK_seq2seq}} for choosing \code{K} through cross-validation.

Other feature extraction methods: \code{\link{aseq2feature_seq2seq}},
  \code{\link{seq2feature_mds_large}},
  \code{\link{seq2feature_mds}},
  \code{\link{seq2feature_seq2seq}},
  \code{\link{tseq2feature_seq2seq}}
}
\concept{feature extraction methods}
