% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RFplus.R
\name{RFplus}
\alias{RFplus}
\alias{RFplus.default}
\alias{RFplus.data.table}
\title{Machine learning algorithm for fusing ground and satellite precipitation data.}
\usage{
RFplus(BD_Insitu, Cords_Insitu, Covariates, ...)

\method{RFplus}{default}(
  BD_Insitu,
  Cords_Insitu,
  Covariates,
  n_round = NULL,
  wet.day = FALSE,
  ntree = 2000,
  seed = 123,
  training = 1,
  Rain_threshold = 0.1,
  method = c("RQUANT", "QUANT", "none"),
  ratio = 15,
  save_model = FALSE,
  name_save = NULL,
  ...
)

\method{RFplus}{data.table}(
  BD_Insitu,
  Cords_Insitu,
  Covariates,
  n_round = NULL,
  wet.day = FALSE,
  ntree = 2000,
  seed = 123,
  training = 1,
  Rain_threshold = 0.1,
  method = c("RQUANT", "QUANT", "none"),
  ratio = 15,
  save_model = FALSE,
  name_save = NULL,
  ...
)
}
\arguments{
\item{BD_Insitu}{`data.table` containing the ground truth measurements (dependent variable) used to train the RFplus model.
Each column represents a ground station, and station identifiers are stored as column names. The class of `BD_Insitu`
must be `data.table`. Each row represents a time step with measurements of the corresponding station.}

\item{Cords_Insitu}{`data.table` containing metadata for the ground stations. Must include the following columns:
- `Cod`: Unique identifier for each ground station.
- `X`: Latitude of the station in UTM format.
- `Y`: Longitude of the station in UTM format.}

\item{Covariates}{A list of covariates used as independent variables in the RFplus model. Each covariate should be a
`SpatRaster` object (from the `terra` package) and can represent satellite-derived weather variables or a Digital
 Elevation Model (DEM). All covariates should have the same number of layers (bands), except for the DEM, which must have only one layer.}

\item{...}{Additional arguments to pass to the underlying methods (e.g., for model tuning or future extensions).}

\item{n_round}{Numeric indicating the number of decimal places to round the corrected values. If `n_round` is set to `NULL`, no rounding is applied.}

\item{wet.day}{Numeric value indicating the threshold for wet day correction. Values below this threshold will be set to zero.
- `wet.day = FALSE`: No correction is applied (default).
- For wet day correction, provide a numeric threshold (e.g., `wet.day = 0.1`).}

\item{ntree}{Numeric indicating the maximum number trees to grow in the Random Forest algorithm. The default value is set to 2000. This should not be set to too small a number, to ensure that every input row gets predicted at least a few times. If this value is too low, the prediction may be biased.}

\item{seed}{Integer for setting the random seed to ensure reproducibility of results (default: 123).}

\item{training}{Numerical value between 0 and 1 indicating the proportion of data used for model training. The remaining data are used for validation. Note that if you enter, for example, 0.8 it means that 80 % of the data will be used for training and 20 % for validation.
If you do not want to perform validation, set training = 1. (Default training = 1).}

\item{Rain_threshold}{Numeric value that defines the precipitation threshold for classifying rainfall events.
Precipitation values above this threshold will be considered as rainfall events, while values below it will be considered as no-rain events.
This parameter is used to calculate key performance metrics such as the Probability of Detection (POD), False Alarm Rate (FAR), and Critical Success Index (CSI),
which help assess the accuracy of rainfall event predictions.
Note: This parameter should only be set when `training` is not equal to 1, as it is needed to calculate the POD, FAR, and CSI metrics.
The default value for this parameter is 0.1.}

\item{method}{A character string specifying the quantile mapping method used for distribution adjustment. Options are:
- `"RQUANT"`: Robust quantile mapping to adjust satellite data distribution to observed data.
- `"QUANT"`: Standard quantile mapping.
- `"none"`: No distribution adjustment is applied. Only Random Forest-based bias correction and residual correction are performed.}

\item{ratio}{integer Maximum search radius (in kilometers) for the quantile mapping setting using the nearest station. (default = 15 km)}

\item{save_model}{Logical value indicating whether the corrected raster layers should be saved to disk. The default is `FALSE`.
If set to `TRUE`, make sure to set the working directory beforehand using `setwd(path)` to specify where the files should be saved.}

\item{name_save}{Character string. Base name for output file (default: NULL). The output file will be saved as "Model_RFplus.nc".
If you set a different name, make sure you do not set the ".nc" format,
as the code will internally assign it.}
}
\value{
Returns a list containing two elements:
  \item{Ensamble}{A `SpatRaster` object containing the bias-corrected layers for each time step. The number of layers
  corresponds to the number of dates for which the correction is applied. This represents the corrected satellite data adjusted for bias.}
  \item{Validation}{A data frame or similar object containing the statistical results obtained from the validation process.
  These statistics assess the performance of the bias correction applied to the satellite data.}.
}
\description{
MS-GOP is a machine learning algorithm for merging satellite-based and ground precipitation data.
It combines Random Forest for spatial prediction, residual modeling for bias correction, and quantile mapping for final adjustment, ensuring accurate precipitation estimates across different temporal scales
}
\details{
The `RFplus` method implements a three-step approach:
1. **Base Prediction**: Random Forest model is trained using satellite data and covariates.
2. **Residual Correction**: A second Random Forest model is used to correct the residuals from the base prediction.
3. **Distribution Adjustment**: Quantile mapping (QUANT or RQUANT) is applied to adjust the distribution of satellite data to match the observed data distribution.

The final result combines all three steps, correcting the biases while preserving the outliers, and improving the accuracy of satellite-derived data such as precipitation and temperature.
}
\examples{
\donttest{
# Load the libraries
library(terra)
library(data.table)

# Load the data
 data("BD_Insitu", package = "RFplus")
 data("Cords_Insitu", package = "RFplus")

# Load the covariates
Covariates <- list(
 MSWEP = terra::rast(system.file("extdata/MSWEP.nc", package = "RFplus")),
 CHIRPS = terra::rast(system.file("extdata/CHIRPS.nc", package = "RFplus")),
 DEM = terra::rast(system.file("extdata/DEM.nc", package = "RFplus"))
 )

 # Apply the RFplus bias correction model
model = RFplus(BD_Insitu, Cords_Insitu, Covariates, n_round = 1, wet.day = 0.1,
        ntree = 2000, seed = 123, training = 1, Rain_threshold = 0.1, method = "RQUANT",
        ratio = 5, save_model = FALSE, name_save = NULL)
# Visualize the results
# Precipitation results within the study area
modelo_rainfall = model$Ensamble
# Validation statistic results
metrics = model$Validation
# Note: In the above example we used 80\% of the data for training and 20\% for # model validation.
}

}
\author{
Jonnathan Augusto landi Bermeo, jonnathan.landi@outlook.com
}
