% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/overfitSC.R
\name{overfitSC}
\alias{overfitSC}
\title{Testing search.conv overfit}
\usage{
overfitSC(RR,y.list,phylo.list,s=0.25,
  nodes=NULL,state=NULL,declust=FALSE,
  aces=NULL,x1=NULL,aces.x1=NULL,cov=NULL,rootV=NULL, clus=0.5)
}
\arguments{
\item{RR}{an object produced by \code{\link{RRphylo}}.}

\item{y.list}{a list of multivariate phenotype related to the phylogenetic
trees provided as \code{phylo.list} (see Details).}

\item{phylo.list}{a list of phylogenetic trees. The phylogenies in
\code{phylo.list} can either be generated by \code{\link{resampleTree}} or
provided by the user. In this latter case, the function is meant to test
the robustness of results on alternative topologies, thus the phylogenies
must have the same species arranged differently.}

\item{s}{the percentage of tips to be cut off. It is set at 25\% by default.
If \code{phylo.list} is provided, this argument is ignored.}

\item{nodes}{the argument \code{nodes} as passed to \code{\link{search.conv}}.
Please notice, the arguments \code{nodes} and \code{state} can be indicated
at the same time.}

\item{state}{the argument \code{state} as passed to \code{\link{search.conv}}.
Please notice, the arguments \code{nodes} and \code{state} can be indicated
at the same time.}

\item{declust}{the argument \code{declust} as passed to \code{\link{search.conv}}.}

\item{aces}{if used to produce the \code{RR} object, the vector of those
ancestral character values at nodes known in advance must be specified.
Names correspond to the nodes in the tree.}

\item{x1}{the additional predictor to be specified if the RR object has been
created using an additional predictor (i.e. multiple version of
\code{\link{RRphylo}}). \code{'x1'} vector must be as long as the number of nodes
plus the number of tips of the tree, which can be obtained by running
\code{\link{RRphylo}} on the predictor as well, and taking the vector of ancestral
states and tip values to form the \code{x1}.}

\item{aces.x1}{a named vector of ancestral character values at nodes for
\code{x1}. It must be indicated if the RR object has been created using
both \code{aces} and \code{x1}. Names correspond to the nodes in the tree.}

\item{cov}{if used to produce the \code{RR} object, the covariate must be
specified. As in \code{\link{RRphylo}}, the covariate vector must be as long as
the number of nodes plus the number of tips of the tree, which can be
obtained by running \code{\link{RRphylo}} on the covariate as well, and taking the
vector of ancestral states and tip values to form the covariate.}

\item{rootV}{if used to produce the \code{RR} object, the phenotypic value at
the tree root must be specified.}

\item{clus}{the proportion of clusters to be used in parallel computing. To
run the single-threaded version of \code{overfitSC} set \code{clus} = 0.}
}
\value{
The function returns a 'RRphyloList' object containing:

\strong{$RR.list} a 'RRphyloList' including the results of each
  \code{\link{RRphylo}} performed within \code{overfitSC}

\strong{$SCnode.list} a 'RRphyloList' including the results of each
  \code{search.conv - clade condition} performed within \code{overfitSC}

\strong{$SCstate.list} a 'RRphyloList' including the results of each
  \code{search.conv - state condition} performed within \code{overfitSC}

\strong{$conv.results} a list including results for
  \code{\link{search.conv}} performed under \code{clade} and \code{state}
  conditions. If a node pair is specified within \code{conv.args}, the
  \code{$clade} object contains the percentage of simulations producing
  significant p-values for convergence between the clades, and the proportion
  of tested trees (i.e. where the clades identity was preserved; always 1 if
  no \code{phylo.list} is supplied). If a state vector is supplied within
  \code{conv.args}, the object \code{$state} contains the percentage of
  simulations producing significant p-values for convergence within (single
  state) or between states (multiple states).

The output always has an attribute "Call" which returns an unevaluated call to the function.
}
\description{
Testing the robustness of \code{\link{search.conv}}
  (\cite{Castiglione et al. 2019b}) results to sampling effects and
  phylogenetic uncertainty.
}
\details{
Methods using a large number of parameters risk being overfit. This
  usually translates in poor fitting with data and trees other than the those
  originally used. With \code{\link{RRphylo}} methods this risk is usually very low.
  However, the user can assess how robust the results of \code{\link{search.conv}}
  are by running \code{\link{resampleTree}} and \code{overfitSC}. The former is used
  to subsample the tree according to a \code{s} parameter (that is the
  proportion of tips to be removed from the tree) and to alter tree topology
  by means of \code{\link{swapONE}}. Once a list of new phylogenetic trees
  (\code{phylo.list}) is generated, in case the shape data to feed to
  \code{\link{search.conv}} are reduced (e.g. via SVD), it is necessary to recompute
  data reduction, thus obtaining a list of multivariate phenotypes related to
  the phylogenetic trees (\code{y.list}). Finally, \code{overfitSC} performs
  \code{\link{RRphylo}} and \code{\link{search.conv}} on each new set of tree and data.
  Thereby, both the potential for overfit and phylogenetic uncertainty are
  accounted for straight away.


  Otherwise, a list of alternative phylogenies can be supplied to
  \code{overfitSC}. In this case subsampling and swapping arguments are
  ignored, and robustness testing is performed on the alternative topologies
  as they are. If a clade has to be tested in \code{\link{search.conv}}, the
  function scans each alternative topology searching for the corresponding
  clade. If the species within such clade on the alternative topology differ
  more than 10\% from the species within the clade in the original tree, the
  identity of the clade is considered disrupted and the test is not
  performed.
}
\examples{
\dontrun{
require(phytools)
require(Morpho)
require(ape)

cc<- 2/parallel::detectCores()

DataFelids$treefel->treefel
DataFelids$statefel->statefel
DataFelids$landfel->feldata

# perform data reduction via Procrustes superimposition (in this case) and RRphylo
procSym(feldata)->pcafel
pcafel$PCscores->PCscoresfel

RRphylo(treefel,PCscoresfel,clus=cc)->RRfelids

# apply search.conv under nodes and state condition
search.conv(RR=RRfelids, y=PCscoresfel, min.dim=5, min.dist="time38", clus=cc)->sc.clade.time

search.conv(tree=treefel, y=PCscoresfel, state=statefel, declust=TRUE, clus=cc)->sc.state

# select converging clades returned in sc.clade.time
felnods<-rbind(c(85,155),c(85,145))

## overfitSC routine

# generate a list of subsampled and swapped phylogenies to test for search.conv
# robustness. Use as reference tree the phylogeny returned by RRphylo.
# Set the nodes and the categories under testing as arguments of
# resampleTree so that it maintains no less than 5 species at least in each
# clade/state.
treefel.list<-resampleTree(RRfelids$tree,s=0.15,nodes=unique(c(felnods)),categories=statefel,
                        nsim=15,swap.si=0.1,swap.si2=0.1)

# match the original data with each subsampled-swapped phylogeny in treefel.list
#  and repeat data reduction
y.list<-lapply(treefel.list,function(k){
  treedataMatch(k,feldata)[[1]]->ynew
  procSym(ynew)$PCscores
})

# test for robustness of search.conv results by overfitSC
oSC<-overfitSC(RR=RRfelids,phylo.list=treefel.list,y.list=y.list,
               nodes = felnods,state=statefel,clus=cc)

}
}
\references{
Castiglione, S., Serio, C., Tamagnini, D., Melchionna, M.,
  Mondanaro, A., Di Febbraro, M., Profico, A., Piras, P.,Barattolo, F., &
  Raia, P. (2019b). A new, fast method to search for morphological
  convergence with shape data. \emph{PLoS ONE}, 14, e0226949.
  https://doi.org/10.1371/journal.pone.0226949
}
\seealso{
\href{../doc/search.conv.html}{\code{search.conv} vignette};
  \href{../doc/overfit.html}{\code{overfit} vignette};
  \href{../doc/Alternative-trees.html}{\code{Alternative-trees} vignette}
}
\author{
Silvia Castiglione, Giorgia Girardi, Carmela Serio
}
