subroutine calED(Dose,ltx,ndose,inltx,outDose,pars,npars,predtval,&
                 parserrors,value,mcED,method,motoiter,errorflag)
!-------------------------------------------------------------------------------------------------------------------------------
! fitting a dose-response curve and calculate equivalent dose by interpolation
!
! ndose,               input:: integer, length of Dose(Redose1, Redose2,...) or Ltx(Lx1/Tx1, Lx2/Tx2,...)
! npars,               input:: integer, model used for fitting the dose-response curve:
!                              if npars=2, a linear model of the form: y=ax+b will be fitted, where ndat>=2
!                              if npars=3, a Exponential model of the form: y=a(1-exp(-bx))+c  will be fitted, where ndat>=3
!                              if npars=4, a linear+Exponential model of the form: y=a(1-exp(-bx))+cx+d will be fitted, where ndat>=4
! dose(ndose),         input:: real values, Redose data for dose-response curve
! ltx(ndose),          input:: real values, standardlized OSL signal data for dose-response curve
! inltx(2),            input:: real values, standardlized OSL signal values that used for calculating dose values
! outDose(2),         output:: real values, calculated dose and standard errors that correlated to standardlized OSL signal
! pars(npars),  input/output:: real values, initial guess characteristical parameters for dose-response curve, being overwritten as ouput
! predtval(ndose),    output:: real values, fitted values that correspond to ltx
! parserrors(npars),  output:: real values, parameters' standard errors
! value,              output:: real value, sum of the square of residual
! motoiter,           input :: integer, numbers of Moto Carlo simulations when estimating standard error for each dose value
! method,              input:: integer, method for calculating dose's standard error, 1 for simple method; 2 for Moto Carlo method
! errorflag(2),       output:: integer, error message generated when calling subroutine SGC:
!                              1) error appears when calling subroutine lmder1, errorflag(1) will be one in (0,4,5,6,7,8,9,10), 
!                                 else errorflag(1)=123
!                              2) error appears when attempt to calculate parameters' standard errors, errorflag(2)=1, else
!                                 errorflag(2) will be 0
!
! Author:: Peng Jun, 2013.06.22
!
! Dependence:: subroutine lmfit1; subroutine interpolate; subroutine r8vec_normal
!
! References:: Roberts,H.M. and Duller,G.A.T., 2004. Standardised growth curves for optical dating of sediment
!              using multiple-grain aliquots. Radiation Measurements 38, pp. 241-252.
!
!              Duller, G.A.T., 2007. Assessing the error on equivalent dose estimates derived from single aliquot
!              regenerative dose measurements. Ancient TL 25, pp. 15-24.
!----------------------------------------------------------------------------------------------------------------------------------
  implicit none
  integer(kind=4),intent(in)::ndose
  integer(kind=4),intent(in)::npars
  integer(kind=4),intent(in)::motoiter
  integer(kind=4),intent(in)::method
  real   (kind=8),intent(out)::value
  integer(kind=4),intent(out)::errorflag(2)
  real   (kind=8),dimension(ndose),intent(in)::Dose
  real   (kind=8),dimension(ndose,2),intent(in)::ltx
  real   (kind=8),dimension(2),intent(in)::inltx
  real   (kind=8),dimension(2),intent(out)::outDose
  real   (kind=8),dimension(npars),intent(inout)::pars
  real   (kind=8),dimension(npars),intent(out)::parserrors
  real   (kind=8),dimension(ndose),intent(out)::predtval
  real   (kind=8),dimension(motoiter),intent(out)::mcED
  !
  ! local variables
  real   (kind=8),parameter::lmtol=1.0D-07                 ! toleracne for stopping iterations in subroutine lmfit1
  real   (kind=8)::minvalue                                ! minimized value in subroutine interpolate
  real   (kind=8)::maxDose                                 ! maximum Redose
  real   (kind=8)::averageErr                              ! average value for the sum of the squared residual errors
  real   (kind=8),dimension(2)::lowup                      ! low and up boundary dose values
  real   (kind=8)::lowltx,upltx                            ! low and up boundary ltx value
  integer(kind=4)::seed                                    ! seed for normal distribution random number generation
  real   (kind=8),dimension(ndose)::ranltx                 ! random Lx/Tx
  real   (kind=8),dimension(npars)::Ranpars,Ranparserrors  ! characteristical parameters for dose-response curve generated by random number
  real   (kind=8),dimension(ndose)::Ranpredtval            ! fitted values that correspond to ranltx
  real   (kind=8)::ranvalue                                ! sum of the square of residual generaged by random number
  integer(kind=4),dimension(2)::Ranerrorflag               ! errorflag generated during random fitting
  integer(kind=4)::i,j,mccount                             ! iterative count
  real   (kind=8)::sumdose,sumdose2,mcDose                 ! sum of dose, sum of the square of dose, mc based dose for 
                                                           ! each Moto Carlo based dose value
   
  !
  ! default outDose for returning 
  outDose=0.0D+00
  !
  ! call lmfit1 to get the characterized parameters for Dose-Response 
  ! curve, pars will be reset, also return parserrors,predtval,value, errorflag
   call lmfit1(Dose,ltx(:,1),ndose,pars,npars,&
               parserrors,predtval,value,&
               lmtol,errorflag)
  !
  ! check if error appears when calling lmfit1
  if(errorflag(1)/=123 .or. errorflag(2)/=0) return
  !
  ! calculate equivalent dose using inltx(1), store in outDose(1)
  maxDose=maxval(Dose)
  call interpolate(outDose(1),inltx(1),pars,npars,0.0D+00,maxDose*1.1D+00,minvalue)
  !
  ! estimate standard error for Equivalent Dose
  ! 1) method 1, simple transformation
  if(method==1) then
    averageErr=value/real(ndose,kind=8)
    lowltx=inltx(1)-sqrt( (inltx(2))**2 + averageErr )
    upltx=inltx(1)+sqrt( (inltx(2))**2 + averageErr )
    ! calculate low bounded Equivalent Dose
    call interpolate(lowup(1),lowltx,pars,npars,0.0D+00,maxDose,minvalue)
    ! calculate up bounded Equivalent Dose
    call interpolate(lowup(2),upltx,pars,npars,0.0D+00,maxDose*1.3D+00,minvalue)
    ! calculate standard error for Equivalent Dose
    outDose(2)=( lowup(2)-lowup(1) )/2.0D+00
    mcED=-99.0D+00
  ! 2) method 2, Monte Carlo iteration
  else if(method==2) then
    seed=332571951
    ! initializing McCount numbers, Sum of Mc Dose, Sum of the square of Mc Dose
    mccount=0
    sumdose=0.0D+00
    sumdose2=0.0D+00
    i=0
    ! Moto Carlo simulations
    do 
      ! set Ranpars to be pars
      Ranpars=pars
      ! generate random ltx values: mean=ltx(1,j), sd=ltx(2,j)
      do j=1, ndose
        call r8vec_normal (1, ltx(j,1), ltx(j,2), seed, ranltx(j))
      end do
      ! fitting x(Dose) .VS. random ltx
      call lmfit1(Dose,ranltx,ndose,Ranpars,npars,&
                  Ranparserrors,Ranpredtval,Ranvalue,&
                  lmtol,Ranerrorflag)
      ! error check of lmfit1
      if(Ranerrorflag(1)==123 .and. Ranerrorflag(2)==0 ) then
        i=i+1
        ! generate random values mcED(i): mean=inltx(1), sd=inltx(2)
        call r8vec_normal (1, inltx(1), inltx(2), seed, mcED(i))
        ! interpolation using Ranpars and random values mcED(i)
        call interpolate(mcDose,mcED(i),Ranpars,npars,0.0D+00,maxDose*1.5D+00,minvalue)
        mcED(i)=mcDose
        ! accumulate McCount, SumDose, Sum of squared Dose
        mccount=mccount+1
        sumdose=sumdose+mcDose
        sumdose2=sumdose2+mcDose**2
      end if
      ! checking terminate
      if(i==motoiter) exit
    end do
    ! calculate standard error for Equivalent Dose
    outDose(2)=sqrt( (real(mccount,kind=8)*sumdose2-sumdose**2)/ &
                      real(mccount,kind=8)/real(mccount-1,kind=8) )
  end if
  !
  return
  !
end subroutine calED
