#' Plot SCIntRuler
#'
#' @param fullcluster A list of clusters that generated by the function GetCluster.
#' @param seuratlist A list of Seurat objects, usually can be got by SplitObject().
#' @param testres Result from function PermTest()
#' @param legendtitle Title of legend, default to be NULL
#' @param title Title of figure, default to be NULL
#'
#' @return  A ggplot2 object
#' @export
#'
#' @examples
#' \donttest{
#' data(sim_data_sce)
#' data(sim_result)
#' sim_data <- SCEtoSeurat(sim_data_sce)
#' seuratlist <- Seurat::SplitObject(sim_data, split.by = "Study")
#' PlotSCIR(sim_result[[1]], seuratlist, sim_result[[4]])
#' }

PlotSCIR <- function(fullcluster,seuratlist,testres,legendtitle= NULL,title = NULL) {

  stopifnot(exprs = {
    is.list(fullcluster)
    is.list(seuratlist)
    is.list(testres)
  })


  SCout <- SummCluster(fullcluster)
  refindClust <- SCout$refindClust
  nbroad <- SCout$nbroad
  allrevDiff <- testres$allrevDiff
  allP <- testres$allP


  Sample <- c()
  for (i in seq_along(allP)){
    l <- length(allP[[i]])
    ll <- rep(names(seuratlist)[i],l)
    Sample <- c(Sample,ll)
  }

  res <- data.frame("P value"=unlist(allP), revDiff = na.omit(c(allrevDiff)), Sample)
  ratio <- dim(subset(res, P.value <0.1 & revDiff >0))[1]/length(Sample)

  plot <- ggplot2::ggplot(res) +
    ggplot2::geom_point(ggplot2::aes(revDiff,`P.value`,color = factor(Sample)), size = 3, alpha = 0.7) +
    ggplot2::theme_test()+
    ggplot2::scale_color_brewer(palette= "Dark2") +
    ggplot2::geom_hline(yintercept=0.1, linetype="dashed", color = "blue") +
    ggplot2::geom_vline(xintercept=0, linetype="dashed", color = "red")+
    ggplot2::ylab("P-value") + ggplot2::xlab("Within-between Cluster Relative Distance")+
    ggplot2::annotate("text", x = max(allrevDiff, na.rm = TRUE), y = max(unlist(allP)/2, na.rm = TRUE),
             label= paste("SCIntRuler =",round(ratio,2)),hjust=1, fontface="bold")+
    ggplot2::labs(color= legendtitle) +
    ggplot2::ggtitle(title)+
    ggplot2::guides(colour = ggplot2::guide_legend(override.aes = list(size=2)))

  return(plot)



}
