% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semTools.R
\name{predictSink}
\alias{predictSink}
\title{SEM-based out-of-sample predictions}
\usage{
predictSink(
  object,
  newdata = NULL,
  K_fold = 5,
  source = FALSE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{object}{An object, as that created by the function \code{SEMrun()} with the
argument \code{fit} set to \code{fit = 0} or \code{fit = 1}.}

\item{newdata}{An optional matrix with rows corresponding to subjects, and
columns to graph nodes (variables). If \code{object$fit} is a model with the
group variable (\code{fit = 1}), the first column of newdata must be the
new group binary vector (0=control, 1=case). As a default \code{newdata = NULL}, 
meaning that the K-fold cross validation is applied on the \code{object$data}.
Conversely, if the argument \code{newdata} is specified, this matrix will be
used for testing (out-of-sample predictions) and \code{object$data} will be
used for training.}

\item{K_fold}{The number of subsets (folds) into which the data will be 
partitioned for performing K-fold cross-validation. The model is refit K times, 
each time leaving out one of the K folds (default, K_fold=5). If the argument 
\code{newdata} is specified, the K-fold cross validation will not be done.}

\item{source}{A logical value. If FALSE (default), the predictor variables (x) 
include source and mediators. If TRUE, x includes only the source nodes.}

\item{verbose}{A logical value. If FALSE (default), the processed graph 
will not be plotted to screen.}

\item{...}{Currently ignored.}
}
\value{
A list of 3 objects:
\enumerate{
\item "yobs", the matrix of observed continuous values of sink nodes based on
out-of-bag samples. 
\item "yhat", the matrix of continuous predicted values of sink nodes ased on
out-of-bag samples.
\item "PE", vector of the prediction error equal to the Root Mean Squared Error
(RMSE) for each out-of-bag sink prediction. The first value of PE is the total
RMSE, where we sum over all sink nodes.
}
}
\description{
Given the values of (observed) x-variables in a structural equation
model, this function may be used to predict the values of (observed) y-variables.
Response variables (y) represent sink nodes, and predictor variables (x)
might consist of either (i) just source nodes or (ii) source and mediators from
the fitted graph structure.
}
\details{
The function uses a SEM-based predictive approach (Rooij et al., 2022)
to produce predictions while accounting for the given graph structure. Predictions
(for y given x) are based on the (joint y and x) model-implied variance-covariance
(Sigma) matrix and mean vector (Mu) of the fitted SEM, and the standard expression
for the conditional mean of a multivariate normal distribution. Thus, the structure
described in the SEM is taken into consideration, which differs from ordinary
least squares (OLS) regression. Note that if the model is saturated (and hence
df = 0), or when \code{source = TRUE}, i.e., the set of predictors will include only
the source nodes, the SEM-based predictions are identical or similar to OLS
predictions.
}
\examples{

# load ALS data
ig<- alsData$graph
X<- alsData$exprs
X<- transformData(X)$data
group<- alsData$group

#...with train-test (0.8-0.2) samples
set.seed(1)
train<- sample(1:nrow(X), 0.8*nrow(X))

# SEM fitting
#sem0<- SEMrun(ig, X[train,], algo="lavaan", SE="none")
#sem0<- SEMrun(ig, X[train,], algo="ricf", n_rep=0)
sem0<- SEMrun(ig, X[train,], algo="cggm")

# predictors, source+mediator variables
res1<- predictSink(sem0, newdata=X[-train,]) 
print(res1$PE)

# predictors, source variables
res2<- predictSink(sem0, newdata=X[-train,], source=TRUE) 
print(res2$PE)

#...with 5-fold cross-validation samples
set.seed(2)

# SEM fitting
#sem0<- SEMrun(ig, X, algo="lavaan", SE="none")
#sem0<- SEMrun(ig, X, algo="ricf", n_rep=0)
sem0<- SEMrun(ig, X, algo="cggm")

# predictors, source+mediator variables	
res3<- predictSink(sem0, K_fold = 5, verbose=TRUE)
print(res3$PE)

# predictors, source variables
res4<- predictSink(sem0, K_fold = 5, source=TRUE, verbose=TRUE) 
print(res4$PE)

\dontrun{

#...with 10-fold cross-validation samples and 10-iterations

# SEM fitting
#sem1<- SEMrun(ig, X, group, algo="lavaan", SE="none")
#sem1<- SEMrun(ig, X, group, algo="ricf", n_rep=0)
sem1<- SEMrun(ig, X, group, algo="cggm")

# predictors, source+mediator+group variables
res<- NULL
for (r in 1:10) {
	set.seed(r)
	cat("rep = ", r, "\n")
	resr<- predictSink(sem1, K_fold = 10)
	res<- rbind(res, resr$PE)
}
res
apply(res, 2, mean)

}

}
\references{
de Rooij M, Karch JD, Fokkema M, Bakk Z, Pratiwi BC, and Kelderman H
(2023). SEM-Based Out-of-Sample Predictions, Structural Equation Modeling:
A Multidisciplinary Journal, 30:1, 132-148
<https://doi.org/10.1080/10705511.2022.2061494>
}
\author{
Mario Grassi \email{mario.grassi@unipv.it}
}
