% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GaussSuppression.R
\name{GaussSuppression}
\alias{GaussSuppression}
\title{Secondary suppression by Gaussian elimination}
\usage{
GaussSuppression(
  x,
  candidates = 1:ncol(x),
  primary = NULL,
  forced = NULL,
  hidden = NULL,
  singleton = rep(FALSE, nrow(x)),
  singletonMethod = "anySum",
  printInc = TRUE,
  tolGauss = (.Machine$double.eps)^(1/2),
  whenEmptySuppressed = warning,
  whenEmptyUnsuppressed = message,
  whenPrimaryForced = warning,
  removeDuplicated = TRUE,
  iFunction = GaussIterationFunction,
  iWait = Inf,
  xExtraPrimary = NULL,
  unsafeAsNegative = FALSE,
  ...
)
}
\arguments{
\item{x}{Matrix that relates cells to be published or suppressed to inner cells. yPublish = crossprod(x,yInner)}

\item{candidates}{Indices of candidates for secondary suppression}

\item{primary}{Indices of primary suppressed cells}

\item{forced}{Indices forced to be not suppressed. \code{forced} has precedence over \code{primary}. See \code{whenPrimaryForced} below.}

\item{hidden}{Indices to be removed from the above \code{candidates} input (see details)}

\item{singleton}{Logical or integer vector of length \code{nrow(x)} specifying inner cells for singleton handling.
Normally, for frequency tables, this means cells with 1s when 0s are non-suppressed and cells with 0s when 0s are suppressed.
For some singleton methods, integer values representing the unique magnitude table contributors are needed.
For all other singleton methods, only the values after conversion with \code{as.logical} matter.}

\item{singletonMethod}{Method for handling the problem of singletons and zeros:
\code{"anySum"} (default), \code{"anySumNOTprimary"}, \code{"subSum"}, \code{"subSpace"}, \code{"sub2Sum"}, \code{"none"}
or a \code{\link{NumSingleton}} method (see details).}

\item{printInc}{Printing "..." to console when TRUE}

\item{tolGauss}{A tolerance parameter for sparse Gaussian elimination and linear dependency. This parameter is used only in cases where integer calculation cannot be used.}

\item{whenEmptySuppressed}{Function to be called when empty input to primary suppressed cells is problematic. Supply NULL to do nothing.}

\item{whenEmptyUnsuppressed}{Function to be called when empty input to candidate cells may be problematic. Supply NULL to do nothing.}

\item{whenPrimaryForced}{Function to be called if any forced cells are primary suppressed (suppression will be ignored). Supply NULL to do nothing.
The same function will also be called when there are forced cells marked as singletons (will be ignored).}

\item{removeDuplicated}{Whether to remove duplicated columns in \code{x} before running the main algorithm.}

\item{iFunction}{A function to be called during the iterations. See the default function, \code{\link{GaussIterationFunction}}, for description of parameters.}

\item{iWait}{The minimum number of seconds between each call to \code{iFunction}.
Whenever \code{iWait<Inf}, \code{iFunction} will also be called after last iteration.}

\item{xExtraPrimary}{Extra x-matrix that defines extra primary suppressed cells in addition to those defined by other inputs.}

\item{unsafeAsNegative}{When \code{TRUE}, unsafe primary cells due to forced cells are included in the output vector as negative indices.}

\item{...}{Extra unused parameters}
}
\value{
Secondary suppression indices
}
\description{
Sequentially the secondary suppression candidates (columns in x) are used to reduce the x-matrix by Gaussian elimination.
Candidates who completely eliminate one or more primary suppressed cells (columns in x) are omitted and made secondary suppressed.
This ensures that the primary suppressed cells do not depend linearly on the non-suppressed cells.
How to order the input candidates is an important choice.
The singleton problem and the related problem of zeros are also handled.
}
\details{
It is possible to specify too many (all) indices as \code{candidates}.
Indices specified as \code{primary} or \code{hidded} will be removed.
Hidden indices (not candidates or primary) refer to cells that will not be published, but do not need protection.
\itemize{
\item \strong{Singleton methods for frequency tables:}
All singleton methods, except \code{"sub2Sum"} and the \code{\link{NumSingleton}} methods, have been implemented with frequency tables in mind.
The singleton method \code{"subSum"} makes new imaginary primary suppressed cells, which are the sum of the singletons
within each group. The \code{"subSpace"} method is conservative and ignores the singleton dimensions when looking for
linear dependency. The default method, \code{"anySum"}, is between the other two. Instead of making imaginary cells of
sums within groups, the aim is to handle all possible sums, also across groups. In addition, \code{"subSumSpace"}  and
\code{"subSumAny"} are possible methods, primarily for testing. These methods are similar to \code{"subSpace"} and \code{"anySum"},
and additional cells are created as in \code{"subSum"}. It is believed that the extra cells are redundant.
Note that in order to give information about unsafe cells, \code{"anySum"}  is internally changed to \code{"subSumAny"} when there are forced cells.
All the above methods assume that any published singletons are primary suppressed.
When this is not the case, \code{"anySumNOTprimary"} must be used.
\item \strong{Singleton methods for magnitude tables:}
The singleton method \code{"sub2Sum"} makes new imaginary primary suppressed cells, which are the sum of two inner cells.
This is done when a group contains exactly two primary suppressed inner cells provided that at least one of them is singleton.
This was the first method implemented. Other magnitude methods follow the coding according to \code{\link{NumSingleton}}.
The \code{"sub2Sum"} method is equivalent to \code{"numFFT"}.
Also note that \code{"num"}, \code{"numFFF"} and \code{"numFTF"} are equivalent to \code{"none"}.
\item \strong{Combined:}
For advanced use, \code{singleton} can be a two-element list with names \code{"freq"} and \code{"num"}.
Then \code{singletonMethod} must be a corresponding named two-element vector.
For example: \code{singletonMethod = c(freq = "anySumNOTprimary", num = "sub2Sum")}
}
}
\examples{
# Input data
df <- data.frame(values = c(1, 1, 1, 5, 5, 9, 9, 9, 9, 9, 0, 0, 0, 7, 7), 
                 var1 = rep(1:3, each = 5), 
                 var2 = c("A", "B", "C", "D", "E"), stringsAsFactors = FALSE)

# Make output data frame and x 
fs <- FormulaSums(df, values ~ var1 * var2, crossTable = TRUE, makeModelMatrix = TRUE)
x <- fs$modelMatrix
datF <- data.frame(fs$crossTable, values = as.vector(fs$allSums))

# Add primary suppression 
datF$primary <- datF$values
datF$primary[datF$values < 5 & datF$values > 0] <- NA
datF$suppressedA <- datF$primary
datF$suppressedB <- datF$primary
datF$suppressedC <- datF$primary

# zero secondary suppressed
datF$suppressedA[GaussSuppression(x, primary = is.na(datF$primary))] <- NA

# zero not secondary suppressed by first in ordering
datF$suppressedB[GaussSuppression(x, c(which(datF$values == 0), which(datF$values > 0)), 
                            primary = is.na(datF$primary))] <- NA

# with singleton
datF$suppressedC[GaussSuppression(x, c(which(datF$values == 0), which(datF$values > 0)), 
                            primary = is.na(datF$primary), singleton = df$values == 1)] <- NA

datF

}
