\name{modelSim}
\alias{modelSim}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Stochastic volatility with correlated return and volatility factor jumps data simulation function
}
\description{
The \code{modelSim} function generates returns and variances for a wide class of stochastic volatility models.
}
\usage{
modelSim(t, mu = 0.038, kappa = 3.689, theta = 0.032, sigma = 0.446,
  rho = -0.745, omega = 5.125, delta = 0.03, alpha = -0.014,
  rho_z = -1.809, nu = 0.004, p = 0.01, phi = 0.965,
  model = "Heston", h = 1/252,
  mu_x, mu_y, sigma_x, sigma_y,
  j_x = FALSE, j_y = FALSE, jump_dist = rpois,
  mu_x_params, mu_y_params, sigma_x_params, sigma_y_params,
  jump_params = 0)
}
\arguments{
 \item{t}{Number of observations to be simulated.}
 \item{mu}{Annual expected rate of return.}
 \item{kappa}{Variance rate of mean reversion.}
 \item{theta}{Unconditional mean variance.}
 \item{sigma}{Volatility of the variance.}
 \item{rho}{Correlation between the return and the variances' noise.}
 \item{omega}{Jump arrival intensity for models with Poisson jumps.}
 \item{delta}{Standard deviation of return jumps.}
 \item{alpha}{Mean of return jumps.}
 \item{rho_z}{Pseudo-correlation parameter between return and variance jumps.}
 \item{nu}{Mean for variance jumps.}
 \item{p}{Jump probability for models with Bernoulli jumps.}
 \item{phi}{Volatility persistence parameter.}
  \item{model}{Model from which the data is generated. The options are \cr \code{"DuffiePanSingleton"}, \code{"Bates"}, \code{"Heston"}, \code{"PittMalikDoucet"}, \cr \code{"TaylorWithLeverage"}, \code{"Taylor"}, and \code{"Custom"}. If \code{model = "Custom"}, users should pass the drift functions (i.e., \code{mu_x} and \code{mu_y}), the diffusion functions (i.e., \code{sigma_x} and \code{sigma_y}), and the jump distribution, (i.e., \code{jump_dist}) as well as their parameters to the \code{DNF} function. See Examples.}
  \item{h}{The time interval between observations (i.e., \code{h = 1/252} for daily data).}
  \item{mu_x}{Function for variance drift (to be used with a custom model).}
  \item{mu_y}{Function for returns drift (to be used with a custom model).}
  \item{sigma_x}{Function for variance diffusion (to be used with a custom model).}
  \item{sigma_y}{Function for returns diffusion (to be used with a custom model).}
  \item{j_x}{If \code{TRUE}, exponentially distributed (with mean \code{nu}) volatility jumps will be included in the custom model.}
  \item{j_y}{If \code{TRUE}, normally distributed (with mean \code{alpha}, and standard deviation \code{delta}) returns jumps will be included in the custom model.}
  \item{jump_dist}{Probability mass function used to generate return or volatility jumps at each timestep (if both types of jumps are in the model, they are assumed to occur simulaneously).}
  \item{jump_params}{List of parameters to be used as arguments in the \code{jump_dist} function (parameters should be listed in order that \code{jump_dist} uses them).}
  \item{mu_x_params}{List of parameters to be used as arguments in the \code{mu_x} function (parameters should be listed in order that \code{mu_x} uses them).}
  \item{mu_y_params}{List of parameters to be used as arguments in the \code{mu_y} function (parameters should be listed in order that \code{mu_y} uses them).}
  \item{sigma_x_params}{List of parameters to be used as arguments in the \code{sigma_x} function (parameters should be listed in order that \code{sigma_x} uses them).}
  \item{sigma_y_params}{List of parameters to be used as arguments in the \code{sigma_y} function (parameters should be listed in order that \code{sigma_y} uses them).}
}
\value{
\item{Variances}{Vector of the instantaneous volatility factor values generated by the \code{modelSim} function.}
\item{Returns}{Vector of the returns generated by the \code{modelSim} function.}
}
\examples{
set.seed(1)
# Generate 250 returns from the DuffiePanSingleton model
DuffiePanSingleton_sim = modelSim(t = 250, model = 'DuffiePanSingleton')

# Plot the volatility factor and returns that were generated
plot(DuffiePanSingleton_sim$volatility_factor, type = 'l',
  main = 'DuffiePanSingleton Model Simulated Volatility Factor', ylab = 'Volatility Factor')
 
plot(DuffiePanSingleton_sim$returns, type = 'l',
  main = 'DuffiePanSingleton Model Simulated Returns', ylab = 'Returns')
 
# Generate 250 steps from a custom model
# Set parameters
kappa <- 100; theta <- 0.05; sigma <- 2.3; h <- 1/252 ; mu <- 0.04
rho <- -0.8; omega <- 5; alpha <- -0.025; nu <- 0.01; rho_z <- -1; delta <- 0.025
# Jump compensator
alpha_bar <- exp(alpha + 0.5 * delta^2) / (1 - rho_z * nu) - 1

# Define returns drift and diffusion functions
# Returns drift and diffusion
mu_y <- function(x, mu, alpha_bar, omega, h){
  return(h * (mu - x / 2 - alpha_bar * omega))
}
mu_y_params <- list(mu, alpha_bar, omega , h)
sigma_y <- function(x, h, sigma){
  return(sigma * sqrt(h) * pmax(x,0))
}
sigma_y_params <- list(h, sigma)

# Volatility factor drift and diffusion functions
mu_x <- function(x, h, kappa, theta){
  return(x + h * kappa * pmax(0,x) * (theta - pmax(0,x)))
}
mu_x_params <- list(h, kappa, theta)

sigma_x <- function(x, sigma, h){
  return(sigma * sqrt(h) * pmax(0,x))
}
sigma_x_params <- list(sigma, h)

# Include simultaneous return and volatility factor jumps
# based on the Poisson distribution for jump times
j_x <- TRUE; j_y <- TRUE
jump_dist <- rpois
jump_params <- list(omega * h)

custom <- modelSim(t = 250, model = "Custom", mu_x = mu_x, mu_y = mu_y,
  sigma_x = sigma_x, sigma_y = sigma_y,
  mu_x_params = mu_x_params, mu_y_params = mu_y_params,
  sigma_x_params = sigma_x_params, sigma_y_params = sigma_y_params,
  jump_dist = jump_dist, jump_params = jump_params,
  j_y = j_y, j_x = j_x, nu = nu)
  
plot(custom$volatility_factor, type = 'l',
  main = 'Custom Model Simulated Volatility Factor', ylab = 'Volatility Factor')
plot(custom$returns, type = 'l',
  main = 'Custom Model Simulated Returns', ylab = 'Returns')
}
