% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_corr_y.R
\name{calc_corr_y}
\alias{calc_corr_y}
\title{Calculate Expected Correlation Matrix of Outcomes (Y) for Correlated Systems of Continuous Variables}
\usage{
calc_corr_y(betas = NULL, corr.x = list(), corr.e = NULL, vars = list(),
  mix_pis = list(), mix_mus = list(), mix_sigmas = list(),
  error_type = c("non_mix", "mix"))
}
\arguments{
\item{betas}{a matrix of the slope coefficients calculated with \code{\link[SimRepeat]{calc_betas}}, rows represent the outcomes}

\item{corr.x}{list of length \code{M}, each component a list of length \code{M}; \code{corr.x[[p]][[q]]} is matrix of correlations
for independent variables in equations p (\eqn{X_{(pj)}} for outcome \eqn{Y_p}) and q (\eqn{X_{(qj)}} for outcome \eqn{Y_q});
if p = q, \code{corr.x[[p]][[q]]} is a correlation matrix with \code{nrow(corr.x[[p]][[q]])} = # \eqn{X_{(pj)}} for outcome \eqn{Y_p};
if p != q, \code{corr.x[[p]][[q]]} is a non-symmetric matrix of correlations where rows correspond to covariates for \eqn{Y_p}
so that \code{nrow(corr.x[[p]][[q]])} = # \eqn{X_{(pj)}} for outcome \eqn{Y_p} and
columns correspond to covariates for \eqn{Y_q} so that \code{ncol(corr.x[[p]][[q]])} = # \eqn{X_{(qj)}} for outcome \eqn{Y_q};
order is 1st continuous non-mixture and 2nd components of continuous mixture variables}

\item{corr.e}{correlation matrix for continuous non-mixture or components of mixture error terms}

\item{vars}{a list of same length as \code{corr.x} of vectors of variances for \eqn{X_{(pj)}, E}; E term should be last;
order should be the same as in \code{corr.x}}

\item{mix_pis}{a list of same length as \code{corr.x}, where \code{mix_pis[[p]][[j]]} is a vector of mixing probabilities for \eqn{X_{mix(pj)}} that sum to 1,
the j-th mixture covariate for outcome \eqn{Y_p}; the last element of \code{mix_pis[[p]]} is for \eqn{E_p} (if \code{error_type = "mix"});
if \eqn{Y_p} has no mixture variables, use \code{mix_pis[[p]] = NULL}}

\item{mix_mus}{a list of same length as \code{corr.x}, where \code{mix_mus[[p]][[j]]} is a vector of means for \eqn{X_{mix(pj)}},
the j-th mixture covariate for outcome \eqn{Y_p}; the last element of \code{mix_mus[[p]]} is for \eqn{E_p} (if \code{error_type = "mix"});
if \eqn{Y_p} has no mixture variables, use \code{mix_mus[[p]] = NULL}}

\item{mix_sigmas}{a list of same length as \code{corr.x}, where \code{mix_sigmas[[p]][[j]]} is a vector of standard deviations for \eqn{X_{mix(pj)}},
the j-th mixture covariate for outcome \eqn{Y_p}; the last element of \code{mix_sigmas[[p]]} is for \eqn{E_p} (if \code{error_type = "mix"});
if \eqn{Y_p} has no mixture variables, use \code{mix_sigmas[[p]] = NULL}}

\item{error_type}{"non_mix" if all error terms have continuous non-mixture distributions, "mix" if all error terms have continuous mixture distributions,
defaults to "non_mix"}
}
\value{
\code{corr.y} the correlation matrix for the outcomes \eqn{Y}
}
\description{
This function calculates the expected correlation matrix for outcomes (Y) in a correlated system of continuous variables.
    This system is generated with \code{\link[SimRepeat]{nonnormsys}} using the techniques of Headrick and Beasley (\doi{10.1081/SAC-120028431}).
    These correlations are determined based on the beta (slope) coefficients calculated with \code{\link[SimRepeat]{calc_betas}}, the correlations
    between independent variables \eqn{X_{(pj)}} for a given outcome \eqn{Y_p}, for \code{p = 1, ..., M}, the
    correlations between error terms, and the variances.  The result can be used to compare the simulated correlation matrix to the theoretical correlation matrix.
    If there are continuous mixture variables and the betas are specified in terms of non-mixture and mixture variables
    and/or \code{error_type = "mix"}, then the correlations in \code{corr.x} and/or \code{corr.e} will be calculated in terms of non-mixture and mixture variables using
    \code{\link[SimCorrMix]{rho_M1M2}} and \code{\link[SimCorrMix]{rho_M1Y}}.  In this case, the dimensions of the matrices in \code{corr.x} should not
    match the number of columns of \code{betas}.  The vignette \strong{Theory and Equations for Correlated Systems of Continuous Variables}
    gives the equations, and the vignette \strong{Correlated Systems of Statistical Equations with Non-Mixture and Mixture Continuous
    Variables} gives examples.  There are also vignettes in \code{\link[SimCorrMix]{SimCorrMix}} which provide more details on continuous
    non-mixture and mixture variables.
}
\examples{
# Example: system of three equations for 2 independent variables, where each
# error term has unit variance, from Headrick & Beasley (2002)
corr.yx <- list(matrix(c(0.4, 0.4), 1), matrix(c(0.5, 0.5), 1),
  matrix(c(0.6, 0.6), 1))
corr.x <- list()
corr.x[[1]] <- corr.x[[2]] <- corr.x[[3]] <- list()
corr.x[[1]][[1]] <- matrix(c(1, 0.1, 0.1, 1), 2, 2)
corr.x[[1]][[2]] <- matrix(c(0.1974318, 0.1859656, 0.1879483, 0.1858601),
  2, 2, byrow = TRUE)
corr.x[[1]][[3]] <- matrix(c(0.2873190, 0.2589830, 0.2682057, 0.2589542),
  2, 2, byrow = TRUE)
corr.x[[2]][[1]] <- t(corr.x[[1]][[2]])
corr.x[[2]][[2]] <- matrix(c(1, 0.35, 0.35, 1), 2, 2)
corr.x[[2]][[3]] <- matrix(c(0.5723303, 0.4883054, 0.5004441, 0.4841808),
  2, 2, byrow = TRUE)
corr.x[[3]][[1]] <- t(corr.x[[1]][[3]])
corr.x[[3]][[2]] <- t(corr.x[[2]][[3]])
corr.x[[3]][[3]] <- matrix(c(1, 0.7, 0.7, 1), 2, 2)
corr.e <- matrix(0.4, nrow = 3, ncol = 3)
diag(corr.e) <- 1
vars <- list(rep(1, 3), rep(1, 3), rep(1, 3))
betas <- calc_betas(corr.yx, corr.x, vars)
calc_corr_y(betas, corr.x, corr.e, vars)

}
\references{
Headrick TC, Beasley TM (2004).  A Method for Simulating Correlated Non-Normal Systems of Linear Statistical Equations.
    Communications in Statistics - Simulation and Computation, 33(1).  \doi{10.1081/SAC-120028431}
}
\seealso{
\code{\link[SimRepeat]{nonnormsys}}, \code{\link[SimRepeat]{calc_betas}}, \code{\link[SimCorrMix]{rho_M1M2}}, \code{\link[SimCorrMix]{rho_M1Y}}
}
\keyword{Beasley}
\keyword{Headrick}
\keyword{continuous}
\keyword{mixture}
