\name{besag.newell}
\alias{besag.newell}
\title{
	Besag-Newell Cluster Detection Method
}

\description{
Besag-Newell cluster detection method.  There are differences with the original paper and our implementation:
\itemize{
\item we base our analysis on \eqn{k} cases, rather than \eqn{k} \emph{other} cases as prescribed in the paper.
\item we do not subtract 1 from the \emph{accumulated numbers of other cases} and \emph{accumulated numbers of others at risk}, as was prescribed in the paper to discount selection bias
\item \eqn{M} is the total number of areas included, not the number of additional areas included.  i.e. \eqn{M} starts at 1, not 0.
\item \eqn{p}-values are not based on the original value of \eqn{k}, rather the actual number of cases observed until we view \eqn{k} or more cases.  Ex:  if \eqn{k = 10}, but as we consider neighbors we encounter 1, 2, 9 then 12 cases, we base our \eqn{p}-values on \eqn{k=12}
\item we do not provide a Monte-Carlo simulated \eqn{R}:  the number of tests that attain significance at a fixed level \eqn{\alpha}
}
The first two and last differences are because we view the testing on an area-by-area level, rather than a case-by-case level.
}
\usage{
	besag.newell(geo, population, cases, expected.cases=NULL, k, alpha.level)
}
\arguments{
  \item{geo}{an \code{n x 2} table of the (x,y)-coordinates of the area centroids}
  \item{cases}{aggregated case counts for all \code{n} areas}
  \item{population}{aggregated population counts for all \code{n} areas}
  \item{expected.cases}{expected numbers of disease for all \code{n} areas}
  \item{k}{number of cases to consider}
  \item{alpha.level}{\eqn{\alpha}-level threshold used to declare significance}
}
\details{
For the \code{population} and \code{cases} tables, the rows are bunched by areas first, and then for each area, the counts for each strata are listed.  It is important that the tables are balanced:  the strata information are in the same order for each area, and counts for each area/strata combination appear exactly once (even if zero). 
}
\value{
List containing
  \item{clusters}{information on all clusters that are \eqn{\alpha}-level significant, in decreasing order of the \eqn{p}-value}
  \item{p.values}{for each of the \eqn{n} areas, \eqn{p}-values of each cluster of size at least \eqn{k}}
  \item{m.values}{for each of the \eqn{n} areas, the number of areas need to observe at least \eqn{k} cases}
  \item{observed.k.values}{based on \code{m.values}, the actual number of cases used to compute the \eqn{p}-values}
} 
\note{
The \code{clusters} list elements are themselves lists reporting:\cr\cr
\tabular{ll}{
  \code{location.IDs.included} \tab ID's of areas in cluster, in order of distance\cr
  \code{population} \tab population of cluster\cr
  \code{number.of.cases} \tab number of cases in cluster\cr
  \code{expected.cases} \tab expected number of cases in cluster\cr
  \code{SMR} \tab estimated SMR of cluster\cr
  \code{p.value} \tab \eqn{p}-value\cr
}
}
\references{
Besag J. and Newell J. (1991)
The Detection of Clusters in Rare Diseases \emph{Journal of the Royal Statistical Society. Series A (Statistics in Society)}, \bold{154}, 143--155
}
\author{Albert Y. Kim}
\seealso{\code{\link{pennLC}}, \code{\link{expected}}}
\examples{
## Load Pennsylvania Lung Cancer Data
data(pennLC)
data <- pennLC$data

## Process geographical information and convert to grid
geo <- pennLC$geo[,2:3]
geo <- latlong2grid(geo)

## Get aggregated counts of population and cases for each county
population <- tapply(data$population,data$county,sum)
cases <- tapply(data$cases,data$county,sum)

## Based on the 16 strata levels, computed expected numbers of disease
n.strata <- 16
expected.cases <- expected(data$population, data$cases, n.strata)

## Set Parameters
k <- 1250
alpha.level <- 0.05

# not controlling for stratas
results <- besag.newell(geo, population, cases, expected.cases=NULL, k, alpha.level)

# controlling for stratas
results <- besag.newell(geo, population, cases, expected.cases, k, alpha.level)
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ file }
