% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wlsPower.R
\name{wlsPower}
\alias{wlsPower}
\title{Compute power via weighted least squares}
\usage{
wlsPower(
  Cl = NULL,
  timepoints = NULL,
  DesMat = NULL,
  trtDelay = NULL,
  incomplete = NULL,
  timeAdjust = "factor",
  period = NULL,
  dsntype = "SWD",
  mu0,
  mu1,
  marginal_mu = FALSE,
  sigma = NULL,
  tau = NULL,
  eta = NULL,
  AR = NULL,
  rho = NULL,
  gamma = NULL,
  psi = NULL,
  alpha_0_1_2 = NULL,
  CovMat = NULL,
  N = NULL,
  power = NULL,
  family = "gaussian",
  N_range = c(1, 1000),
  sig.level = 0.05,
  dfAdjust = "none",
  INDIV_LVL = FALSE,
  verbose = 1
)
}
\arguments{
\item{Cl}{integer (vector), number of clusters per sequence group (in SWD),
or number in control and intervention (in parallel designs)}

\item{timepoints}{numeric (scalar or vector), number of timepoints (periods).
If design is swd, timepoints defaults to length(Cl)+1.
Defaults to 1 for parallel designs.}

\item{DesMat}{Either an object of class `DesMat` or a matrix indicating the
treatment status for each cluster at each timepoint. If supplied,
`timepoints`,`Cl`,`trtDelay` are ignored.}

\item{trtDelay}{numeric (possibly vector), value(s)
between 0 and 1 specifying the proportion of intervention effect
in the first (second ... ) intervention phase.}

\item{incomplete}{integer, either a scalar (only for SWD) or a matrix.
A vector defines the number of periods before and after the switch from
control to intervention that are observed. A matrix consists of 1's for
observed clusterperiods and 0's for unobserved clusterperiods.}

\item{timeAdjust}{character, specifies adjustment for time periods.
One of the following: "factor", "linear", "none", "periodic".
Defaults to "factor".}

\item{period}{numeric (scalar)}

\item{dsntype}{character, defines the type of design. Options are "SWD",
"parallel" and "parallel_baseline", defaults to "SWD".}

\item{mu0}{numeric (scalar), mean under control}

\item{mu1}{numeric (scalar), mean under treatment}

\item{marginal_mu}{logical. Only relevant for non-gaussian outcome.
Indicates whether mu0 and mu1 are to be interpreted as marginal prevalence
under control  and under treatment, respectively, or whether they denote
the prevalence conditional on random effects being 0
(It defaults to the latter). *(experimental!)*}

\item{sigma}{numeric, residual error of cluster means if no N given.}

\item{tau}{numeric, standard deviation of random intercepts}

\item{eta}{numeric (scalar or matrix), standard deviation of random slopes.
If `eta` is given as scalar, `trtMat` is needed as well.}

\item{AR}{numeric, vector containing up to three values, each between 0 and 1.
Defaults to NULL. It defines the AR(1)-correlation of random effects.
The first element corresponds to the cluster intercept, the second to the
treatment effect and the third to subject specific intercept.
If only one element is provided, autocorrelation of all random effects is
assumed to be the same.
*Currently not compatible with `rho`!=0 !*}

\item{rho}{numeric (scalar), correlation of `tau` and `eta`}

\item{gamma}{numeric (scalar), random time effect}

\item{psi}{numeric (scalar), random subject specific intercept.
Leads to a closed cohort setting}

\item{alpha_0_1_2}{numeric vector or list of length 2 or 3, that consists of
alpha_0, alpha_1 and alpha_2. Can be used instead of random effects to define
the correlation structure, following Li et al. (2018). When omitting alpha_2,
this describes a cross-sectional design, where alpha_0 and alpha_1 define
the intracluster correlation and cluster autocorrelation, respectively - as
defined by Hooper et al. (2016).}

\item{CovMat}{numeric, a positive-semidefinite matrix with
(#Clusters \eqn{\cdot} timepoints) rows and columns. If `CovMat` is given,
`sigma`, `tau`, `eta`, `rho`, `gamma` and `psi` as well as `alpha_0_1_2`
must be NULL.}

\item{N}{numeric, number of individuals per cluster. Either a scalar, vector
of length #Clusters or a matrix of dimension #Clusters x timepoints.
Defaults to 1 if not passed.}

\item{power}{numeric, a specified target power.
If supplied, the minimal `N` is returned.}

\item{family}{character, distribution family. One of "gaussian", "binomial".
Defaults to "gaussian"}

\item{N_range}{numeric, vector specifying the lower and upper bound for `N`,
ignored if `power` is NULL.}

\item{sig.level}{numeric (scalar), significance level, defaults to 0.05}

\item{dfAdjust}{character, one of the following: "none","between-within",
"containment", "residual".}

\item{INDIV_LVL}{logical, should the computation be conducted on an
individual level? This leads to longer run time and is
mainly for diagnostic purposes.}

\item{verbose}{integer, how much information should the function return?}
}
\value{
The return depends on the `verbose` parameter.
If `verbose`=0, only the power is returned
If `verbose`=1 (the default), a list containing power and the
parameters of the specific setting is returned.
If requested (by `verbose`=2) this list also contains relevant matrices.
}
\description{
This is the main function of the SteppedPower package.
It calls the constructor functions for the design matrix and
covariance matrix, and then calculates the variance of the
intervention effect estimator. The latter is then used
to compute the power of a Wald test of a (given) intervention effect.
}
\details{
Let \eqn{\theta:= \mu_1-\mu_0} the treatment effect under investigation.
The variance of the treatment effect estimator \eqn{\hat\theta} can then be
estimated via weighted least squares (see also vignette 'Getting Started').
}
\examples{
## See also vignette for more examples
##
##
## stepped wedge design with 5 Clusters in 5 sequences,
## residual standard deviation 2,
## cluster effect sd = 0.33, and 10 individuals per cluster.
## Further, let the mean under the null and alternative hypothesis 0 and 1,
## respectively.
wlsPower(mu0=0, mu1=1, Cl=rep(1,5), sigma=2, tau=0.33, N=10)
##
##
## ... with auto-regressive cluster effect `AR=0.7`.
wlsPower(mu0=0, mu1=1, Cl=rep(1,5), sigma=2, tau=0.33, AR=0.7, N=10)
##
##
## ... with varying cluster size
wlsPower(mu0=0, mu1=1, Cl=rep(1,5), sigma=2, tau=0.33, N=c(12,8,10,9,14))
wlsPower(mu0=0, mu1=1, Cl=rep(1,5), sigma=2, tau=0.33,
              N=matrix(c(12,8,10,9,14,
                         11,8,10,9,13,
                         11,7,11,8,12,
                         10,7,10,8,11,
                          9,7, 9,7,11,
                          9,6, 8,7,11),5,6))
##
##
## ... with random treatment effect (with standard deviation 0.2),
## which is correlated with the cluster effect with `rho`=0.25.
wlsPower(mu0=0, mu1=1, Cl=rep(1,5), sigma=2, tau=0.33, eta=.2, rho=.25, N=10)
##
##
## ... with missing observations (a.k.a. incomplete stepped wedge design)
wlsPower(mu0=0, mu1=1, Cl=rep(1,5), sigma=2, tau=0.33, N=10, incomplete=3)
wlsPower(mu0=0, mu1=1, Cl=rep(1,5), sigma=2, tau=0.33, N=10,
             incomplete=matrix(c(1,1,1,0,0,
                                 1,1,1,1,0,
                                 1,1,1,1,1,
                                 1,1,1,1,1,
                                 0,1,1,1,1,
                                 0,0,1,1,1),5,6))
## -> the same.
##
## ... with two levels of clustering. This arises if the patients are
## observed over the whole  study period
## (often referred to as closed cohort design) or if subclusters exist
## (such as wards within clinics). For
mod_aggr  <- wlsPower(mu0=0, mu1=1, Cl=rep(1,5),
                          sigma=2, tau=0.33, psi=.25,
                          N=10, incomplete=3, verbose=2)
mod_indiv <- wlsPower(mu0=0, mu1=1, Cl=rep(1,5),
                          sigma=2, tau=0.33, psi=.25,
                          N=10, incomplete=3, verbose=2, INDIV_LVL=TRUE)
mod_aggr
mod_indiv
## Compare covariance matrices of first cluster
mod_aggr$CovarianceMatrix[1:6,1:6] ; mod_indiv$CovarianceMatrix[1:60,1:60]
##
##
## stepped wedge design with 5 Clusters in 5 sequences, residual sd = 2,
## cluster effect sd = 0.33. How many Individuals are needed to achieve a
## power of 80\% ?
wlsPower(mu0=0, mu1=1, Cl=rep(1,5), sigma=2, tau=0.33, power=.8)
##
## ... How many are needed if we have a closed cohort design with a random
## individuum effect of .7?
wlsPower(mu0=0, mu1=1, Cl=rep(1,5), sigma=2, tau=0.33, psi=.7, power=.8)
##
##
## longitudinal parallel design, with 5 time periods, 3 clusters in treatment
## and control arm each.
wlsPower(mu0=0, mu1=1, Cl=c(3,3), sigma=2, tau=0.33, N=10,
              dsntype="parallel", timepoints=5)
##
##
##
## ... with one baseline period and four parallel periods
wlsPower(mu0=0, mu1=1, Cl=c(3,3), sigma=2, tau=0.33, N=10,
              dsntype="parallel_baseline", timepoints=c(1,4))
##
##
##
## cross-over design with two timepoints before and two after the switch
wlsPower(mu0=0, mu1=1, Cl=c(3,3), sigma=2, tau=0.33, N=10,
              dsntype="crossover", timepoints=c(2,2))
##
##
##
## stepped wedge design with 32 Individuals in 8 sequences, binomial outcome,
## 50\% incidence under control, 25\% incidence under interventional treatment.
## cluster effect sd = 0.5 (ICC of 1/3 under control),
## every individual is its own cluster.
## ... with incidences defined conditional on cluster effect=0
wlsPower(mu0=0.5, mu1=0.25, Cl=rep(4,8), tau=0.5, N=1,
             family="binomial")
##
##
## ... with  marginally defined proportions
wlsPower(mu0=0.5, mu1=0.25, Cl=rep(4,8), tau=0.5, N=1,
              family="binomial", marginal_mu=TRUE)

}
