#
# e2e_compare_runs_bar.R
#
#' Create tornado bar-plots of the differences between two model runs.
#'
#' Create a tornado bar-plot diagram of the differences in either annual average masses of ecology model varibles, or annual fishery catches, between two
#' different runs of the StrathE2E model, referred to as baseline and sceanrio runs. 
#'
#' A tornado plot is a horizontal barplot which shows the difference between baseline and sceanrio runs. Bars to the right indicate scenario values greater than the baseline, bars to the left indicate scenario values less than baseline.
#' In this function the difference between scenario and baseline data is calculated as either ((scenario - baseline)/baseline) and plotted on a linear (percentage) scale, or (scenario/baseline) and plotted on a log10 scale. In both cases, zero represents scenario = baseline.
#'
#' The choice of whether to plot data on the masses of ecosystem components, or fishery catches, is determined by an argument setting in the function call. In either case two panels of tornado diagrams are plotted. In the case of mass data, the panels show
#' data on water column components of the ecoststem (upper panel) and seabed components of the system (lower panel). In the case of catch data, the upper panel shows landings, the lower shows discards. For mass data and landings, positive values (scenario > baseline) are
#' indicated by green bars to the right, negative values (scenario < baseline) by red bars to the left. For discards, positive values are in black, negative in grey. For any bars extending outsided the selected plotting range, the numeric value is displayed inside the relevant bar.
#'
#' The function accepts data from either saved csv files created by the e2e_run() function, or directly from e2e_run() results objects.
#' The relevant csv files are: /results/Modelname/Variantname/ZONE_model_anav_biomass-*.csv where ZONE is INSHORE, OFFSHORE or WHOLEDOMAIN 
#' and * represents the model run identifier (model.ident) embedded in the R-list object created by the e2e_read() function.
#'
#' In addition to generating graphics output the function returns a list object of the data presented in the plot. The object comprises two dataframes (changewater and changeseabed where annual average mass data are selected; changeland and changedisc where catch data are selected).
#' The first column in each dataframe is the proportional difference expressed on a log10 scale, the second column as a percentage.
#'
#' @param selection Text string from a list identifying which type of data are to be plotted. Select from: "AAM", "CATCH", corresponding to annual average mass data, and catch data respectively (default = "AAM"). Remember to include the phrase within "" quotes.
#' @param model1 R-list object defining the baseline model configuration compiled by the e2e_read() function.
#' @param use.saved1 Logical. If TRUE then use baseline data from a prior user-defined run held as csv files data in the current results folder (default=FALSE).
#' @param results1 R-list object of baseline model output generated by the e2e_run() (default=NULL).
#' @param model2 R-list object defining the scenario model configuration compiled by the e2e_read() function.
#' @param use.saved2 Logical. If TRUE then use scenario data from a prior user-defined run held as csv files data in the current results folder (default=FALSE).
#' @param results2 R-list object of scenario model output generated by the e2e_run() (default=NULL).
#' @param log.pc Value="LG" for data to be plotted on a log10 scale, value = "PC" for data to be plotted on a percentage difference scale (default = "PC").
#' @param zone Value = "O" for offshore, "I" for inshore, or "W" for whole model domain (all upper case) (default = "W").
#' @param bpmin Axis minimum for plot - i.e. the maximum NEGATIVE value of (scenario-baseline). Default = -50, given log.pc="PC" (percentage differences). Needs to be reset to e.g. -0.3 if log.pc="LG" (log scale).
#' @param bpmax Axis maximum for plot - i.e. the maximum POSITIVE value of (scenario-baseline). Default = +50, given log.pc="PC" (percentage differences). Needs to be reset to e.g. +0.3 if log.pc="LG" (log scale).
#' @param maintitle A optional descriptive text field (in quotes) to be added above the plot. Keep to 45 characters including spaces (default="").
#'
#' @return List object comprising two dataframes of the displayed data, graphical display in a new graphics window.
#'
#' @seealso \code{\link{e2e_read}}, \code{\link{e2e_run}}
#'
#' @importFrom graphics text
#'
#' @export
#'
#' @examples
#' # Load the 1970-1999 version of the internal North Sea model and run for 1 year
#'     m1 <- e2e_read("North_Sea", "1970-1999",model.ident="70_99")
#'     r1 <-e2e_run(m1,nyears=1)
#' 
#' # Load the 2003-2013 version of the internal North Sea model and run for 1 year
#'     m2 <- e2e_read("North_Sea", "2003-2013",model.ident="03_13")
#'     r2 <-e2e_run(m2,nyears=1)
#'
#' # Compare the annual average mass in 1970-1999 (as the baseline case) with 2002-2013
#' # (as the scenario case): 
#'     mdiff_results1 <- e2e_compare_runs_bar(selection="AAM",
#'                                       model1=NA, use.saved1=FALSE, results1=r1,
#'				         model2=NA, use.saved2=FALSE, results2=r2,
#'		                         log.pc="PC", zone="W",
#'                  		         bpmin=(-50),bpmax=(+50),
#'       		                 maintitle="2003-2013 compared with 1970-1999")
#'     mdiff_results1
#'
#' # Compare the annual catch in 1970-1999 (as the baseline case) with 2002-2013
#' # (as the scenario case): 
#'     mdiff_results2 <- e2e_compare_runs_bar(selection="CATCH",
#'                                       model1=NA, use.saved1=FALSE, results1=r1,
#'				         model2=NA, use.saved2=FALSE, results2=r2,
#'		                         log.pc="LG", zone="W",
#'                      	         bpmin=(-0.9),bpmax=(+0.4),
#'       		                 maintitle="2003-2013 compared with 1970-1999")
#'     mdiff_results2
#'
#' \donttest{
#' # Create a new scenario run from 2003-2013 case. 
#' # Copy the 2003-2013 configuration into a new model object:
#'     scen1_model   <- m2
#'     scen1_model$setup$model.ident <- "scenario1"
#' # Gear 4 (Beam_Trawl_BT1+BT2) activity rate rescaled to 0.5*baseline:
#'     scen1_model$data$fleet.model$gear_mult[4] <- 0.5
#'     scen1_results <- e2e_run(scen1_model,nyears=20)
#'
#' # Compare the annual average mass from the the 2003-2013 baseline with scenario1 data 
#'     mdiff_results3 <- e2e_compare_runs_bar(selection="AAM",
#'                                  model1=NA, use.saved1=FALSE, results1=r2,
#'                                  model2=NA, use.saved2=FALSE, results2=scen1_results,
#'                                  log.pc="PC", zone="W",
#'                                  bpmin=(-30),bpmax=(+30),
#'                                  maintitle="Beam Trawl activity reduced by half")
#'     mdiff_results3
#' }
#'
#' \donttest{
#' # Create a second sceanario from the 1970-1999 case, this time saving the results to a file;
#' # csv output to temporary folder since results.path was not set in e2e_read() when creating m1.
#' # Copy the baseline configuration into a new model object:
#'     scen2_model   <- m1
#'     scen2_model$setup$model.ident <- "scenario2"
#' # Gear 1 (Pelagic_Trawl+Seine) activity rate rescaled to 0.5*baseline:
#'     scen2_model$data$fleet.model$gear_mult[1] <- 0.5
#'     scen2_results <- e2e_run(scen2_model,nyears=20, csv.output=TRUE)
#'
#' # Compare the annual catches in the 1970-1999 base line with the Pelagic Trawl/seine scenario
#'     mdiff_results4 <- e2e_compare_runs_bar(selection="CATCH",
#'                                  model1=NA, use.saved1=FALSE, results1=r1,
#'                      	    model2=scen2_model,use.saved2=TRUE, results2=NA,
#'                      	    log.pc="LG", zone="W",
#'                      	    bpmin=(-0.4),bpmax=(+0.6),
#'                      	    maintitle="Pelagic Trawl/Seine activity reduced by half")
#'     mdiff_results4
#' }
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Author: Mike Heath                                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: May 2020                                    |
# |                                                                   |
# ---------------------------------------------------------------------

e2e_compare_runs_bar <- function(selection="AAM",
                                 model1=NA, use.saved1=FALSE, results1,
				 model2=NA, use.saved2=FALSE, results2,
				 log.pc="PC",
				 zone="W",
				 bpmin=(-50),bpmax=(+50),
				 maintitle="" ) {

   oo <- options()
   on.exit(options(oo))


plotted_data<-NULL

if(selection=="AAM"){
	plotted_data <- compare_two_runs_aam(model1=model1, from.csv1=use.saved1, results1=results1,
				 model2=model2, from.csv2=use.saved2, results2=results2,
				 log.pc=log.pc,
				 zone=zone,
				 bpmin=(bpmin),bpmax=(bpmax),
				 maintitle=maintitle )

} else if(selection=="CATCH"){
	plotted_data <- compare_two_runs_catch(model1=model1, from.csv1=use.saved1, results1=results1,
				 model2=model2, from.csv2=use.saved2, results2=results2,
				 log.pc=log.pc,
				 zone=zone,
				 bpmin=(bpmin),bpmax=(bpmax),
				 maintitle=maintitle )

} else {
	stop("Error: unknown selection '", selection, "' !\n")
}

return(plotted_data)

}

#-----------------------------------------------------------------


