% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TRR.fit.R
\name{TRR.fit}
\alias{TRR.fit}
\alias{TRR}
\title{Tensor response regression}
\usage{
TRR.fit(x, y, u, method=c('standard', 'FG', '1D', 'ECD', 'PLS'), Gamma_init)
}
\arguments{
\item{x}{The predictor matrix of dimension \eqn{p \times n}. Vector of length \eqn{n} is acceptable. If \code{y} is missing, \code{x} should be a list or an environment consisting of predictor and response datasets.}

\item{y}{The response tensor instance with dimension \eqn{r_1\times r_2\times\cdots\times r_m \times n}, where \eqn{n} is the sample size. Array with the same dimensions and matrix with dimension \eqn{r\times n} are acceptable.}

\item{u}{The dimension of envelope subspace. \eqn{u=(u_1,\cdots, u_m)}. Used for methods "FG", "1D", "ECD" and "PLS". User can use \code{\link{TensEnv_dim}} to select dimension.}

\item{method}{The method used for estimation of tensor response regression. There are four possible choices.
\itemize{
  \item{\code{"standard"}}: The standard OLS type estimation.
  \item{\code{"FG"}}: Envelope estimation with full Grassmannian (FG) algorithm.
  \item{\code{"1D"}}: Envelope estimation with one dimensional optimization approaches by 1D algorithm.
  \item{\code{"ECD"}}: Envelope estimation with one dimensional optimization approaches by ECD algorithm.
  \item{\code{"PLS"}}: The SIMPLS-type estimation without manifold optimization.
}}

\item{Gamma_init}{A list specifying the initial envelope subspace basis for "FG" method. If unspecified, use the estimation from "1D" algorithm.}
}
\value{
\item{x}{The original predictor dataset.}
 \item{y}{The original response dataset.}
 \item{call}{The method call.}
 \item{method}{The method used.}
 \item{coefficients}{The estimation of regression coefficient tensor.}
 \item{Gamma}{The estimation of envelope subspace basis.}
 \item{Sigma}{A lists of estimated covariance matrices at each mode for the error term.}
 \item{fitted.values}{The fitted response tensor.}
 \item{residuals}{The residuals tensor.}
}
\description{
This function is used for estimation of tensor response regression. The available method including standard OLS type estimation, PLS type of estimation as well as envelope estimation with FG, 1D and ECD approaches.
}
\details{
Please refer to \strong{Details} of \code{\link{TRR_sim}} about the description of the tensor response regression model.
}
\examples{

rm(list=ls())

# The dimension of response
r <- c(10, 10, 10)
# The envelope dimensions u.
u <- c(2, 2, 2)
# The dimension of predictor
p <- 5
# The sample size
n <- 100

# Simulate the data with TRR_sim.
dat <- TRR_sim(r = r, p = p, u = u, n = n)
x <- dat$x
y <- dat$y
B <- dat$coefficients

fit_std <- TRR.fit(x, y, method="standard")
fit_fg <- TRR.fit(x, y, u, method="FG")
fit_1D <- TRR.fit(x, y, u, method="1D")
fit_pls <- TRR.fit(x, y, u, method="PLS")
fit_ECD <- TRR.fit(x, y, u, method="ECD")

rTensor::fnorm(B-stats::coef(fit_std))
rTensor::fnorm(B-stats::coef(fit_fg))
rTensor::fnorm(B-stats::coef(fit_1D))
rTensor::fnorm(B-stats::coef(fit_pls))
rTensor::fnorm(B-stats::coef(fit_ECD))

## ----------- Pass a list or an environment to x also works ------------- ##
# Pass a list to x
l <- dat[c("x", "y")]
fit_std_l <- TRR.fit(l, method="standard")

# Pass an environment to x
e <- new.env()
e$x <- dat$x
e$y <- dat$y
fit_std_e <- TRR.fit(e, method="standard")

## ----------- Use dataset "bat" included in the package ------------- ##
data("bat")
x <- bat$x
y <- bat$y
fit_std <- TRR.fit(x, y, method="standard")

}
\references{
Li L, Zhang X (2017). “Parsimonious Tensor Response Regression.” Journal of the American Statistical Association, 112(519), 1131–1146.
}
\seealso{
\code{\link{summary.Tenv}} for summaries, calculating mean squared error from the prediction.

\code{\link{plot.Tenv}}(via \code{graphics::image}) for drawing the two-dimensional coefficient plot and \eqn{p}-value plot.

\code{\link{predict.Tenv}} for prediction.

The generic functions \code{\link{coef}, \link{residuals}, \link{fitted}}.

\code{\link{TensEnv_dim}} for selecting the dimension of envelope by information criteria.

\code{\link{TRR_sim}} for generating the simulated data used in tensor response regression.

The simulated data \code{\link{bat}} used in tensor response regression.
}
