% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/growth_length_age.R
\name{growth_length_age}
\alias{growth_length_age}
\title{Estimation of growth parameter using length-at-age data}
\usage{
growth_length_age(
  param,
  method,
  Linf_est = NA,
  Linf_init = 10,
  K_init = 0.1,
  t0_init = 0,
  CI = FALSE,
  ci.level = 0.95,
  age_plot = NULL,
  do.sim = FALSE,
  nsim = 10000
)
}
\arguments{
\item{param}{a list consisting of following parameters:
\itemize{
  \item \strong{age}: age measurements,
  \item \strong{length}: corresponding lengths in cm.
}}

\item{method}{indicating which of following methods should be applied:
\code{"GullandHolt"},
\code{"FordWalford"}, \code{"Chapman"}, \code{"BertalanffyPlot"},
or \code{"LSM"}}

\item{Linf_est}{BertalanffyPlot requires an estimate for Linf to derive K and t0
(for more information see Details).}

\item{Linf_init}{initital parameter of Linf for non-linear sqaures fitting (default 10)}

\item{K_init}{initital parameter of K for non-linear sqaures fitting (default 0.1)}

\item{t0_init}{initital parameter of t0 for non-linear sqaures fitting (default 0)}

\item{CI}{logical; Should confidence intervals be calculated? This option only
works for the LSM method. Default is FALSE.}

\item{ci.level}{required confidence level (for LSM method only)}

\item{age_plot}{sequence with ages used for plotting (LSM method only). By default
age_plot = seq(min(param$age),max(param$age),0.1)}

\item{do.sim}{logical. Should Monte Carlo simulation be applied? Default = FALSE}

\item{nsim}{the number of Monte Carlo simulations to be performed,
minimum is 10000 (default).}
}
\value{
A list with the input parameters and following parameters:
\itemize{
 \item \strong{x}: independent variable used for regression analysis,
 \item \strong{y}: dependent variable used for regression analysis,
 \item \strong{mod}: (non) linear model,
 \item \strong{Linf}: infinite length for investigated species in cm [cm],
 \item \strong{K}: growth coefficent for investigated species per year [1/year],
 \item \strong{t0}: theoretical time zero, at which individuals of this species hatch
     (only for Bertalanffy plot and LSM method).
 \item \strong{estimates}: dataframe with growth parameters and confidence intervals
     (only if LSM method was applied).
}
}
\description{
This function estimates growth parameters from
   length-at-age data. It
   allows to perform different methods: Gulland and Holt, Ford Walford plot,
   Chapman's method, Bertalanffy plot, or non linear least squares method (LSM).
}
\details{
Gulland and Holt plot assumes
infinitestimal delta t (only reasonable approximation of growth parameters if delta t
is small). Ford Walford plot and Chapman assume constant time intervals between ages
(delta t). The Bertalanffy plot is a robust method, however it requires an estimate of Linf. As
long as this estimate is reasonable the resulting estimate of K is reasonable. For
a first estimate of Linf the Powell Wetherall method \link{powell_wetherall} can
be used. Otherwise, the largest fish or the average of the ten largest fish can be
used for a small or large sample, respectively. All lengths have to be smaller than
Linf as otherwise the logarithm is not defined. Oldest fish (if larger than Linf) have
to be omitted. Non-linear least squares fitting is the preferred method to estimate
growth parameters according to Sparre and Venema (1998).
}
\examples{
# synthetical length at age data
dat <- list(age = rep(1:7,each = 5),
   length = c(rnorm(5,25.7,0.9),rnorm(5,36,1.2),rnorm(5,42.9,1.5),rnorm(5,47.5,2),
   rnorm(5,50.7,0.4),rnorm(5,52.8,0.5),rnorm(5,54.2,0.7)))

growth_length_age(dat, method = "GullandHolt")

# Bertalaffy plot
growth_length_age(dat, method = "BertalanffyPlot", Linf_est = 50)

# non linear least squares method
\donttest{
output <- growth_length_age(param = dat, method = "LSM",
     Linf_init = 30, CI = TRUE, age_plot=NULL)
summary(output$mod)
}

}
\references{
Sparre, P., Venema, S.C., 1998. Introduction to tropical fish stock assessment.
Part 1. Manual. \emph{FAO Fisheries Technical Paper}, (306.1, Rev. 2). 407 p.
}
