/******************************************************************************
 * @section MODIFICATION
 *
 * Modification by Ruida Zhong for the R package VICmodel on May 20th, 2018:
 * The `#include <pwd.h>` is surrounded by `#ifdef __linux__` and `#endif`
 * to ensure the usability for platforms except linux.
 * `char   vic_run_ref_str[MAXSTRING];` is added by `extern` and its definition
 * is remove to `global.cpp` since global variables defined in head file is not
 * available in C++.
 *
 * @section DESCRIPTION
 *
 * Definition header file
 *
 * @section LICENSE
 *
 * The Variable Infiltration Capacity (VIC) macroscale hydrological model
 * Copyright (C) 2016 The Computational Hydrology Group, Department of Civil
 * and Environmental Engineering, University of Washington.
 *
 * The VIC model is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *****************************************************************************/

#ifndef VIC_DEF_H
#define VIC_DEF_H

#define _BSD_SOURCE
#define __USE_XOPEN
#define _GNU_SOURCE

#include <float.h>
#include <math.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stdbool.h>
#include <stddef.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#ifdef __linux__
#include <pwd.h>
#endif

#include <vic_physical_constants.h>
#include <vic_log.h>

/***** Model Constants *****/
#define MAXSTRING    2048
#define MISSING      -99999.   /**< missing value */
#define MISSING_USI  99999.    /**< missing value for unsigned ints */
#define MISSING_S    "MISSING"    /**< missing value for strings */
#define NODATA_VH    -1        /**< missing value for veg_hist inputs */
#define NODATA_VEG   -1        /**< flag for veg types not in grid cell */
#define ERROR        -999      /**< Error Flag returned by subroutines */

/***** Define maximum array sizes for model source code *****/
#define MAX_LAYERS      3      /**< maximum number of soil moisture layers */
#define MAX_NODES       50     /**< maximum number of soil thermal nodes */
#define MAX_FRONTS      3      /**< maximum number of freezing and thawing front depths to store */
#define MAX_FROST_AREAS 10     /**< maximum number of frost sub-areas */
#define MAX_LAKE_NODES  20     /**< maximum number of lake thermal nodes */
#define MAX_ZWTVMOIST   11     /**< maximum number of points in water table vs moisture curve for each soil layer; should include points at lower and upper boundaries of the layer */

/***** Define minimum values for model parameters *****/
#define MINSOILDEPTH    0.001  /**< Minimum layer depth with which model can work (m) */
#define MIN_FCANOPY    0.0001  /**< Minimum allowable canopy fraction */
#define MIN_SNOW_WETFRAC 0.01  /**< Minimum fraction of snow depth to be considered wet */

/***** Define minimum and maximum values for model timesteps *****/
#define MIN_SUBDAILY_STEPS_PER_DAY  1
#define MAX_SUBDAILY_STEPS_PER_DAY  1440

#ifndef WET
#define WET 0
#define DRY 1
#endif

#ifndef SNOW
#define RAIN 0
#define SNOW 1
#endif

#define min(a, b) (a < b) ? a : b
#define max(a, b) (a > b) ? a : b

extern size_t NR;       /**< array index for force struct that indicates
                             the model step avarage or sum */
extern size_t NF;       /**< array index loop counter limit for force
                             struct that indicates the SNOW_STEP values */
extern char   vic_run_ref_str[MAXSTRING];

/******************************************************************************
 * @brief   Snow Density parametrizations
 *****************************************************************************/
enum
{
    DENS_BRAS,
    DENS_SNTHRM
};

/******************************************************************************
 * @brief   Baseflow parametrizations
 *****************************************************************************/
enum
{
    ARNO,
    NIJSSEN2001
};

/******************************************************************************
 * @brief   Aerodynamic Resistance options
 *****************************************************************************/
enum
{
    AR_406,
    AR_406_LS,
    AR_406_FULL,
    AR_410
};

/******************************************************************************
 * @brief   Ground Flux options
 *****************************************************************************/
enum
{
    GF_406,
    GF_410
};

/******************************************************************************
 * @brief   Canopy resistance parametrizations
 *****************************************************************************/
enum
{
    RC_JARVIS,
    RC_PHOTO
};

/******************************************************************************
 * @brief   Photosynthesis parametrizations
 *****************************************************************************/
enum
{
    PS_FARQUHAR,
    PS_MONTEITH
};

/******************************************************************************
 * @brief   Photosynthetic pathways
 *****************************************************************************/
enum
{
    PHOTO_C3,
    PHOTO_C4
};

/***** Data Structures *****/

/******************************************************************************
 * @brief   This structure stores model options.
 *****************************************************************************/
typedef struct {
    // simulation modes
    short AboveTreelineVeg;  /**< Default veg type to use above treeline;
                                Negative number indicates bare soil. */
    unsigned short int AERO_RESIST_CANSNOW;  /**< "AR_406" = multiply aerodynamic resistance
                                                by 10 for latent heat but not
                                                for sensible heat (as in
                                                VIC 4.0.6); do NOT apply stability
                                                correction; use surface aero_resist
                                                for ET when no snow in canopy.
                                                "AR_406_LS" = multiply aerodynamic resistance
                                                by 10 for BOTH latent heat AND
                                                sensible heat; do NOT apply
                                                stability correction;
                                                use surface aero_resist
                                                for ET when no snow in canopy.
                                                "AR_406_FULL" = multiply aerodynamic resistance
                                                by 10 for BOTH latent heat AND
                                                sensible heat; do NOT apply
                                                stability correction;
                                                always use canopy aero_resist
                                                for ET.
                                                "AR_410" = do not multiply aerodynamic
                                                resistance by 10 in snow-filled
                                                canopy (as in VIC 4.1.0);
                                                DO apply stability correction;
                                                always use canopy aero_resist
                                                for ET. */
    bool BLOWING;        /**< TRUE = calculate sublimation from blowing snow */
    bool BLOWING_VAR_THRESHOLD;
    bool BLOWING_CALC_PROB;
    bool BLOWING_SIMPLE;
    bool BLOWING_FETCH;
    bool BLOWING_SPATIAL_WIND;
    bool CARBON;         /**< TRUE = simulate carbon cycling processes;
                            FALSE = no carbon cycling (default) */
    bool CLOSE_ENERGY;   /**< TRUE = all energy balance calculations are
                            iterated to minimize the total column (air,
                            canopy, snow and ground) error; FALSE = no
                            iteration is used and the model estimates the new
                            fluxes based on those from the previous time step,
                            results should be similar, however, the model will
                            report energy balance errors. */
    bool COMPUTE_TREELINE; /**< TRUE = Determine treeline and exclude overstory
                              vegetation from higher elevations */
    bool CONTINUEONERROR; /**< TRUE = VIC will continue to run after a cell has an error */
    bool CORRPREC;       /**< TRUE = correct precipitation for gage undercatch */
    bool EQUAL_AREA;     /**< TRUE = RESOLUTION stores grid cell area in km^2;
                            FALSE = RESOLUTION stores grid cell side length in degrees */
    bool EXP_TRANS;      /**< TRUE = Uses grid transform for exponential node
                            distribution for soil heat flux calculations*/
    bool FROZEN_SOIL;    /**< TRUE = Use frozen soils code */
    bool FULL_ENERGY;    /**< TRUE = Use full energy code */
    unsigned short int GRND_FLUX_TYPE; /**< "GF_406"  = use (flawed) formulas for ground flux, deltaH, and fusion
                                          from VIC 4.0.6 and earlier
                                          "GF_410"  = use formulas from VIC 4.1.0 */
    bool IMPLICIT;       /**< TRUE = Use implicit solution when computing
                            soil thermal fluxes */
    bool JULY_TAVG_SUPPLIED; /**< If TRUE and COMPUTE_TREELINE is also true,
                                then average July air temperature will be read
                                from soil file and used in calculating treeline */
    bool LAKES;          /**< TRUE = use lake energy code */
    size_t Ncanopy;      /**< Number of canopy layers in the model. */
    size_t Nfrost;       /**< Number of frost subareas in model */
    size_t Nlakenode;    /**< Number of lake thermal nodes in the model. */
    size_t Nlayer;       /**< Number of layers in model */
    size_t Nnode;        /**< Number of soil thermal nodes in the model */
    bool NOFLUX;         /**< TRUE = Use no flux lower bondary when computing
                            soil thermal fluxes */
    size_t NVEGTYPES;    /**< number of vegetation types in veg_param file */
    size_t NLAKENODES;   /**< number of lake layers in lake_param file */
    unsigned short int RC_MODE;        /**< RC_JARVIS = compute canopy resistance via Jarvis formulation (default)
                                          RC_PHOTO = compute canopy resistance based on photosynthetic activity */
    size_t ROOT_ZONES;   /**< Number of root zones used in simulation */
    bool QUICK_FLUX;     /**< TRUE = Use Liang et al., 1999 formulation for
                            ground heat flux, if FALSE use explicit finite
                            difference method */
    bool QUICK_SOLVE;    /**< TRUE = Use Liang et al., 1999 formulation for
                            iteration, but explicit finite difference
                            method for final step. */
    bool SHARE_LAYER_MOIST; /**< TRUE = transpiration in moisture-limited layers can draw from other layers (default) */
    unsigned short int SNOW_DENSITY;   /**< DENS_BRAS: Use algorithm of Bras, 1990; DENS_SNTHRM: Use algorithm of SNTHRM89 adapted for 1-layer pack */
    size_t SNOW_BAND;    /**< Number of elevation bands over which to solve the
                            snow model */
    bool SPATIAL_FROST;   /**< TRUE = use a uniform distribution to simulate the
                             spatial distribution of soil frost; FALSE = assume
                             that the entire grid cell is frozen uniformly. */
    bool SPATIAL_SNOW;    /**< TRUE = use a uniform distribution to simulate the
                             partial coverage of the surface by a thin snowpack.
                             Coverage is assumed to be uniform after snowfall
                             until the pack begins to melt. */
    bool TFALLBACK;      /**< TRUE = when any temperature iterations fail to converge,
                                   use temperature from previous time step; the number
                                   of instances when this occurs will be logged and
                                   reported at the end of the cell's simulation
                            FALSE = when iterations fail to converge, report an error
                                    and abort simulation for current grid cell
                            Default = TRUE */

    // input options
    unsigned short int BASEFLOW;     /**< ARNO: read Ds, Dm, Ws, c; NIJSSEN2001: read d1, d2, d3, d4 */
    unsigned short int GRID_DECIMAL; /**< Number of decimal places in grid file extensions */
    bool VEGLIB_FCAN;    /**< TRUE = veg library file contains monthly fcanopy values */
    bool VEGLIB_PHOTO;   /**< TRUE = veg library contains photosynthesis parameters */
    bool VEGPARAM_ALB;   /**< TRUE = veg param file contains monthly albedo values */
    bool VEGPARAM_FCAN;  /**< TRUE = veg param file contains monthly fcanopy values */
    bool VEGPARAM_LAI;   /**< TRUE = veg param file contains monthly LAI values */
    unsigned short int ALB_SRC;        /**< FROM_VEGLIB = use albedo values from veg library file
                                          FROM_VEGPARAM = use albedo values from the veg param file */
    unsigned short int FCAN_SRC;       /**< FROM_VEGLIB = use fcanopy values from veg library file
                                          FROM_VEGPARAM = use fcanopy values from the veg param file */
    unsigned short int LAI_SRC;        /**< FROM_VEGLIB = use LAI values from veg library file
                                          FROM_VEGPARAM = use LAI values from the veg param file */
    bool LAKE_PROFILE;   /**< TRUE = user-specified lake/area profile */
    bool ORGANIC_FRACT;  /**< TRUE = organic matter fraction of each layer is read from the soil parameter file; otherwise set to 0.0. */

    // state options
    unsigned short int STATE_FORMAT;  /**< TRUE = model state file is binary (default) */
    bool INIT_STATE;     /**< TRUE = initialize model state from file */
    bool SAVE_STATE;     /**< TRUE = save state file */

    // output options
    size_t Noutstreams;  /**< Number of output stream */
} option_struct;

/******************************************************************************
 * @brief   This structure stores all model run global parameters.
 *****************************************************************************/
typedef struct {
    double wind_h;                 /**< height of wind measurements (m) */
    double resolution;             /**< Model resolution (degrees) */
    double dt;                     /**< Time step in seconds */
    double snow_dt;                /**< Snow model time step in seconds */
    double runoff_dt;              /**< Runoff time step in seconds */
    double atmos_dt;               /**< Atmos time step in seconds */
    size_t model_steps_per_day;    /**< Number of model timesteps per day */
    size_t snow_steps_per_day;     /**< Number of snow timesteps per day */
    size_t runoff_steps_per_day;   /**< Number of runoff timesteps per day */
    size_t atmos_steps_per_day;    /**< Number of atmos timesteps per day */
    unsigned short int endday;     /**< Last day of model simulation */
    unsigned short int endmonth;   /**< Last month of model simulation */
    unsigned short int endyear;    /**< Last year of model simulation */
    unsigned short int forceday[2];  /**< day forcing files starts */
    unsigned int forcesec[2];          /**< seconds since midnight when forcing
                                          files starts */
    unsigned short int forcemonth[2];  /**< month forcing files starts */
    unsigned short int forceoffset[2];  /**< counter to keep track of offset in reading
                                           forcing files; updated after every read */
    unsigned int forceskip[2];   /**< number of model time steps to skip at
                                      the start of the forcing file */
    unsigned short int forceyear[2];  /**< year forcing files start */
    size_t nrecs;                /**< Number of time steps simulated */
    unsigned short int startday;  /**< Starting day of the simulation */
    unsigned short int startmonth;  /**< Starting month of the simulation */
    unsigned int startsec;          /**< Seconds since midnight when simulation
                                       will start */
    unsigned short int startyear;  /**< Starting year of the simulation */
    unsigned short int stateday;   /**< Day of the simulation at which to save
                                      model state */
    unsigned short int statemonth;  /**< Month of the simulation at which to save
                                       model state */
    unsigned int statesec;          /**< Seconds since midnight at which to save state */
    unsigned short int stateyear;  /**< Year of the simulation at which to save
                                      model state */
    unsigned short int calendar;  /**< Date/time calendar */
    unsigned short int time_units;  /**< Units for numeric times */
    double time_origin_num;        /**< Numeric date origin */
    char time_origin_str[MAXSTRING];  /**< string date origin */
} global_param_struct;

/******************************************************************************
 * @brief    This structure holds the model parameters.
 *****************************************************************************/
typedef struct {
    // Lapse Rate
    double LAPSE_RATE;  /**< temperature lapse rate (C/m) */

    // Precipitation Guage Height
    double GAUGE_HEIGHT;   /**< precipitation gauge height (m) */

    // Huge Resistance Term
    double HUGE_RESIST;  /**< Extermely large resistance term (s/m) */

    // Surface Albedo Parameters
    double ALBEDO_BARE_SOIL;  /**< Broadband albedo of bare soil */

    // Surface Emissivities
    double EMISS_GRND;  /**< Emissivity of bare soil */
    double EMISS_VEG;  /**< Emissivity of vegetation */
    double EMISS_ICE;  /**< Emissivity of bare ice */
    double EMISS_SNOW;  /**< Emissivity of snow */
    double EMISS_H2O;  /**< Emissivity of open water surface */

    // Soil Constraints
    double SOIL_RARC;  /**< Architectural resistance (s/m) of soil when computing soil evaporation via Penman-Monteith eqn */
    double SOIL_RESID_MOIST;  /**< Default residual moisture content of soil colum */
    double SOIL_SLAB_MOIST_FRACT;  /**< Volumetric moisture content (fraction of porosity) in the soil/rock below the bottom soil layer; this assumes that the soil below the bottom layer has the same texture as the bottom layer. */
    double SOIL_WINDH;  /**< Default wind measurement height over soil (m) */

    // Vegetation Parameters
    double VEG_LAI_SNOW_MULTIPLIER;  /**< multiplier to calculate the amount of available snow interception as a function of LAI (m) */
    double VEG_LAI_WATER_FACTOR;  /**< Coefficient multiplied by the LAI to determine the amount of water that can be stored in the canopy */
    double VEG_MIN_INTERCEPTION_STORAGE;  /**< the amount of snow on the canopy that can only be melted off. (m) */
    double VEG_RATIO_DH_HEIGHT;  /**< Ratio of displacement height (m) to vegetation height (m) */
    double VEG_RATIO_RL_HEIGHT;  /**< Ratio of roughness length (m) to vegetation height (m) */

    // Canopy Parameters
    double CANOPY_CLOSURE;  /**< Threshold vapor pressure deficit for stomatal closure (Pa) */
    double CANOPY_RSMAX;  /**< Maximum allowable resistance (s/m) */
    double CANOPY_VPDMINFACTOR;  /**< Minimum allowable vapor pressure deficit factor */

    // Lake Parameters
    double LAKE_TMELT;
    double LAKE_MAX_SURFACE;  /**< max. surface layer thickness for E-B (m) */
    double LAKE_BETA;  /**< Curve shape parameter for lake profile. */
    double LAKE_FRACMIN;  /**< min ice thickness in meters */
    double LAKE_FRACLIM;  /**< lower limit on fractional ice cover */
    double LAKE_DM;  /**< molecular diffusivity of water */
    double LAKE_SNOWCRIT;  /**< for albedo (m) */
    double LAKE_ZWATER;
    double LAKE_ZSNOW;
    double LAKE_RHOSNOW;  /**< density of snow (kg m-3) */
    double LAKE_CONDI;  /**< thermal conductivity of ice */
    double LAKE_CONDS;  /**< thermal conductivity of snow */
    double LAKE_LAMISW;  /**< attenuation of shortwave radiation through ice (1/m) */
    double LAKE_LAMILW;  /**< attenuation of longwave radiation through ice (1/m) */
    double LAKE_LAMSSW;  /**< attenuation of shortwave radiation through snow (1/m) */
    double LAKE_LAMSLW;  /**< attenuation of longwave radiation through snow (1/m) */
    double LAKE_LAMWSW;  /**< attenuation of shortwave radiation through water (1/m) */
    double LAKE_LAMWLW;  /**< attenuation of longwave radiation through water (1/m) */
    double LAKE_A1;  /**< Percent of radiation in visible band. */
    double LAKE_A2;  /**< Percent of radiation in infrared band. */
    double LAKE_QWTAU;  /**< D. Pollard sub-ice time constant. */
    int LAKE_MAX_ITER;

    // Saturation Vapor Pressure Parameters
    double SVP_A;  /**< constant for saturated vapor pressure curve (kPa) */
    double SVP_B;  /**< constant for saturated vapor pressure curve (kPa) */
    double SVP_C;  /**< constant for saturated vapor pressure curve (kPa) */

    // Photosynthesis Parameters
    double PHOTO_OMEGA;  /**< single leaf scattering albedo */
    double PHOTO_LAIMAX;  /**< Maximum LAI in nitrogen scaling */
    double PHOTO_LAILIMIT;  /**< Minimum LAI in nitrogen scaling and maximum LAI in PAR computation */
    double PHOTO_LAIMIN;  /**< Minimum LAI in PAR computation */
    double PHOTO_EPAR;  /**< Energy content of PAR [J/mol photons] = (4.6 mol/MJ PAR)^-1 */
    double PHOTO_FCMAX;  /**< Maximum fractional veg cover; (1-FcMax) = min amount of ground visible */
    double PHOTO_FCMIN;  /**< Minimum fractional veg cover; (1-FcMin) = max amount of ground visible */
    double PHOTO_ZENITHMIN;  /**< Check for solar zenith angle > 89 deg */
    double PHOTO_ZENITHMINPAR;  /**< Cosine of the minimum solar zenith angle for photosynthesis to take place */
    double PHOTO_ALBSOIPARMIN;  /**< Minimum soil reflectivity in PAR range */
    double PHOTO_MINMAXETRANS;  /**< Minimum of maximum electron transport rate [10e-12 mol/(m^2 s)] */
    double PHOTO_MINSTOMCOND;  /**< Minimum stomatal conductance [mol H2O/m2s] */
    double PHOTO_FCI1C3;  /**< C3 Plants factor that relate leaf internal CO2 concentration to ambient CO2 concentration */
    double PHOTO_FCI1C4;  /**< C4 Plants factor that relate leaf internal CO2 concentration to ambient CO2 concentration */
    double PHOTO_OX;  /**< OXYGEN CONCENTRATION [MOL(O2) / MOL(AIR)] */
    double PHOTO_KC;  /**< MICHAELIS-MENTEN CONSTANT FOR CO2 AT 25C [MOL(CO2) / MOL(AIR)] */
    double PHOTO_KO;  /**< MICHAELIS-MENTEN CONSTANT FOR O2 AT 25C [MOL(O2) / MOL(AIR)] */
    double PHOTO_EC;  /**< ACTIVATION ENERGY FOR KC [J / MOL] */
    double PHOTO_EO;  /**< ACTIVATION ENERGY FOR KO [J / MOL] */
    double PHOTO_EV;  /**< ACTIVATION ENERGY FOR VCMAX [J / MOL] */
    double PHOTO_ER;  /**< ACTIVATION ENERGY FOR DARK RESPIRATION [J / MOL] */
    double PHOTO_ALC3;  /**< EFFICIENCY OF OF PHOTON CAPTURE */
    double PHOTO_FRDC3;  /**< RATIO OF DARK RESPIRATION TO "PVM" AT 25C for C3 */
    double PHOTO_EK;  /**< = Q10=2 (Collatz et al. 1992) */
    double PHOTO_ALC4;  /**< EFFECTIVE QUANTUM EFFICIENCY */
    double PHOTO_FRDC4;  /**< RATIO OF DARK RESPIRATION TO "PVM" AT 25C for C4 */
    double PHOTO_THETA;  /**< CURVATURE PARAMETER */
    double PHOTO_FRLEAF;  /**< Ratio of canopy leaf respiration to whole plant maintenance respiration */
    double PHOTO_FRGROWTH;  /**< Ratio of plant growth respiration to NPP */

    // Soil Respiration Parameters
    double SRESP_E0_LT;  /**< Lloyd-Taylor E0 parameter [K] */
    double SRESP_T0_LT;  /**< Lloyd-Taylor T0 parameter [K] */
    double SRESP_WMINFM;  /**< minimum soil moisture (fraction) at which soil respiration can occur */
    double SRESP_WMAXFM;  /**< maximum soil moisture (fraction) at which soil respiration can occur */
    double SRESP_WOPTFM;  /**< soil moisture (fraction) at which maximum soil respiration occurs */
    double SRESP_RHSAT;  /**< ratio of soil respiration rate under saturated conditions (w=wmaxFM) to that under optimal conditions (w=woptFM) */
    double SRESP_RFACTOR;  /**< scaling factor to account for other (non-moisture) sources of inhibition of respiration */
    double SRESP_TAULITTER;  /**< Litter pool turnover time [y] */
    double SRESP_TAUINTER;  /**< Intermediate pool turnover time [y] */
    double SRESP_TAUSLOW;  /**< Slow pool turnover time [y] */
    double SRESP_FAIR;  /**< Fraction of respired carbon from litter pool that is lost to atmosphere */
    double SRESP_FINTER;  /**< Fraction of [respired carbon from litter pool that goes to soil] that goes to intermediate pool */

    // Snow Parameters
    double SNOW_MAX_SURFACE_SWE;  /**< maximum depth of the surface layer in water equivalent (m) */
    double SNOW_LIQUID_WATER_CAPACITY;  /**< water holding capacity of snow as a fraction of snow-water-equivalent */
    double SNOW_NEW_SNOW_DENSITY;  /**< density of new fallen snow */
    double SNOW_NEW_SNOW_DENS_MAX; /**< new snow density max for Hedstrom and Pomeroy 1998 equation [Warren et al. 1999, Bormann et al. 2013, Maidment Figure 7.2.3] */
    double SNOW_DEPTH_THRES;  /**< Snow depth threshold below which we do not consider the ground flux out of the snowpack in calculating change in cold content (m) */
    double SNOW_DENS_DMLIMIT;  /**< Density limit used in calculation of destructive metamorphism (kg/m^3) */
    double SNOW_DENS_DMLIMIT_FACTOR;  /**< Density limit factor used in calculation of destructive metamorphism (kg/m^3) */
    double SNOW_DENS_MAX_CHANGE;  /**< maximum change in snowfall depth (fraction of swe) */
    double SNOW_DENS_ETA0;  /**< viscosity of snow at T = 0C and density = 0 used in calculation of true viscosity (Ns/m2) */
    double SNOW_DENS_C1;  /**< Constant in snow density computation */
    double SNOW_DENS_C2;  /**< Constant in snow density computation */
    double SNOW_DENS_C3;  /**< Constant in snow density computation */
    double SNOW_DENS_C3_CONST;  /**< Constant in snow density computation */
    double SNOW_DENS_C4;  /**< Constant in snow density computation */
    double SNOW_DENS_C4WET;  /**< Constant in snow density computation */
    double SNOW_DENS_C5;  /**< constant used in snow viscosity calculation, taken from SNTHRM.89 (/C) */
    double SNOW_DENS_C6;  /**< constant used in snow viscosity calculation, taken from SNTHRM.89 (kg/m3) */
    double SNOW_DENS_F;  /**< internal compaction rate coefficient */
    double SNOW_DENS_EXP;  /**< exponent in snow density compaction equation [Bras pg. 257 ]*/
    double SNOW_DENS_DENOM;  /**< denomenator in snow density compaction equation [Bras pg. 257] */
    double SNOW_NEW_SNT_C1; /**< Constant in Sntherm new snow density computation. */
    double SNOW_NEW_SNT_C2; /**< Constant in Sntherm new snow density computation. */
    double SNOW_NEW_SNT_C3; /**< Constant in Sntherm new snow density computation. */
    double SNOW_NEW_BRAS_DENOM;  /**< Constant in Bras new snow density computation. */
    double SNOW_MIN_SWQ_EB_THRES;  /**< Minimum SWQ for which the snowpack energy balance is computed independent of the soil surface temperature */
    double SNOW_A1;  /**< Attenuation coefficient for shortwave in a snowpack. Value and equation taken from Patterson and Hamblin, 1988 */
    double SNOW_A2;  /**< Attenuation coefficient for shortwave in a snowpack. Value and equation taken from Patterson and Hamblin, 1988 */
    double SNOW_L1;  /**< Attenuation coefficient for shortwave in a snowpack. Value and equation taken from Patterson and Hamblin, 1988 (1/m) */
    double SNOW_L2;  /**< Attenuation coefficient for shortwave in a snowpack. Value and equation taken from Patterson and Hamblin, 1988 (1/m) */
    double SNOW_NEW_SNOW_ALB;  /**< Snow albedo curve parameters. */
    double SNOW_ALB_ACCUM_A;  /**< Snow albedo curve parameters. */
    double SNOW_ALB_ACCUM_B;  /**< Snow albedo curve parameters. */
    double SNOW_ALB_THAW_A;  /**< Snow albedo curve parameters. */
    double SNOW_ALB_THAW_B;  /**< Snow albedo curve parameters. */
    double SNOW_TRACESNOW;  /**< Defines the minimum amount of new snow (mm) which will reset the snowpack albedo to new snow */
    double SNOW_CONDUCT;  /**< conductivity of snow (W/mK) */
    double SNOW_MAX_SNOW_TEMP;  /**< maximum temperature (C) at which snow can fall */
    double SNOW_MIN_RAIN_TEMP;  /**< minimum temperature (C) at which rain can fall */

    // Blowing Snow Parameters
    double BLOWING_KA;  /**< thermal conductivity of air (W/mK) */
    double BLOWING_CSALT;  /**< saltation constant m/s */
    double BLOWING_UTHRESH;  /**< threshold shear velocity m/s */
    double BLOWING_KIN_VIS;  /**< Kinemativ viscosity of air (m2/s) */
    int BLOWING_MAX_ITER;     /**< Max. iterations for numerical integration */
    int BLOWING_K;
    double BLOWING_SETTLING;  /**< Particle settling velocity m/s */
    int BLOWING_NUMINCS;     /**< Number of prob intervals to solve for wind. */

    // Treeline temperature
    double TREELINE_TEMPERATURE;  /**< Number of prob intervals to solve for wind. */

    // Iteration Bracket Widths
    double SNOW_DT;  /**< Used to bracket snow surface temperatures while computing the snow surface energy balance (C) */
    double SURF_DT;  /**< Used to bracket soil surface temperatures while computing energy balance (C) */
    double SOIL_DT;  /**< Used to bracket soil temperatures while solving the soil thermal flux (C) */
    double CANOPY_DT;  /**< Used to bracket canopy air temperatures while computing energy balance (C) */
    double CANOPY_VP;  /**< Used to bracket canopy vapor pressures while computing moisture balance (Pa) */

    // Convergence Tolerances
    double TOL_GRND;
    double TOL_OVER;

    // Frozen Soil Parameters
    int FROZEN_MAXITER;

    // Canopy Iterations
    int MAX_ITER_GRND_CANOPY;

    // Newton-Raphson Solver Parameters
    int NEWT_RAPH_MAXTRIAL;
    double NEWT_RAPH_TOLX;
    double NEWT_RAPH_TOLF;
    double NEWT_RAPH_R_MAX;
    double NEWT_RAPH_R_MIN;
    double NEWT_RAPH_RELAX1;
    double NEWT_RAPH_RELAX2;
    double NEWT_RAPH_RELAX3;
    double NEWT_RAPH_EPS2;

    // Root-Brent parameters
    int ROOT_BRENT_MAXTRIES;
    int ROOT_BRENT_MAXITER;
    double ROOT_BRENT_TSTEP;
    double ROOT_BRENT_T;
} parameters_struct;

/******************************************************************************
 * @brief   This structure stores the soil parameters for a grid cell.
 *****************************************************************************/
typedef struct {
    bool FS_ACTIVE;                   /**< if TRUE frozen soil algorithm is
                                         active in current grid cell */
    double Ds;                        /**< fraction of maximum subsurface flow
                                         rate */
    double Dsmax;                     /**< maximum subsurface flow rate
                                         (mm/day) */
    double Ksat[MAX_LAYERS];          /**< saturated hydraulic  conductivity
                                         (mm/day) */
    double Wcr[MAX_LAYERS];           /**< critical moisture level for soil
                                         layer, evaporation is no longer
                                         affected moisture stress in the
                                         soil (mm) */
    double Wpwp[MAX_LAYERS];          /**< soil moisture content at permanent
                                         wilting point (mm) */
    double Ws;                        /**< fraction of maximum soil moisture */
    double AlbedoPar;                 /**< soil albedo in PAR range (400-700nm) */
    double alpha[MAX_NODES];          /**< thermal solution constant */
    double annual_prec;               /**< annual average precipitation (mm) */
    double avg_temp;                  /**< average soil temperature (C) */
    double avgJulyAirTemp;            /**< Average July air temperature (C) */
    double b_infilt;                  /**< infiltration parameter */
    double beta[MAX_NODES];           /**< thermal solution constant */
    double bubble[MAX_LAYERS];        /**< bubbling pressure, HBH 5.15 (cm) */
    double bubble_node[MAX_NODES];    /**< bubbling pressure (cm) */
    double bulk_density[MAX_LAYERS];  /**< soil bulk density (kg/m^3) */
    double bulk_dens_min[MAX_LAYERS]; /**< bulk density of mineral soil (kg/m^3) */
    double bulk_dens_org[MAX_LAYERS]; /**< bulk density of organic soil (kg/m^3) */
    double c;                         /**< exponent in ARNO baseflow scheme */
    double depth[MAX_LAYERS];         /**< thickness of each soil moisture layer (m) */
    double dp;                        /**< soil thermal damping depth (m) */
    double dz_node[MAX_NODES];        /**< thermal node thickness (m) */
    double Zsum_node[MAX_NODES];      /**< thermal node depth (m) */
    double expt[MAX_LAYERS];          /**< layer-specific exponent n (=3+2/lambda) in Campbell's eqn for hydraulic conductivity, HBH 5.6 */
    double expt_node[MAX_NODES];      /**< node-specific exponent n (=3+2/lambda) in Campbell's eqn for hydraulic conductivity, HBH 5.6 */
    double frost_fract[MAX_FROST_AREAS]; /**< spatially distributed frost coverage fractions */
    double frost_slope;               /**< slope of frost distribution */
    double gamma[MAX_NODES];          /**< thermal solution constant */
    double init_moist[MAX_LAYERS];    /**< initial layer moisture level (mm) */
    double max_infil;                 /**< maximum infiltration rate */
    double max_moist[MAX_LAYERS];     /**< maximum moisture content (mm) per layer */
    double max_moist_node[MAX_NODES]; /**< maximum moisture content (mm/mm) per node */
    double max_snow_distrib_slope;    /**< Maximum slope of snow depth distribution [m].  This should equal 2*depth_min, where depth_min = minimum snow pack depth below which coverage < 1.  Comment, ported from user_def.h, with questionable units: SiB uses 0.076; Rosemount data imply 0.155cm depth ~ 0.028mm swq. */
    double phi_s[MAX_LAYERS];         /**< soil moisture diffusion parameter (mm/mm) */
    double porosity[MAX_LAYERS];      /**< porosity (fraction) */
    double quartz[MAX_LAYERS];        /**< quartz content of soil (fraction of mineral soil volume) */
    double organic[MAX_LAYERS];       /**< organic content of soil (fraction of total soil volume) */
    double resid_moist[MAX_LAYERS];   /**< residual moisture content of soil layer */
    double rough;                     /**< soil surface roughness (m) */
    double snow_rough;                /**< snow surface roughness (m) */
    double soil_density[MAX_LAYERS];  /**< soil particle density (kg/m^3) */
    double soil_dens_min[MAX_LAYERS]; /**< particle density of mineral soil (kg/m^3) */
    double soil_dens_org[MAX_LAYERS]; /**< particle density of organic soil (kg/m^3) */
    double *BandElev;                 /**< Elevation of each snow elevation band */
    double *AreaFract;                /**< Fraction of grid cell included in each snow elevation band */
    double *Pfactor;                  /**< Change in Precipitation due to elevation (fract) in each snow elevation band */
    double *Tfactor;                  /**< Change in temperature due to elevation (C) in each snow elevation band */
    bool *AboveTreeLine;             /**< Flag to indicate if band is above the treeline */
    double elevation;                 /**< grid cell elevation (m) */
    double lat;                       /**< grid cell central latitude */
    double lng;                       /**< grid cell central longitude */
    double cell_area;                 /**< Area of grid cell (m^2) */
    double time_zone_lng;             /**< central meridian of the time zone */
    unsigned int gridcel;             /**< grid cell number */
    double zwtvmoist_zwt[MAX_LAYERS + 2][MAX_ZWTVMOIST]; /**< zwt values in the zwt-v-moist curve for each layer */
    double zwtvmoist_moist[MAX_LAYERS + 2][MAX_ZWTVMOIST]; /**< moist values in the zwt-v-moist curve for each layer */
    double slope;
    double aspect;
    double ehoriz;
    double whoriz;
} soil_con_struct;

/******************************************************************************
 * @brief   This structure stores information about the vegetation coverage of
 *          the current grid cell.
 *****************************************************************************/
typedef struct {
    double albedo[MONTHS_PER_YEAR];   /**< climatological vegetation albedo
                                         (fraction) */
    double *CanopLayerBnd;  /**< Upper boundary of each canopy layer,
                               expressed as fraction of total LAI */
    double Cv;              /**< fraction of vegetation coverage */
    double displacement[MONTHS_PER_YEAR]; /**< climatological vegetation
                                             displacement height (m) */
    double fcanopy[MONTHS_PER_YEAR]; /**< climatological fractional area
                                        covered by plant canopy (fraction) */
    double fetch;           /**< Average fetch length for each vegetation
                               class. */
    double LAI[MONTHS_PER_YEAR]; /**< climatological leaf area index (m2/m2) */
    int LAKE;               /**< TRUE = this tile is a lake/wetland tile */
    double lag_one;         /**< Lag one gradient autocorrelation of
                               terrain slope */
    double root[MAX_LAYERS]; /**< percent of roots in each soil layer
                                (fraction) */
    double roughness[MONTHS_PER_YEAR]; /**< climatological vegetation
                                          roughness length (m) */
    double sigma_slope;     /**< Std. deviation of terrain slope for each
                               vegetation class */
    int veg_class;          /**< vegetation class id number */
    size_t vegetat_type_num; /**< number of vegetation types in the grid
                                cell */
    double Wdmax[MONTHS_PER_YEAR]; /**< climatological maximum dew holding
                                      capacity (mm) */
    double *zone_depth;     /**< depth of root zone */
    double *zone_fract;     /**< fraction of roots within root zone */
} veg_con_struct;

/******************************************************************************
 * @brief   This structure stores parameters for individual vegetation types.
 *****************************************************************************/
typedef struct {
    double albedo[MONTHS_PER_YEAR];  /**< vegetation albedo (added for full
                                        energy) (fraction) */
    double displacement[MONTHS_PER_YEAR]; /**< vegetation displacement
                                             height (m) */
    double emissivity[MONTHS_PER_YEAR]; /**< vegetation emissivity (fraction) */
    double fcanopy[MONTHS_PER_YEAR];  /**< fractional area covered by plant
                                         canopy (fraction) */
    double LAI[MONTHS_PER_YEAR];  /**< leaf area index */
    size_t NVegLibTypes;   /**< number of vegetation classes defined in
                              library */
    bool overstory;        /**< TRUE = overstory present, important for snow
                              accumulation in canopy */
    double rad_atten;      /**< radiation attenuation due to canopy,
                              default = 0.5 (N/A) */
    double rarc;           /**< architectural resistance (s/m) */
    double rmin;           /**< minimum stomatal resistance (s/m) */
    double roughness[MONTHS_PER_YEAR];  /**< vegetation roughness length (m) */
    double trunk_ratio;    /**< ratio of trunk height to tree height,
                              default = 0.2 (fraction) */
    double Wdmax[MONTHS_PER_YEAR];  /**< maximum dew holding capacity (mm) */
    double wind_atten;     /**< wind attenuation through canopy,
                              default = 0.5 (N/A) */
    double wind_h;         /**< height at which wind is measured (m) */
    double RGL;            /**< Value of solar radiation below which there
                              will be no transpiration (ranges from
                              ~30 W/m^2 for trees to ~100 W/m^2 for crops) */
    unsigned short int veg_class; /**< vegetation class reference number */
    // Carbon terms
    char Ctype;            /**< Photosynthetic pathway; 0 = C3; 1 = C4 */
    double CO2Specificity; /**< CO2 specificity at 25 deg C (mol(CO2)/m2s)
                              (C4 plants) */
    double LightUseEff;    /**< Light-use efficiency (mol(CO2)/mol(photons)) */
    double MaxCarboxRate;  /**< maximum carboxlyation rate at 25 deg C
                              (mol(CO2)/m2s) */
    double MaxETransport;  /**< maximum electron transport rate at 25 deg C
                              (mol(CO2)/m2s) (C3 plants) */
    double NPPfactor_sat;  /**< photosynthesis multiplier (fraction of
                              maximum) when top soil layer is saturated */
    bool NscaleFlag;       /**< TRUE = nitrogen-scaling factors are
                              applicable to this veg class */
    double Wnpp_inhib;     /**< moisture level (fraction of maximum moisture)
                              above which photosynthesis experiencing
                              saturation inhibition, i.e. too wet for optimal
                              photosynthesis; only applies to top soil layer */
} veg_lib_struct;

/******************************************************************************
 * @brief   This structure stores vegetation parameter forcing data for each
 * model time step for a single veg tile.  Each array stores the values for the
 * SNOW_STEPs during the current model step and the value for the entire model
 * step.  The latter is referred to by array[NR].  Looping over the SNOW_STEPs
 * is done by for (i = 0; i < NF; i++)
 *****************************************************************************/
typedef struct {
    double *albedo;       /**< vegetation albedo (fraction) */
    double *displacement; /**< vegetation displacement height (m) */
    double *fcanopy;      /**< fractional area covered by plant canopy
                             (fraction) */
    double *LAI;          /**< leaf area index (m2/m2) */
    double *roughness;    /**< vegetation roughness length (m) */
} veg_hist_struct;

/******************************************************************************
 * @brief   This structure stores the forcing data for each model
 * time step for a single grid cell.  Each array stores the values for the
 * SNOW_STEPs during the current model step and the value for the entire model
 * step.  The latter is referred to by array[NR].  Looping over the SNOW_STEPs
 * is done by for (i = 0; i < NF; i++)
 *****************************************************************************/
typedef struct {
    double *air_temp; /**< air temperature (C) */
    double *Catm;    /**< atmospheric CO2 mixing ratio (mol CO2/ mol air) */
    double *channel_in; /**< incoming channel inflow for time step (mm) */
    double *coszen;  /**< cosine of the solar zenith angle */
    double *density; /**< atmospheric density (kg/m^3) */
    double *fdir;    /**< fraction of incoming shortwave that is direct (fraction) */
    double *longwave; /**< incoming longwave radiation (W/m^2) (net incoming
                         longwave for water balance model) */
    double out_prec;  /**< Total precipitation for time step - accounts
                         for corrected precipitation totals */
    double out_rain;  /**< Rainfall for time step (mm) */
    double out_snow;  /**< Snowfall for time step (mm) */
    double *par;     /**< incoming photosynthetically active radiation () */
    double *prec;    /**< average precipitation in grid cell (mm) */
    double *pressure; /**< atmospheric pressure (kPa) */
    double *shortwave; /**< incoming shortwave radiation (W/m^2) */
    bool *snowflag;    /**< TRUE if there is snowfall in any of the snow
                          bands during the timestep, FALSE otherwise*/
    double *vp;      /**< atmospheric vapor pressure (kPa) */
    double *vpd;     /**< atmospheric vapor pressure deficit (kPa) */
    double *wind;    /**< wind speed (m/s) */
} force_data_struct;

/******************************************************************************
 * @brief   This structure stores information about the time and date of the
 *          current time step.
 *****************************************************************************/
typedef struct {
    unsigned short int day;         /**< current day */
    unsigned short int day_in_year; /**< julian day in year */
    unsigned short int month;       /**< current month */
    int year;                       /**< current year */
    unsigned int dayseconds;        /**< seconds since midnight */
} dmy_struct;                       /**< array of length nrec created */

/******************************************************************************
 * @brief   This structure stores all soil variables for each layer in the
 *          soil column.
 *****************************************************************************/
typedef struct {
    // State variables
    double Cs;              /**< average volumetric heat capacity of the
                               current layer (J/m^3/K) */
    double T;               /**< temperature of the unfrozen sublayer (C) */
    double ice[MAX_FROST_AREAS]; /**< ice content of the frozen sublayer (mm) */
    double kappa;           /**< average thermal conductivity of the current
                               layer (W/m/K) */
    double moist;           /**< moisture content of the unfrozen sublayer
                               (mm) */
    double phi;             /**< moisture diffusion parameter */
    double zwt;             /**< water table position relative to soil surface within the layer (cm) */
    // Fluxes
    double esoil;           /**< soil evaporation from soil layer (mm) */
    double evap;            /**< evapotranspiration from soil layer (mm) */
    double transp;          /**< transpiration from soil layer (mm) */
} layer_data_struct;

/******************************************************************************
 * @brief   This structure stores soil variables for the complete soil column
 *          for each grid cell.
 *****************************************************************************/
typedef struct {
    // State variables
    double aero_resist[2];             /**< The (stability-corrected) aerodynamic
                                          resistance (s/m) that was actually used
                                          in flux calculations.
                                          [0] = surface (bare soil, non-overstory veg, or snow pack)
                                          [1] = overstory */
    double asat;                       /**< saturated area fraction */
    double CLitter;                    /**< carbon storage in litter pool [gC/m2] */
    double CInter;                     /**< carbon storage in intermediate pool [gC/m2] */
    double CSlow;                      /**< carbon storage in slow pool [gC/m2] */
    layer_data_struct layer[MAX_LAYERS]; /**< structure containing soil variables
                                            for each layer (see above; including both
                                            state and flux variables) */
    double rootmoist;                  /**< total of layer.moist over all layers
                                          in the root zone (mm) */
    double wetness;                    /**< average of
                                          (layer.moist - Wpwp)/(porosity*depth - Wpwp)
                                          over all layers (fraction) */
    double zwt;                        /**< average water table position [cm] - using lowest unsaturated layer */
    double zwt_lumped;                 /**< average water table position [cm] - lumping all layers' moisture together */

    // Fluxes
    double pot_evap;                   /**< potential evaporation (mm) */
    double baseflow;                   /**< baseflow from current cell (mm/TS) */
    double runoff;                     /**< runoff from current cell (mm/TS) */
    double inflow;                     /**< moisture that reaches the top of
                                          the soil column (mm) */
    double RhLitter;                   /**< soil respiration from litter pool [gC/m2] */
    double RhLitter2Atm;               /**< soil respiration from litter pool [gC/m2] that goes to atmosphere */
    double RhInter;                    /**< soil respiration from intermediate pool [gC/m2] */
    double RhSlow;                     /**< soil respiration from slow pool [gC/m2] */
    double RhTot;                      /**< total soil respiration over all pools [gC/m2] (=RhLitter2Atm+RhInter+RhSlow) */
} cell_data_struct;

/******************************************************************************
 * @brief   This structure stores energy balance components, and variables used
 *          to solve the thermal fluxes through the soil column.
 *****************************************************************************/
typedef struct {
    // State variables
    double AlbedoLake;           /**< albedo of lake surface (fract) */
    double AlbedoOver;           /**< albedo of intercepted snow (fract) */
    double AlbedoUnder;          /**< surface albedo (fraction) */
    double Cs[2];                /**< heat capacity for top two layers (J/m^3/K) */
    double Cs_node[MAX_NODES];   /**< heat capacity of the soil thermal nodes (J/m^3/K) */
    double fdepth[MAX_FRONTS];   /**< all simulated freezing front depths */
    bool frozen;                   /**< TRUE = frozen soil present */
    double ice[MAX_NODES];       /**< thermal node ice content */
    double kappa[2];             /**< soil thermal conductivity for top two layers (W/m/K) */
    double kappa_node[MAX_NODES]; /**< thermal conductivity of the soil thermal nodes (W/m/K) */
    double moist[MAX_NODES];     /**< thermal node moisture content */
    size_t Nfrost;               /**< number of simulated freezing fronts */
    size_t Nthaw;                /**< number of simulated thawing fronts */
    double T[MAX_NODES];         /**< thermal node temperatures (C) */
    bool T_fbflag[MAX_NODES];      /**< flag indicating if previous step's temperature was used */
    unsigned int T_fbcount[MAX_NODES]; /**< running total number of times that previous step's temperature was used */
    int T1_index;                   /**< soil node at the bottom of the top layer */
    double Tcanopy;              /**< temperature of the canopy air */
    bool Tcanopy_fbflag;           /**< flag indicating if previous step's temperature was used */
    unsigned int Tcanopy_fbcount;    /**< running total number of times that previous step's temperature was used */
    double tdepth[MAX_FRONTS];   /**< all simulated thawing front depths */
    double Tfoliage;             /**< temperature of the overstory vegetation */
    bool Tfoliage_fbflag;            /**< flag indicating if previous step's temperature was used */
    unsigned int Tfoliage_fbcount;   /**< running total number of times that previous step's temperature was used */
    double Tsurf;                /**< temperature of the understory */
    bool Tsurf_fbflag;           /**< flag indicating if previous step's temperature was used */
    unsigned int Tsurf_fbcount;      /**< running total number of times that previous step's temperature was used */
    double unfrozen;             /**< frozen layer water content that is unfrozen */
    // Fluxes
    double advected_sensible;    /**< net sensible heat flux advected to snowpack (Wm-2) */
    double advection;            /**< advective flux (Wm-2) */
    double AtmosError;
    double AtmosLatent;          /**< latent heat exchange with atmosphere */
    double AtmosLatentSub;       /**< latent sub heat exchange with atmosphere */
    double AtmosSensible;        /**< sensible heat exchange with atmosphere */
    double canopy_advection;     /**< advection heat flux from the canopy (W/m^2) */
    double canopy_latent;        /**< latent heat flux from the canopy (W/m^2) */
    double canopy_latent_sub;    /**< latent heat flux of sublimation from the canopy (W/m^2) */
    double canopy_refreeze;      /**< energy used to refreeze/melt canopy intercepted snow (W/m^2) */
    double canopy_sensible;      /**< sensible heat flux from canopy interception (W/m^2) */
    double deltaCC;              /**< change in snow heat storage (Wm-2) */
    double deltaH;               /**< change in soil heat storage (Wm-2) */
    double error;                /**< energy balance error (W/m^2) */
    double fusion;               /**< energy used to freeze/thaw soil water */
    double grnd_flux;            /**< ground heat flux (Wm-2) */
    double latent;               /**< net latent heat flux (Wm-2) */
    double latent_sub;           /**< net latent heat flux from snow (Wm-2) */
    double longwave;             /**< net longwave flux (Wm-2) */
    double LongOverIn;           /**< incoming longwave to overstory */
    double LongUnderIn;          /**< incoming longwave to understory */
    double LongUnderOut;         /**< outgoing longwave from understory */
    double melt_energy;          /**< energy used to reduce snow cover fraction (Wm-2) */
    double NetLongAtmos;         /**< net longwave radiation to the atmosphere (W/m^2) */
    double NetLongOver;          /**< net longwave radiation from the overstory (W/m^2) */
    double NetLongUnder;         /**< net longwave radiation from the understory (W/m^2) */
    double NetShortAtmos;        /**< net shortwave to the atmosphere */
    double NetShortGrnd;         /**< net shortwave penetrating snowpack */
    double NetShortOver;         /**< net shortwave radiation from the overstory (W/m^2) */
    double NetShortUnder;        /**< net shortwave radiation from the understory (W/m^2) */
    double out_long_canopy;      /**< outgoing longwave to canopy */
    double out_long_surface;     /**< outgoing longwave to surface */
    double refreeze_energy;      /**< energy used to refreeze the snowpack (Wm-2) */
    double sensible;             /**< net sensible heat flux (Wm-2) */
    double shortwave;            /**< net shortwave radiation (Wm-2) */
    double ShortOverIn;          /**< incoming shortwave to overstory */
    double ShortUnderIn;         /**< incoming shortwave to understory */
    double snow_flux;            /**< thermal flux through the snow pack (Wm-2) */
} energy_bal_struct;

/******************************************************************************
 * @brief   This structure stores vegetation variables for each vegetation type
 *          in a grid cell.
 *****************************************************************************/
typedef struct {
    // State variables
    double albedo;              /**< current vegetation albedo (fraction) */
    double displacement;        /**< current vegetation displacement height
                                   (m) */
    double fcanopy;             /**< current fractional area of plant canopy
                                   (fraction) */
    double LAI;                 /**< current leaf area index (m2/m2) */
    double roughness;           /**< current vegetation roughness length
                                   (m) */
    double Wdew;                /**< dew trapped on vegetation (mm) */
    double Wdmax;               /**< current maximum dew holding capacity
                                   (mm) */

    // Fluxes
    double canopyevap;          /**< evaporation from canopy (mm/TS) */
    double throughfall;         /**< water that reaches the ground through
                                   the canopy (mm/TS) */

    // Carbon terms - states
    double AnnualNPP;           /**< running total annual NPP [gC/m2] */
    double AnnualNPPPrev;       /**< total annual NPP from previous year
                                   [gC/m2] */
    double Ci;                  /**< whole-canopy leaf-internal CO2 mixing
                                   ratio (mol CO2/mol air) */
    double *CiLayer;            /**< array of per-layer leaf-internal CO2
                                   mixing ratio (mol CO2/mol air) */
    double NPPfactor;           /**< whole-canopy photosynthesis multiplier
                                   to account for inhibition separate from
                                   stomatal resistance */
    double *NscaleFactor;       /**< array of per-layer nitrogen scaling
                                   factors */
    double rc;                  /**< whole-canopy stomatal resistance (s/m) */
    double *rsLayer;            /**< array of per-layer stomatal resistance
                                   (s/m) */

    // Carbon terms - fluxes
    double aPAR;                /**< whole-canopy absorbed PAR
                                   (mol(photons)/m2 leaf area s) */
    double *aPARLayer;          /**< array of per-layer absorbed PAR
                                   (mol(photons)/m2 leaf area s) */
    double GPP;                 /**< whole-canopy gross assimilation
                                   (photosynthesis) (umol(CO2)/m2s) */
    double Litterfall;          /**< flux of carbon from living biomass to
                                   litter pool [gC/m2] */
    double NPP;                 /**< net primary productivity (= GPP - Raut)
                                   (umol(CO2)/m2s) */
    double Raut;                /**< total plant respiration (= Rmaint
                                 + Rgrowth) (umol(CO2)/m2s) */
    double Rdark;               /**< whole-canopy 'dark' respiration
                                   (umol(CO2)/m2s) */
    double Rgrowth;             /**< growth respiration ( = (GPP-Rmaint)
                                 * FRGrowth/(1+FRGrowth) ) (umol(CO2)/m2s) */
    double Rmaint;              /**< plant maintenance respiration
                                   (= Rdark/FRLeaf) (umol(CO2)/m2s) */
    double Rphoto;              /**< whole-canopy photorespiration
                                   (umol(CO2)/m2s) */
} veg_var_struct;

/******************************************************************************
 * @brief   This structure stores snow pack variables needed to run the snow
 *          model.
 *****************************************************************************/
typedef struct {
    // State variables
    double albedo;          /**< snow surface albedo (fraction) */
    double canopy_albedo;   /**< albedo of the canopy (fract) */
    double coldcontent;     /**< cold content of snow pack */
    double coverage;        /**< fraction of snow band that is covered with snow */
    double density;         /**< snow density (kg/m^3) */
    double depth;           /**< snow depth (m) */
    unsigned int last_snow;     /**< time steps since last snowfall */
    double max_snow_depth;  /**< last maximum snow depth - used to determine coverage
                               fraction during current melt period (m) */
    bool MELTING;           /**< flag indicating that snowpack melted
                               previously */
    double pack_temp;       /**< depth averaged temperature of the snowpack (C) */
    double pack_water;      /**< liquid water content of the snow pack (m) */
    bool snow;              /**< TRUE = snow, FALSE = no snow */
    double snow_canopy;     /**< amount of snow on canopy (m) */
    double store_coverage;  /**< stores coverage fraction covered by new snow (m) */
    bool store_snow;        /**< flag indicating whether or not new accumulation
                               is stored on top of an existing distribution */
    double store_swq;       /**< stores newly accumulated snow over an
                               established snowpack melt distribution (m) */
    double surf_temp;       /**< depth averaged temperature of the snow pack surface layer (C) */
    unsigned int surf_temp_fbcount; /**< running total number of times that previous step's temperature was used */
    bool surf_temp_fbflag;    /**< flag indicating if previous step's temperature was used */
    double surf_water;      /**< liquid water content of the surface layer (m) */
    double swq;             /**< snow water equivalent of the entire pack (m) */
    double snow_distrib_slope; /**< current slope of uniform snow distribution (m/fract) */
    double tmp_int_storage; /**< temporary canopy storage, used in snow_canopy */
    // Fluxes
    double blowing_flux;    /**< depth of sublimation from blowing snow (m) */
    double canopy_vapor_flux; /**< depth of water evaporation, sublimation, or
                                 condensation from intercepted snow (m) */
    double mass_error;      /**< snow mass balance error */
    double melt;            /**< snowpack melt (mm) */
    double Qnet;            /**< Residual of energy balance at snowpack surface */
    double surface_flux;    /**< depth of sublimation from blowing snow (m) */
    double transport;       /**< flux of snow (potentially) transported from veg type */
    double vapor_flux;      /**< depth of water evaporation, sublimation, or
                               condensation from snow pack (m) */
} snow_data_struct;

/******************************************************************************
 * @brief   This structures stores variables averaged over a grid cell
 *****************************************************************************/
typedef struct {
    // Grid cell averaged variables
    double avg_albedo;            /**< Average albedo over a grid cell */
} gridcell_avg_struct;

/******************************************************************************
 * @brief   This structure stores the lake/wetland parameters for a grid cell
 *****************************************************************************/
typedef struct {
    // Lake basin dimensions
    size_t numnod;                /**< Maximum number of lake nodes for this grid cell */
    double z[MAX_LAKE_NODES + 1]; /**< Elevation of each lake node (when lake storage is at maximum), relative to lake's deepest point (m) */
    double basin[MAX_LAKE_NODES + 1]; /**< Area of lake basin at each lake node (when lake storage is at maximum) (m^2) */
    double Cl[MAX_LAKE_NODES + 1]; /**< Fractional coverage of lake basin at each node (when lake storage is at maximum) (fraction of grid cell area) */
    double b;                     /**< Exponent in default lake depth-area profile (y=Ax^b) */
    double maxdepth;              /**< Maximum allowable depth of liquid portion of lake (m) */
    double mindepth;              /**< Minimum allowable depth of liquid portion of lake (m) */
    double maxvolume;             /**< Lake volume when lake depth is at maximum (m^3) */
    double minvolume;             /**< Lake volume when lake depth is at minimum (m^3) */
    // Hydrological properties
    double bpercent;              /**< Fraction of wetland baseflow (subsurface runoff) that flows into lake */
    double rpercent;              /**< Fraction of wetland surface runoff that flows into lake */
    double wfrac;                 /**< Width of lake outlet, expressed as fraction of lake perimeter */
    // Initial conditions
    double depth_in;              /**< Initial lake depth (distance from surface to deepest point) (m) */
    int lake_idx;                 /**< index number of the lake/wetland veg tile */
} lake_con_struct;

/******************************************************************************
 * @brief   This structure stores the lake/wetland variables for a grid cell
 *****************************************************************************/
typedef struct {
    // Current lake dimensions and liquid water state variables
    unsigned short int activenod;     /**< Number of nodes whose corresponding layers currently contain water */
    double dz;                    /**< Vertical thickness of all horizontal water layers below the surface layer (m) */
    double surfdz;                /**< Vertical thickness of surface (top) water layer (m) */
    double ldepth;                /**< Current depth of liquid water in lake (distance from surface to deepest point) (m) */
    double surface[MAX_LAKE_NODES + 1]; /**< Area of horizontal cross-section of liquid water in lake at each node (m^2) */
    double sarea;                 /**< Current surface area of ice+liquid water on lake surface (m^2) */
    double sarea_save;            /**< Surface area of ice+liquid water on lake surface (m^2) at beginning of time step */
    double volume;                /**< Current lake water volume, including liquid water equivalent of lake ice (m^3) */
    double volume_save;           /**< Lake water volume, including liquid water equivalent of lake ice (m^3) at beginning of time step */
    double temp[MAX_LAKE_NODES];  /**< Lake water temperature at each node (C) */
    double tempavg;               /**< Average liquid water temperature of entire lake (C) */
    // Current properties (state variables) specific to lake ice/snow
    double areai;                 /**< Area of ice coverage (at beginning of time step) (m^2) */
    double new_ice_area;          /**< Area of ice coverage (at end of time step) (m^2) */
    double ice_water_eq;          /**< Liquid water equivalent volume of lake ice (m^3) */
    double hice;                  /**< Height of lake ice at thickest point (m) */
    double tempi;                 /**< Lake ice temperature (C) */
    double swe;                   /**< Water equivalence of lake snow cover - end of step (m^3) */
    double swe_save;              /**< Water equivalence of lake snow cover - beginning of step (m^3) */
    double surf_temp;             /**< Temperature of surface snow layer (C) */
    double pack_temp;             /**< Temperature of pack snow layer (C) */
    double coldcontent;           /**< cold content of snow pack */
    double surf_water;            /**< Water content of surface snow layer (m^3) */
    double pack_water;            /**< Water content of pack snow layer (m^3) */
    double SAlbedo;               /**< Albedo of lake snow (fraction) */
    double sdepth;                /**< Depth of snow on top of ice (m^3) */
    // Other current lake properties (derived from state variables and forcings)
    double aero_resist;           /**< Aerodynamic resistance (s/m) after stability correction */
    double density[MAX_LAKE_NODES]; /**< Lake water density profile (kg/m^3) */
    // Moisture fluxes
    double baseflow_in;           /**< Volume of baseflow into lake from the rest of the grid cell (m3) */
    double baseflow_out;          /**< Volume of baseflow out of lake to channel network (m3) */
    double channel_in;            /**< Volume of channel inflow into lake (m3) */
    double evapw;                 /**< Volume of evaporative flux from lake (and ice/snow) surface (m3) */
    double ice_throughfall;       /**< Volume of precipitation reaching lake water surface, i.e. total precip minus accumulation of snow on ice (m3) */
    double prec;                  /**< Volume of precipitation falling on lake (and ice/snow) surface (m3) */
    double recharge;              /**< Volume of recharge from lake to wetland (m3) */
    double runoff_in;             /**< Volume of surface runoff into lake from the rest of the grid cell (m3) */
    double runoff_out;            /**< Volume of surface runoff out of lake to channel network (m3) */
    double snowmlt;               /**< Volume of moisture released by melting of lake snow (m3) */
    double vapor_flux;            /**< Volume of moisture sublimated from lake snow (m3) */
    // Structures compatible with other land cover types
    // Some of this information is currently redundant with other variables in the lake_var structure
    snow_data_struct snow;        /**< Snow pack on top of lake ice */
    energy_bal_struct energy;     /**< Energy fluxes and soil temperatures */
    cell_data_struct soil;        /**< Soil column below lake */
} lake_var_struct;

/******************************************************************************
 * @brief   This structure stores all variables needed to solve, or save
 *          solututions for all versions of this model.
 *****************************************************************************/
typedef struct {
    cell_data_struct **cell;      /**< Stores soil layer variables */
    energy_bal_struct **energy;   /**< Stores energy balance variables */
    lake_var_struct lake_var;     /**< Stores lake/wetland variables */
    snow_data_struct **snow;      /**< Stores snow variables */
    veg_var_struct **veg_var;     /**< Stores vegetation variables */
    gridcell_avg_struct gridcell_avg;   /**< Stores gridcell average variables */
} all_vars_struct;

#endif
