\name{RVineLogLik}        
\alias{RVineLogLik}                

\title{Log-likelihood of an R-vine copula model}

\description{
This function calculates the log-likelihood of a d-dimensional R-vine copula model for a given copula data set.
}

\usage{
RVineLogLik(data, RVM, par=RVM$par, par2=RVM$par2, separate=FALSE)
}

\arguments{
  \item{data}{An N x d data matrix (with uniform margins).}
  \item{RVM}{An \code{\link{RVineMatrix}} object including the structure and the pair-copula families and parameters.}
  \item{par}{A d x d matrix with the pair-copula parameters (optional; default: \code{par = RVM$par}).}
  \item{par2}{A d x d matrix with the second parameters of pair-copula families with two parameters (optional; default: \code{par2 = RVM$par2}).}
  \item{separate}{Logical; whether log-likelihoods are returned pointwisely (default: \code{separate = FALSE}).}
}

\value{
  \item{loglik}{The calculated log-likelihood value of the R-vine copula model.}
  \item{V}{The stored transformations (h-functions and log-likelihoods of each pair-copula) which may be used for posterior updates
    (three matrices: \code{direct}, \code{indirect} and \code{value}).}
}

\details{
For observations \eqn{\boldsymbol{u}=(\boldsymbol{u}_1^\prime,...,\boldsymbol{u}_N^\prime)^\prime}
the log-likelihood of a \eqn{d}-dimensional R-vine copula with \eqn{d-1} trees and corresponding edge sets \eqn{E_1,...,E_{d-1}} is given by
\deqn{
\texttt{loglik}:=l_{RVine}\left(\boldsymbol{\theta}|\boldsymbol{u}\right)
}
\deqn{
=\sum_{i=1}^N \sum_{\ell=1}^{d-1} \sum_{e\in E_\ell}
\ln\left[c_{j(e),k(e)|D(e)}\left(F(u_{i,j(e)}|\boldsymbol{u}_{i,D(e)}),F(u_{i,k(e)}|\boldsymbol{u}_{i,D(e)})|\boldsymbol{\theta}_{j(e),k(e)|D(e)}\right)\right],
}
where \eqn{\boldsymbol{u}_i=(u_{i,1},...,u_{i,d})^\prime\in[0,1]^d,\ i=1,...,N}.
Further \eqn{c_{j(e),k(e)|D(e)}} denotes a bivariate copula density associated to an edge \eqn{e} and with parameter(s) \eqn{\boldsymbol{\theta}_{j(e),k(e)|D(e)}}.
Conditional distribution functions such as \eqn{F(u_{i,j(e)}|\boldsymbol{u}_{i,D(e)})} are obtained recursively using the relationship
\deqn{
h(u|\boldsymbol{v},\boldsymbol{\theta}) := F(u|\boldsymbol{v}) =
\frac{\partial C_{uv_j|\boldsymbol{v}_{-j}}(F(u|\boldsymbol{v}_{-j}),F(v_j|\boldsymbol{v}_{-j}))}{\partial F(v_j|\boldsymbol{v}_{-j})},
}
where \eqn{C_{uv_j|\boldsymbol{v}_{-j}}} is a bivariate copula distribution function with parameter(s) \eqn{\boldsymbol{\theta}}{\theta}
and \eqn{\boldsymbol{v}_{-j}} denotes a vector with the \eqn{j}-th component \eqn{v_j} removed.
The notation of h-functions is introduced for convenience. For more details see Dissmann et al. (2013).
}

\references{
Dissmann, J. F., E. C. Brechmann, C. Czado, and D. Kurowicka (2013).
Selecting and estimating regular vine copulae and application to financial returns.
Computational Statistics & Data Analysis, 59 (1), 52-69.
}

\author{Ulf Schepsmeier, Jeffrey Dissmann, Jakob Stoeber}

\seealso{\code{\link{BiCopHfunc}}, \code{\link{RVineMatrix}}, \code{\link{RVineMLE}}, \code{\link{RVineAIC}}, \code{\link{RVineBIC}}}

\examples{
# define 5-dimensional R-vine tree structure matrix
Matrix = c(5,2,3,1,4,0,2,3,4,1,0,0,3,4,1,0,0,0,4,1,0,0,0,0,1)
Matrix = matrix(Matrix,5,5)

# define R-vine pair-copula family matrix
family = c(0,1,3,4,4,0,0,3,4,1,0,0,0,4,1,0,0,0,0,3,0,0,0,0,0)
family = matrix(family,5,5)

# define R-vine pair-copula parameter matrix
par = c(0,0.2,0.9,1.5,3.9,0,0,1.1,1.6,0.9,0,0,0,1.9,0.5,
        0,0,0,0,4.8,0,0,0,0,0)
par = matrix(par,5,5)

# define second R-vine pair-copula parameter matrix
par2 = matrix(0,5,5)

# define RVineMatrix object
RVM = RVineMatrix(Matrix=Matrix,family=family,par=par,par2=par2,
                  names=c("V1","V2","V3","V4","V5"))

# simulate a sample of size 300 from the R-vine copula model
simdata = RVineSim(300,RVM)

# compute the log-likelihood
ll = RVineLogLik(simdata,RVM,separate=FALSE)
ll$loglik

# compute the pointwise log-likelihoods
ll = RVineLogLik(simdata,RVM,separate=TRUE)
ll$loglik
}

