% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weightit2user.R
\name{method_user}
\alias{method_user}
\title{User-Defined Functions for Estimating Weights}
\description{
This page explains the details of estimating weights using a user-defined function. The function must take in arguments that are passed to it by \code{\link[=weightit]{weightit()}} or \code{\link[=weightitMSM]{weightitMSM()}} and return a vector of weights or a list containing the weights.

To supply a user-defined function, the function object should be entered directly to \code{method}; for example, for a function \code{fun}, \code{method = fun}.
\subsection{Point Treatments}{

The following arguments are automatically passed to the user-defined function, which should have named parameters corresponding to them:
\itemize{
\item{\code{treat}: a vector of treatment status for each unit. This comes directly from the left hand side of the formula passed to \code{weightit()} and so will have it's type (e.g., numeric, factor, etc.), which may need to be converted.}
\item{\code{covs}: a data frame of covariate values for each unit. This comes directly from the right hand side of the formula passed to \code{weightit()}. The covariates are processed so that all columns are numeric; all factor variables are split into dummies and all interactions are evaluated. All levels of factor variables are given dummies, so the matrix of the covariates is not full rank. Users can use \code{\link[=make_full_rank]{make_full_rank()}}, which accepts a numeric matrix or data frame and removes columns to make it full rank, if a full rank covariate matrix is desired.}
\item{\code{s.weights}: a numeric vector of sampling weights, one for each unit.}
\item{\code{ps}: a numeric vector of propensity scores.}
\item{\code{subset}: a logical vector the same length as \code{treat} that is \code{TRUE} for units to be included in the estimation and \code{FALSE} otherwise. This is used to subset the input objects when \code{exact} is used. \code{treat}, \code{covs}, \code{s.weights}, and \code{ps}, if supplied, will already have been subsetted by \code{subset}.}
\item{\code{estimand}: a character vector of length 1 containing the desired estimand. The characters will have been converted to uppercase. If "ATC" was supplied to estimand, \code{weightit()} sets \code{focal} to the control level (usually 0 or the lowest level of \code{treat}) and sets \code{estimand} to "ATT".}
\item{\code{focal}: a character vector of length 1 containing the focal level of the treatment when the estimand is the ATT (or the ATC as detailed above). \code{weightit()} ensures the value of focal is a level of \code{treat}.}
\item{\code{stabilize}: a logical vector of length 1. It is not processed by \code{weightit()} before it reaches the fitting function.}
\item{\code{moments}: a numeric vector of length 1. It is not processed by \code{weightit()} before it reaches the fitting function except that \code{as.integer()} is applied to it. This is used in other methods to determine whether polynomials of the entered covariates are to be used in the weight estimation.}
\item{\code{int}: a logical vector of length 1. It is not processed by \code{weightit()} before it reaches the fitting function. This is used in other methods to determine whether interactions of the entered covariates are to be used in the weight estimation.}
}
None of these parameters are required to be in the fitting function. These are simply those that are automatically available.

In addition, any additional arguments supplied to \code{weightit()} will be passed on to the fitting function. \code{weightit()} ensures the arguments correspond to the parameters of the fitting function and throws an error if an incorrectly named argument is supplied and the fitting function doesn't include \verb{\\dots} as a parameter.

The fitting function must output either a numeric vector of weights or a list (or list-like object) with an entry named wither "w" or "weights". If a list, the list can contain other named entries, but only entries named "w", "weights", "ps", and "fit.obj" will be processed. "ps" is a vector of propensity scores and "fit.obj" should be an object used in the fitting process that a user may want to examine and that is included in the \code{weightit} output object as "obj" when \code{include.obj = TRUE}. The "ps" and "fit.obj" components are optional, but "weights" or "w" is required.
}

\subsection{Longitudinal Treatments}{

Longitudinal treatments can be handled either by running the fitting function for point treatments for each time point and multiplying the resulting weights together or by running a method that accommodates multiple time points and outputs a single set of weights. For the former, \code{weightitMSM()} can be used with the user-defined function just as it is with \code{weightit()}. The latter method is not yet accommodated by \code{weightitMSM()}, but will be someday, maybe.
}
}
\examples{

library("cobalt")
data("lalonde", package = "cobalt")

#A user-defined version of method = "ps"
my.ps <- function(treat, covs, estimand, focal = NULL) {
  covs <- make_full_rank(covs)
  d <- data.frame(treat, covs)
  f <- formula(d)
  ps <- glm(f, data = d, family = "binomial")$fitted
  w <- get_w_from_ps(ps, treat = treat, estimand = estimand,
                     focal = focal)

  list(w = w, ps = ps)
}

#Balancing covariates between treatment groups (binary)
(W1 <- weightit(treat ~ age + educ + married +
                  nodegree + re74, data = lalonde,
                method = my.ps, estimand = "ATT"))
summary(W1)
bal.tab(W1)

data("msmdata")
(W2 <- weightitMSM(list(A_1 ~ X1_0 + X2_0,
                        A_2 ~ X1_1 + X2_1 +
                          A_1 + X1_0 + X2_0,
                        A_3 ~ X1_2 + X2_2 +
                          A_2 + X1_1 + X2_1 +
                          A_1 + X1_0 + X2_0),
                   data = msmdata,
                   method = my.ps))

summary(W2)
bal.tab(W2)

# Kernel balancing using the `kbal` package, available
# using `pak::pak_install("chadhazlett/KBAL")`.
# Only the ATT and ATC are available.

\dontrun{
  kbal.fun <- function(treat, covs, estimand, focal, verbose, ...) {
    args <- list(...)

    if (!estimand \%in\% c("ATT", "ATC"))
      stop('`estimand` must be "ATT" or "ATC".', call. = FALSE)

    treat <- as.numeric(treat == focal)

    args <- args[names(args) \%in\% names(formals(kbal::kbal))]
    args$allx <- covs
    args$treatment <- treat
    args$printprogress <- verbose

    cat_cols <- apply(covs, 2, function(x) length(unique(x)) <= 2)

    if (all(cat_cols)) {
      args$cat_data <- TRUE
      args$mixed_data <- FALSE
      args$scale_data <- FALSE
      args$linkernel <- FALSE
      args$drop_MC <- FALSE
    }
    else if (any(cat_cols)) {
      args$cat_data <- FALSE
      args$mixed_data <- TRUE
      args$cat_columns <- colnames(covs)[cat_cols]
      args$allx[,!cat_cols] <- scale(args$allx[,!cat_cols])
      args$cont_scale <- 1
    }
    else {
      args$cat_data <- FALSE
      args$mixed_data <- FALSE
    }

    k.out <- do.call(kbal::kbal, args)
    w <- k.out$w

    list(w = w, fit.obj = k.out)
  }

  (Wk <- weightit(treat ~ age + educ + married +
                    nodegree + re74, data = lalonde,
                  method = kbal.fun, estimand = "ATT",
                  include.obj = TRUE))
  summary(Wk)
  bal.tab(Wk, stats = c("m", "ks"))
}

}
\seealso{
\code{\link[=weightit]{weightit()}}, \code{\link[=weightitMSM]{weightitMSM()}}
}
