% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cumulative_cutoff.R
\name{cumulative_cutoff}
\alias{cumulative_cutoff}
\title{Cumulative access based on a travel cost cutoff}
\usage{
cumulative_cutoff(
  travel_matrix,
  land_use_data,
  opportunity,
  travel_cost,
  cutoff,
  group_by = character(0),
  active = TRUE,
  fill_missing_ids = TRUE
)
}
\arguments{
\item{travel_matrix}{A data frame. The travel matrix describing the costs
(i.e. travel time, distance, monetary cost, etc.) between the origins and
destinations in the study area. Must contain the columns \code{from_id}, \code{to_id}
and any others specified in \code{travel_cost}.}

\item{land_use_data}{A data frame. The distribution of opportunities within
the study area cells. Must contain the columns \code{id} and any others
specified in \code{opportunity}.}

\item{opportunity}{A string. The name of the column in \code{land_use_data}
with the number of opportunities/resources/services to be considered when
calculating accessibility levels.}

\item{travel_cost}{A \code{character} vector. The name of the columns in
\code{travel_matrix} with the travel costs between origins and destinations to
be considered in the calculation.}

\item{cutoff}{Either a \code{numeric} vector or a list of \code{numeric} vectors, one
for each cost specified in \code{travel_cost}. The travel cost cutoffs to
consider when calculating accessibility levels. If a list, the function
finds every single possible cutoff combination and use them to calculate
accessibility (e.g. if one specifies that travel time cutoffs should be 30
and 60 minutes and that monetary cost cutoffs should be 5 and 10 dollars,
the output includes accessibility estimates limited at 30 min & 5 dollars,
30 min & 10 dollars, 60 min & 5 dollars and 60 min & 10 dollars). In these
cases, cost constraints are considered simultaneously - i.e. only trips
that take 30 minutes or less AND 5 dollars or less to be completed, for
example, are included in the accessibility output. The cutoff parameter is
not included in the final output if the input includes only a single cutoff
for a single travel cost.}

\item{group_by}{A \code{character} vector. When not \code{character(0)} (the default),
indicates the \code{travel_matrix} columns that should be used to group the
accessibility estimates by. For example, if \code{travel_matrix} includes a
departure time column, that specifies the departure time of each entry in
the data frame, passing \code{"departure_time"} to this parameter results in
accessibility estimates grouped by origin and by departure time.}

\item{active}{A logical. Whether to calculate active accessibility (the
of opportunities that can be reached from a given origin, the default) or
passive accessibility (by how many people each destination can be reached).}

\item{fill_missing_ids}{A \code{logical}. Calculating cumulative accessibility may
result in missing ids if the they cannot reach any of the destinations
within the specified travel cost cutoff. For example, using a travel time
cutoff of 20 minutes, when estimating the accessibility of origin \code{A} that
can only reach destinations with more than 40 minutes results in id \code{A}
not being included in the output. When \code{TRUE} (the default), the function
identifies which origins would be left out and fills their respective
accessibility values with 0, which incurs in a performance penalty.}
}
\value{
A data frame containing the accessibility estimates for each
origin/destination (depending if \code{active} is \code{TRUE} or \code{FALSE}) in the
travel matrix.
}
\description{
Calculates the number of opportunities accessible under a given specified
travel cost cutoff.

This function is generic over any kind of numeric travel cost,
such as distance, time and money.
}
\examples{
\dontshow{if (identical(tolower(Sys.getenv("NOT_CRAN")), "true")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
data_dir <- system.file("extdata", package = "accessibility")
travel_matrix <- readRDS(file.path(data_dir, "travel_matrix.rds"))
land_use_data <- readRDS(file.path(data_dir, "land_use_data.rds"))

# active accessibility: number of schools accessible from each origin
df <- cumulative_cutoff(
  travel_matrix = travel_matrix,
  land_use_data = land_use_data,
  cutoff = 30,
  opportunity = "schools",
  travel_cost = "travel_time"
)
head(df)

df <- cumulative_cutoff(
  travel_matrix = travel_matrix,
  land_use_data = land_use_data,
  cutoff = c(30, 60),
  opportunity = "schools",
  travel_cost = "travel_time"
)
head(df)

# passive accessibility: number of people that can reach each destination
df <- cumulative_cutoff(
  travel_matrix = travel_matrix,
  land_use_data = land_use_data,
  cutoff = 30,
  opportunity = "population",
  travel_cost = "travel_time",
  active = FALSE
)
head(df)

# using multiple travel costs
pareto_frontier <- readRDS(file.path(data_dir, "pareto_frontier.rds"))

df <- cumulative_cutoff(
  pareto_frontier,
  land_use_data = land_use_data,
  opportunity = "jobs",
  travel_cost = c("travel_time", "monetary_cost"),
  cutoff = list(c(20, 30), c(0, 5, 10))
)
head(df)
\dontshow{\}) # examplesIf}
}
\seealso{
Other cumulative access: 
\code{\link{cumulative_interval}()}
}
\concept{cumulative access}
