% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_accumulation_metrics.R
\name{compute_accumulation_metrics}
\alias{compute_accumulation_metrics}
\title{Compute activity accumulation metrics}
\usage{
compute_accumulation_metrics(
  data,
  col_time = "time",
  col_cat_int = "intensity_category",
  behaviour = c("sed", "pa"),
  dates = NULL,
  valid_wear_time_start = "00:00:00",
  valid_wear_time_end = "23:59:59",
  zoom_from = "00:00:00",
  zoom_to = "23:59:59"
)
}
\arguments{
\item{data}{A dataframe obtained using the \code{\link{prepare_dataset}}, \code{\link{mark_wear_time}}, and then the \code{\link{mark_intensity}} functions.}

\item{col_time}{A character value to indicate the name of the variable containing time data.}

\item{col_cat_int}{A character value indicating the name of the variable where intensity category (SED, LPA, MVPA) is provided.}

\item{behaviour}{A character value indicating whether metrics should be computed for sedentary behaviour or physical activity.}

\item{dates}{A character vector containing the dates to be retained for analysis. The dates must be with the "YYYY-MM-DD" format. Default is NULL.}

\item{valid_wear_time_start}{A character value with the HH:MM:SS format to set the start of the daily period that will be considered for computing metrics.}

\item{valid_wear_time_end}{A character value with the HH:MM:SS format to set the end of the daily period that will be considered for computing metrics.}

\item{zoom_from}{A character value with the HH:MM:SS format to set the start of the daily period to visualize regarding the daily breaks.}

\item{zoom_to}{A character value with the HH:MM:SS format to set the end of the daily period to visualize regarding the daily breaks.}
}
\value{
A list of numeric and graphic objects related to mean daily total breaks, alpha, MBD, UBD and Gini index. The list also contains
the processed datasets that were used to provide these metrics: \code{recap_bouts_by_day} used to compute \code{mean_break}, \code{recap_bouts} used
to compute \code{alpha} and \code{MBD}, \code{summarised_bouts} used to compute \code{UBD}, and \code{summarised_bouts2} used to compute \verb{Gini index}.
}
\description{
This function computes metrics that summarise the pattern of accumulation
of either sedentary behaviour or physical activity (depending on the
configuration of the function) over time:
\itemize{
\item \strong{mean_breaks}: mean daily number of transitions from a sedentary bout to a
physical activity bout (or from a physical activity bout to a sedentary bout);
this actually corresponds to the mean daily total number of sedentary (or physical activity) bouts detected.

\item \strong{alpha}: provides information on the relative proportion of
short and long bouts.  The higher the alpha coefficient, the more the individual tends to
accumulate sedentary (or physical activity) time using relatively short bouts. Alpha is
computed using all the bouts of the days and periods of the day considered for analysis. Alpha
is computed using the following equation provided by Chastin et al. (2010; doi: 10.1016/j.gaitpost.2009.09.002):
\eqn{\alpha = 1 + n \left[\sum_{i = 1}^{n}{ln}\frac{x_{i}}{x_{min}}\right]^{-1}}, with \eqn{n} the total number of bouts,
\eqn{x_{i}} the duration of the bout \eqn{i}, and \eqn{x_{min}} the shortest recorded bout duration.

\item \strong{median bout duration (MBD)}: refers to the median sedentary (or physical activity) bout duration.
MBD is computed using all the bouts of the days and periods of the day considered for analysis.

\item \strong{usual bout duration (UBD)}: refers to the bout duration under/above which 50\% of sedentary
(or physical activity) time is accumulated. UBD is computed using all the bouts of the days and periods
of the day considered for analysis. UBD is determined as described in Belletiere al. (2017; doi:10.1371/journal.pone.0180119)
supplementary file 1. More precisely, UBD is found using non-linear regression with the following model: \eqn{y = \frac{t^n}{t^n + UBD^n}},
with \eqn{t} the bout duration, \eqn{n} a free parameter, and \eqn{y} the fraction of total time accumulated in bouts \eqn{\le} \eqn{t}.

\item \strong{Gini index}: provides information on the equality with which bout durations
contribute to total sedentary (or physical activity) time. A value of 1 reveals perfect
inequality, and a value of 0 reveals perfect equality. Gini index is computed using all
the bouts of the days and periods of the day considered for analysis. Gini index is computed following
the procedure described at the following link: https://www.statology.org/gini-coefficient-excel/. This method
provides similar results as the frequency method implemented in the \code{Gini} function from the \code{DescTools} R package.
}

The appearance of the graphics generated by the function has been inspired by the supplementary file provided by Belletiere al. (2017; doi:10.1371/journal.pone.0180119).
}
\examples{
\donttest{
file <- system.file("extdata", "acc.agd", package = "activAnalyzer")
mydata <- prepare_dataset(data = file)
mydata_with_wear_marks <- mark_wear_time(
    dataset = mydata, 
    TS = "TimeStamp", 
    to_epoch = 60,
    cts  = "vm",
    frame = 90, 
    allowanceFrame = 2, 
    streamFrame = 30
    )
mydata_with_intensity_marks <- mark_intensity(
    data = mydata_with_wear_marks, 
    col_axis = "vm", 
    equation = "Sasaki et al. (2011) [Adults]",
    sed_cutpoint = 200, 
    mpa_cutpoint = 2690, 
    vpa_cutpoint = 6167, 
    age = 32,
    weight = 67,
    sex = "male",
    )
compute_accumulation_metrics(
   data = mydata_with_intensity_marks,
   behaviour = "sed",
   dates = c("2021-04-07", "2021-04-08", "2021-04-09", "2021-04-10", "2021-04-11"),
   valid_wear_time_start = "00:00:00",
   valid_wear_time_end = "23:59:59",
   zoom_from = "00:00:00",
   zoom_to = "23:59:59"
    )
}

}
