% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sample_tmb.R
\name{sample_tmb}
\alias{sample_tmb}
\title{Bayesian inference of a TMB model using the no-U-turn sampler.}
\usage{
sample_tmb(obj, iter = 2000, init, chains = 3, seeds = NULL,
  warmup = floor(iter/2), lower = NULL, upper = NULL, thin = 1,
  parallel = FALSE, cores = NULL, path = NULL, algorithm = "NUTS",
  laplace = FALSE, control = NULL, ...)
}
\arguments{
\item{obj}{A TMB model object.}

\item{iter}{The number of samples to draw.}

\item{init}{A list of lists containing the initial parameter vectors,
one for each chain or a function. It is strongly recommended to
initialize multiple chains from dispersed points. A of NULL signifies
to use the starting values present in the model (i.e., \code{obj$par})
for all chains.}

\item{chains}{The number of chains to run.}

\item{seeds}{A vector of seeds, one for each chain.}

\item{warmup}{The number of warmup iterations.}

\item{lower}{A vector of lower bounds for parameters. Allowed values are
-Inf and numeric.}

\item{upper}{A vector of upper bounds for parameters. Allowed values are
Inf and numeric.}

\item{thin}{The thinning rate to apply to samples. Typically not used
with NUTS.}

\item{parallel}{A boolean for whether to use parallel cores. The package
snowfall is used if TRUE.}

\item{cores}{The number of cores to use for parallel execution.}

\item{path}{The path to the TMB DLL. This is only required if using
parallel, since each core needs to link to the DLL again.}

\item{algorithm}{The algorithm to use. NUTS is the default and
recommended one, but "RWM" for the random walk Metropolis sampler and
"HMC" for the static HMC sampler are available. These last two are
deprecated but may be of use in some situations. These algorithms
require different arguments; see their help files for more
information.}

\item{laplace}{Whether to use the Laplace approximation if some
parameters are declared as random. Default is to turn off this
functionality and integrate across all parameters with MCMC.}

\item{control}{A list to control the sampler. See details for further
use.}

\item{...}{Further arguments to be passed to the algorithm. See help
files for the samplers for further arguments.}
}
\value{
A list containing the samples, and properties of the sampler
  useful for diagnosing behavior and efficiency.
}
\description{
Draw Bayesian posterior samples from a Template Model Builder (TMB)
model using an MCMC algorithm. This function generates posterior samples
from which inference can be made. Adaptation schemes are used so
specification tuning parameters are not necessary, and parallel
execution reduces overall run time.
}
\details{
This function implements algorithm 6 of Hoffman and Gelman (2014),
and loosely follows package \code{rstan}. The step size can be
  adapted or specified manually. The metric (i.e., mass matrix) can be
  unit diagonal, adapted diagonal (default and recommended), or a dense
  matrix specified by the user. Further control of algorithms can be
  specified with the \code{control} argument.  Elements are:
\describe{
\item{adapt_delta}{The target acceptance rate.}
\item{metric}{The mass metric to use. Options are: "unit" for a unit diagonal
  matrix; "diag" to estimate a diagonal matrix during warmup; a matrix
  to be used directly (in untransformed space).}
\item{adapt_engaged}{Whether adaptation of step size and metric is turned on.}
\item{max_treedepth}{Maximum treedepth for the NUTS algorithm.}
\item{stepsize}{The stepsize for the NUTS algorithm. If \code{NULL} it
  will be adapted during warmup.}
}
}
\section{Warning}{

The user is responsible for specifying the model properly (priors,
  starting values, desired parameters fixed, etc.), as well as assessing
  the convergence and validity of the resulting samples (e.g., through
  the \code{coda} package), or with function
  \code{\link{launch_shinytmb}} before making inference. Specifically,
  priors must be specified in the template file for each
  parameter. Unspecified priors will be implicitly uniform.
}

\examples{
## Build a fake TMB object with objective & gradient functions and some
## other flags
f <- function(x, order=0){
  if(order != 1) # negative log density
    -sum(dnorm(x=x, mean=0, sd=1, log=TRUE))
  else x # gradient of negative log density
}
init <- function() rnorm(2)
obj <- list(env=list(DLL='demo', last.par.best=c(x=init()), f=f,
  beSilent=function() NULL))
## Run NUTS for this object
fit <- sample_tmb(obj, iter=1000, chains=3, init=init)
## Check basic diagnostics
mon <- rstan::monitor(fit$samples, print=FALSE)
Rhat <- mon[,"Rhat"]
max(Rhat)
ess <- mon[, 'n_eff']
min(ess)
## Or do it interactively with ShinyStan
\dontrun{
  launch_shinytmb(fit)
  }

}
\seealso{
\code{\link{extract_samples}} to extract samples and
  \code{\link{launch_shinytmb}} to explore the results graphically which
  is a wrapper for the \code{\link[shinystan]{launch_shinystan}} function.
}
\author{
Cole Monnahan
}
