% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SimFunctions.R
\name{Sim-Functions}
\alias{Sim-Functions}
\alias{sim}
\title{Merger Simulation With User-Supplied Demand Parameters}
\usage{
sim(prices, demand = c("Linear", "AIDS", "LogLin", "Logit", "CES",
  "LogitNests", "CESNests", "LogitCap"), demand.param, ownerPre, ownerPost,
  nests, capacities, mcDelta = rep(0, length(prices)),
  subset = rep(TRUE, length(prices)), priceOutside, priceStart,
  labels = paste("Prod", 1:length(prices), sep = ""), ...)
}
\arguments{
\item{prices}{A length k vector of product prices.}

\item{demand}{A character string indicating the type of demand system
to be used in the merger simulation. Supported demand systems are
linear (\sQuote{Linear}), log-linear(\sQuote{LogLin}), logit (\sQuote{Logit}), nested logit
(\sQuote{LogitNests}), ces (\sQuote{CES}), nested CES (\sQuote{CESNests}) and capacity
constrained Logit (\sQuote{LogitCap}).}

\item{demand.param}{See Below.}

\item{ownerPre}{EITHER a vector of length k whose values
indicate which firm produced a product pre-merger OR
a k x k matrix of pre-merger ownership shares.}

\item{ownerPost}{EITHER a vector of length k whose values
indicate which firm produced a product after the merger OR
a k x k matrix of post-merger ownership shares.}

\item{nests}{A length k vector identifying the nest that each
product belongs to. Must be supplied when \sQuote{demand} equals \sQuote{CESNests} and
\sQuote{LogitNests}.}

\item{capacities}{A length k vector of product capacities. Must be
supplied when \sQuote{demand} equals \sQuote{LogitCap}.}

\item{mcDelta}{A vector of length k where each element equals the
proportional change in a product's marginal costs due to
the merger. Default is 0, which assumes that the merger does not
affect any products' marginal cost.}

\item{subset}{A vector of length k where each element equals TRUE if
the product indexed by that element should be included in the
post-merger simulation and FALSE if it should be excluded.Default is a
length k vector of TRUE.}

\item{priceOutside}{A length 1 vector indicating the price of the
outside good. This option only applies to the \sQuote{Logit} class and its child classes
Default for \sQuote{Logit},\sQuote{LogitNests}, and \sQuote{LogitCap} is 0,
and for \sQuote{CES} and \sQuote{CesNests} is 1.}

\item{priceStart}{A length k vector of starting values used to solve for
equilibrium price. Default is the \sQuote{prices} vector for all values of
demand except for \sQuote{AIDS}, which is set equal to a vector of 0s.}

\item{labels}{A k-length vector of labels. Default is \dQuote{Prod#}, where
\sQuote{#} is a number between 1 and the length of \sQuote{prices}.}

\item{...}{Additional options to feed to the
optimizer used to solve for equilibrium prices.}
}
\value{
\code{sim} returns an instance of the class specified by the
\sQuote{demand} argument.
}
\description{
Simulates the price effects of a merger between two firms
with user-supplied demand parameters under the
assumption that all firms in the market are playing a
differentiated products Bertrand pricing game.

Let k denote the number of products produced by all firms below.
}
\details{
Using user-supplied demand parameters,
\code{sim} simulates the effects of a merger in a market where
firms are playing a differentiated products pricing game.

If \sQuote{demand} equals \sQuote{Linear}, \sQuote{LogLin}, or
\sQuote{AIDS}, then \sQuote{demand.param} must be a
list containing \sQuote{slopes}, a k x k matrix of slope coefficients, and
\sQuote{intercepts}, a length-k vector of intercepts. Additionally, if
\sQuote{demand} equals \sQuote{AIDS}, \sQuote{demand.param} must contain \sQuote{mktElast}, an
estimate of aggregate market elasticity.  For \sQuote{Linear}
demand models, \code{sim} returns an error if any intercepts are
negative, and for both \sQuote{Linear}, \sQuote{LogLin}, and \sQuote{AIDS} models,  \code{sim}
returns an error if not all diagonal elements of the slopes matrix are
negative.

If \sQuote{demand} equals \sQuote{Logit} or \sQuote{LogitNests}, then
\sQuote{demand.param} must equal a list containing
\itemize{
  \item{alpha}{The price coefficient.}
  \item{meanval}{A length-k vector of mean valuations \sQuote{meanval}. If
    none of the values of \sQuote{meanval} are zero, an outside good is assumed
    to exist.}
}
If demand equals \sQuote{CES} or \sQuote{CESNests}, then
\sQuote{demand.param} must equal a list containing

\itemize{
  \item{gamma}{ The price coefficient,}
  \item{alpha}{The coefficient on the numeraire good. May instead be
    calibrated using \sQuote{shareInside},}
  \item{meanval}{A length-k vector of mean valuations \sQuote{meanval}. If
    none of the values of \sQuote{meanval} are zero, an outside good is assumed
    to exist,}
  \item{shareInside}{ The budget share of all products in the
    market. Default is 1, meaning that all consumer wealth is spent on
    products in the market. May instead be specified using \sQuote{alpha}.}

}
}
\examples{
## Calibration and simulation results from a merger between Budweiser and
## Old Style. Note that the in the following model there is no outside
## good; BUD's mean value has been normalized to zero.

## Source: Epstein/Rubenfeld 2004, pg 80


prodNames <- c("BUD","OLD STYLE","MILLER","MILLER-LITE","OTHER-LITE","OTHER-REG")
ownerPre <-c("BUD","OLD STYLE","MILLER","MILLER","OTHER-LITE","OTHER-REG")
ownerPost <-c("BUD","BUD","MILLER","MILLER","OTHER-LITE","OTHER-REG")
nests <- c("Reg","Reg","Reg","Light","Light","Reg")

price    <- c(.0441,.0328,.0409,.0396,.0387,.0497)

demand.param=list(alpha=-48.0457,
                  meanval=c(0,0.4149233,1.1899885,0.8252482,0.1460183,1.4865730)
)

sim.logit <- sim(price,demand="Logit",demand.param,ownerPre=ownerPre,ownerPost=ownerPost)



print(sim.logit)           # return predicted price change
summary(sim.logit)         # summarize merger simulation

elast(sim.logit,TRUE)      # returns premerger elasticities
elast(sim.logit,FALSE)     # returns postmerger elasticities

diversion(sim.logit,TRUE)  # return premerger diversion ratios
diversion(sim.logit,FALSE) # return postmerger diversion ratios


cmcr(sim.logit)            #calculate compensating marginal cost reduction
upp(sim.logit)             #calculate Upwards Pricing Pressure Index

CV(sim.logit)              #calculate representative agent compensating variation

}
\seealso{
The S4 class documentation for: \code{\linkS4class{Linear}},
\code{\linkS4class{AIDS}}, \code{\linkS4class{LogLin}}, \code{\linkS4class{Logit}},
\code{\linkS4class{LogitNests}}, \code{\linkS4class{CES}}, \code{\linkS4class{CESNests}}
}
\author{
Charles Taragin \email{ctaragin@ftc.gov}
}
